const canvas = document.querySelector('canvas');
const context = canvas.getContext('2d');

canvas.width = 1024;
canvas.height = 576;

context.fillRect(0, 0, canvas.width, canvas.height);

const gravity = 0.7;
const background = new Sprite({
    position: {
        x: 0,
        y: 0
    },
    imageSrc: './backgrounds/Background1.png',
    scale: 1,
    isBackground: true // Flag to identify background
})

const shop = new Sprite({
    position: {
        x: 700,
        y: 170
    },
    imageSrc: './backgrounds/shop_anim.png',
    scale: 2.5,
    framesMax: 6
})

const player = new Fighter({
    position: {
        x: 100,
        y: 451  // Positioned on the floor (494 - 200 + 157 offset)
    },
    velocity: {
        x: 0,
        y: 0
    },
    offset: {
        x: 0,
        y: 0
    },
    images: [
        './farmer-peasant/idle/01.png',
        './farmer-peasant/idle/02.png',
        './farmer-peasant/idle/03.png',
        './farmer-peasant/idle/04.png',
        './farmer-peasant/idle/05.png'
    ],
    framesMax: 5,
    scale: 2.5,
    offset: {
        x: 215,
        y: 157
    },
    sprites: {
        idle: {
            images: [
                './farmer-peasant/idle/01.png',
                './farmer-peasant/idle/02.png',
                './farmer-peasant/idle/03.png',
                './farmer-peasant/idle/04.png',
                './farmer-peasant/idle/05.png'
            ],
            framesMax: 5
        },
        run: {
            images: [
                './farmer-peasant/idle/01.png',
                './farmer-peasant/idle/02.png',
                './farmer-peasant/idle/03.png',
                './farmer-peasant/idle/04.png',
                './farmer-peasant/idle/05.png'
            ],
            framesMax: 5
        },
        jump: {
            images: [
                './farmer-peasant/idle/01.png'
            ],
            framesMax: 1
        },
        fall: {
            images: [
                './farmer-peasant/idle/01.png'
            ],
            framesMax: 1
        },
        attack1: {
            images: [
                './farmer-peasant/fight/01.png',
                './farmer-peasant/fight/02.png',
                './farmer-peasant/fight/03.png',
                './farmer-peasant/fight/04.png',
                './farmer-peasant/fight/05.png'
            ],
            framesMax: 5
        },
        takeHit: {
            images: [
                './farmer-peasant/hurt/01.png',
                './farmer-peasant/hurt/02.png',
                './farmer-peasant/hurt/03.png',
                './farmer-peasant/hurt/04.png',
                './farmer-peasant/hurt/05.png'
            ],
            framesMax: 5
        },
        death: {
            images: [
                './farmer-peasant/lose/01.png',
                './farmer-peasant/lose/02.png',
                './farmer-peasant/lose/03.png',
                './farmer-peasant/lose/04.png',
                './farmer-peasant/lose/05.png',
                './farmer-peasant/lose/06.png',
                './farmer-peasant/lose/07.png',
                './farmer-peasant/lose/08.png',
                './farmer-peasant/lose/09.png'
            ],
            framesMax: 9
        }
    },
    attackBox: {
        offset: {
            x: 100,
            y: 50
        },
        width: 160,
        height: 50
    }
});

const enemy = new Fighter({
    position: {
        x: 800,
        y: 464  // Positioned on the floor (494 - 200 + 170 offset)
    },
    velocity: {
        x: 0,
        y: 0
    },
    color: 'blue',
    offset: {
        x: 0,
        y: 0
    },
    images: [
        './desert-rat/idle/left/01.png',
        './desert-rat/idle/left/02.png',
        './desert-rat/idle/left/03.png',
        './desert-rat/idle/left/04.png',
        './desert-rat/idle/left/05.png',
        './desert-rat/idle/left/06.png',
        './desert-rat/idle/left/07.png',
        './desert-rat/idle/left/08.png',
        './desert-rat/idle/left/09.png'
    ],
    framesMax: 9,
    scale: 2.5, // Same scale as player (will be 200px)
    offset: {
        x: 215,
        y: 170
    },
    sprites: {
        idle: {
            images: [
                './desert-rat/idle/left/01.png',
                './desert-rat/idle/left/02.png',
                './desert-rat/idle/left/03.png',
                './desert-rat/idle/left/04.png',
                './desert-rat/idle/left/05.png',
                './desert-rat/idle/left/06.png',
                './desert-rat/idle/left/07.png',
                './desert-rat/idle/left/08.png',
                './desert-rat/idle/left/09.png'
            ],
            framesMax: 9
        },
        run: {
            images: [
                './desert-rat/idle/left/01.png',
                './desert-rat/idle/left/02.png',
                './desert-rat/idle/left/03.png',
                './desert-rat/idle/left/04.png',
                './desert-rat/idle/left/05.png',
                './desert-rat/idle/left/06.png',
                './desert-rat/idle/left/07.png',
                './desert-rat/idle/left/08.png',
                './desert-rat/idle/left/09.png'
            ],
            framesMax: 9
        },
        jump: {
            images: [
                './desert-rat/idle/left/01.png'
            ],
            framesMax: 1
        },
        fall: {
            images: [
                './desert-rat/idle/left/01.png'
            ],
            framesMax: 1
        },
        attack1: {
            images: [
                './desert-rat/attack/left/01.png',
                './desert-rat/attack/left/02.png',
                './desert-rat/attack/left/03.png',
                './desert-rat/attack/left/04.png',
                './desert-rat/attack/left/05.png',
                './desert-rat/attack/left/06.png',
                './desert-rat/attack/left/07.png',
                './desert-rat/attack/left/08.png',
                './desert-rat/attack/left/09.png'
            ],
            framesMax: 9
        },
        takeHit: {
            images: [
                './desert-rat/idle/left/01.png',
                './desert-rat/idle/left/02.png',
                './desert-rat/idle/left/03.png'
            ],
            framesMax: 3
        },
        death: {
            images: [
                './desert-rat/idle/left/01.png',
                './desert-rat/idle/left/02.png',
                './desert-rat/idle/left/03.png',
                './desert-rat/idle/left/04.png',
                './desert-rat/idle/left/05.png',
                './desert-rat/idle/left/06.png',
                './desert-rat/idle/left/07.png'
            ],
            framesMax: 7
        }
    },
    attackBox: {
        offset: {
            x: -170,
            y: 50
        },
        width: 170,
        height: 50
    }
})

// Automatic fighting system - no keyboard controls needed
let currentRound = 1;
let isPlayerTurn = true;
let roundInProgress = false;
let roundCooldown = 0;
let attackCooldown = 0;
let attackExecuted = false;
let returningToPosition = false;
const ROUND_DELAY = 90; // frames between rounds (~1.5 seconds at 60fps) - faster
const ATTACK_DELAY = 15; // frames before attack executes - faster
const ATTACK_DISTANCE = 300; // Distance to maintain for attacks (increased to get closer)
const ATTACK_LUNGE_DISTANCE = 80; // How far forward to move when attacking

// Store starting positions
const playerStartX = 100;
const enemyStartX = 800;
// Return positions (closer to center, not all the way back)
const playerReturnX = 250; // Player returns to this position (not all the way left)
const enemyReturnX = 750; // Enemy returns to this position (not all the way right)

decreaseTimer();

function handleAutomaticFighting() {
    // Don't fight if someone is dead
    if (player.dead || enemy.dead) {
        return;
    }

    // Handle round cooldown
    if (roundCooldown > 0) {
        roundCooldown--;
        return;
    }

    const attacker = isPlayerTurn ? player : enemy;
    const defender = isPlayerTurn ? enemy : player;
    const attackerStartX = isPlayerTurn ? playerStartX : enemyStartX;
    const attackerReturnX = isPlayerTurn ? playerReturnX : enemyReturnX;
    const defenderStartX = isPlayerTurn ? enemyStartX : playerStartX;
    
    // Calculate distance between fighters
    const distance = Math.abs(attacker.position.x - defender.position.x);
    
    // If returning to position after attack
    if (returningToPosition) {
        const distanceToReturn = Math.abs(attacker.position.x - attackerReturnX);
        if (distanceToReturn > 5) {
            // Move back to return position (not all the way to starting position)
            if (attacker.position.x < attackerReturnX) {
                attacker.velocity.x = 8; // Faster return
            } else if (attacker.position.x > attackerReturnX) {
                attacker.velocity.x = -8; // Faster return
            }
        } else {
            // Reached return position
            attacker.position.x = attackerReturnX;
            attacker.velocity.x = 0;
            returningToPosition = false;
            isPlayerTurn = !isPlayerTurn;
            roundInProgress = false;
            roundCooldown = ROUND_DELAY;
            currentRound++;
            attackExecuted = false;
        }
        defender.velocity.x = 0;
        return;
    }
    
    // If too far, move closer to center
    if (distance > ATTACK_DISTANCE) {
        roundInProgress = false;
        attackExecuted = false;
        attackCooldown = 0;
        
        if (isPlayerTurn) {
            // Player moves right toward center
            if (player.position.x < enemy.position.x - ATTACK_DISTANCE) {
                player.velocity.x = 8; // Faster movement
            } else {
                player.velocity.x = 0;
            }
            enemy.velocity.x = 0;
        } else {
            // Enemy moves left toward center
            if (enemy.position.x > player.position.x + ATTACK_DISTANCE) {
                enemy.velocity.x = -8; // Faster movement
            } else {
                enemy.velocity.x = 0;
            }
            player.velocity.x = 0;
        }
    } else {
        // Close enough, perform attack
        if (!roundInProgress) {
            roundInProgress = true;
            attackCooldown = ATTACK_DELAY;
            attackExecuted = false;
        }
        
        // Calculate minimum distance to maintain (don't pass through opponent)
        // Desert rat (enemy) can get closer when attacking
        const minDistance = isPlayerTurn ? 120 : 80; // Enemy gets closer (80px) vs player (120px)
        const currentDistance = Math.abs(attacker.position.x - defender.position.x);
        
        // Move forward during attack
        if (attackCooldown > 0) {
            attackCooldown--;
            // Start moving forward, but stop if too close
            if (currentDistance > minDistance) {
                if (isPlayerTurn) {
                    attacker.velocity.x = 12; // Faster attack movement
                } else {
                    attacker.velocity.x = -12; // Faster attack movement
                }
            } else {
                attacker.velocity.x = 0; // Stop if too close
            }
            defender.velocity.x = 0;
        } else if (!attackExecuted) {
            // Execute attack
            if (!attacker.dead && !defender.dead && !attacker.isAttacking) {
                attacker.attack();
                attackExecuted = true;
                // Continue moving forward during attack to ensure contact, but stop if too close
                if (currentDistance > minDistance) {
                    if (isPlayerTurn) {
                        attacker.velocity.x = 10; // Faster
                    } else {
                        attacker.velocity.x = -10; // Faster
                    }
                } else {
                    attacker.velocity.x = 0; // Stop if too close
                }
            }
        } else {
            // Keep moving forward for a bit during attack animation to ensure hit connects
            const attackFramesMax = attacker.sprites.attack1.framesMax;
            if (attacker.frameCurrent < attackFramesMax - 1) {
                // Still in attack animation, keep moving forward but stop if too close
                if (currentDistance > minDistance) {
                    if (isPlayerTurn) {
                        attacker.velocity.x = 8; // Faster
                    } else {
                        attacker.velocity.x = -8; // Faster
                    }
                } else {
                    attacker.velocity.x = 0; // Stop if too close
                }
            } else {
                // Attack animation done, slow down
                attacker.velocity.x *= 0.9;
            }
        }
        
        // Wait for attack animation to complete, then return to position
        const attackFramesMax = attacker.sprites.attack1.framesMax;
        const attackComplete = attackExecuted && 
            (attacker.image !== attacker.sprites.attack1.image || 
             attacker.frameCurrent >= attackFramesMax - 1);
        
        // Also wait for defender's hit animation to complete if they were hit
        const defenderHitFramesMax = defender.sprites.takeHit.framesMax;
        const hitAnimationComplete = defender.image !== defender.sprites.takeHit.image ||
            defender.frameCurrent >= defenderHitFramesMax - 1;
        
        if (attackComplete && !attacker.isAttacking && hitAnimationComplete) {
            // Start returning to starting position
            returningToPosition = true;
            attacker.velocity.x = 0;
        }
    }
}

function animate() {
    window.requestAnimationFrame(animate);
    context.fillStyle = 'black';
    context.fillRect(0, 0, canvas.width, canvas.height);
    background.update();
    shop.update();
    
    // Handle automatic fighting (this sets velocities)
    handleAutomaticFighting();
    
    player.update();
    enemy.update();

    // Handle sprite animations based on velocity and state
    if (player.isAttacking) {
        // Don't change sprite during attack
    } else if (player.velocity.x !== 0) {
        player.switchSprite('run');
    } else if (player.velocity.y < 0) {
        player.switchSprite('jump');
    } else if (player.velocity.y > 0) {
        player.switchSprite('fall');
    } else if (!player.dead) {
        player.switchSprite('idle');
    }

    if (enemy.isAttacking) {
        // Don't change sprite during attack
    } else if (enemy.velocity.x !== 0) {
        enemy.switchSprite('run');
    } else if (enemy.velocity.y < 0) {
        enemy.switchSprite('jump');
    } else if (enemy.velocity.y > 0) {
        enemy.switchSprite('fall');
    } else if (!enemy.dead) {
        enemy.switchSprite('idle');
    }

    //detect colission & enemy hit
    // Check frames 1, 2, 3, or 4 for player attack (5 frame animation)
    // Check collision more frequently during attack
    if (player.isAttacking && (player.frameCurrent >= 1 && player.frameCurrent <= 4)) {
        if (rectangularCollision({
            rectangle1: player,
            rectangle2: enemy
        })) {
            enemy.takeHit();
            player.isAttacking = false;
            gsap.to('#enemyHealth', {
                width: enemy.health + '%'
            })
        }
    }

    // if player misses - check at end of attack animation
    if (player.isAttacking && player.frameCurrent >= 4) {
        player.isAttacking = false;
    }

    // enemy hits
    // Check frames 2, 3, 4, 5, 6, 7, or 8 for enemy attack (9 frame animation)
    // Check collision more frequently during attack
    if (enemy.isAttacking && (enemy.frameCurrent >= 2 && enemy.frameCurrent <= 8)) {
        if (rectangularCollision({
            rectangle1: enemy,
            rectangle2: player
        })) {
            player.takeHit();
            enemy.isAttacking = false;
            gsap.to('#playerHealth', {
                width: player.health + '%'
            })
        }
    }

    // if enemy misses - check at end of attack animation
    if (enemy.isAttacking && enemy.frameCurrent >= 8) {
        enemy.isAttacking = false;
    }

    // end game based on health
    if (enemy.health <= 0 || player.health <= 0) {
        determinewinner({ player, enemy, timerId });
    }
}

animate();

// Keyboard controls removed - game is now fully automatic