<?php

namespace App\Console;

use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Foundation\Console\Kernel as ConsoleKernel;

class Kernel extends ConsoleKernel
{
    /**
     * Define the application's command schedule.
     */
    protected function schedule(Schedule $schedule): void
    {
        // Regenerate daily tasks every day at midnight
        $schedule->call(function () {
            app(\App\Services\TaskService::class)->regenerateDailyTasks();
        })->daily();

        // Regenerate weekly tasks every Monday at midnight
        $schedule->call(function () {
            app(\App\Services\TaskService::class)->regenerateWeeklyTasks();
        })->weeklyOn(1, '0:0');

        // Grant daily VIP rewards every day at midnight
        $schedule->call(function () {
            $vipService = app(\App\Services\VipService::class);
            $users = \App\Models\User::where('is_vip', true)
                ->whereNotNull('vip_expires_at')
                ->where('vip_expires_at', '>', now())
                ->get();
            
            foreach ($users as $user) {
                $vipService->grantDailyVipRewards($user);
            }
        })->daily();

        // Check and deactivate expired VIP memberships every minute
        $schedule->call(function () {
            $vipService = app(\App\Services\VipService::class);
            $expiredUsers = \App\Models\User::where('is_vip', true)
                ->where(function ($query) {
                    $query->whereNull('vip_expires_at')
                        ->orWhere('vip_expires_at', '<=', now());
                })
                ->get();
            
            foreach ($expiredUsers as $user) {
                $vipService->deactivateVip($user);
            }
        })->everyMinute();

        // Process reminders every minute
        $schedule->command('reminders:process')->everyMinute();

        // Regenerate HP every minute
        $schedule->command('hp:regenerate')->everyMinute();
    }

    /**
     * Register the commands for the application.
     */
    protected function commands(): void
    {
        $this->load(__DIR__.'/Commands');

        require base_path('routes/console.php');
    }
}

