<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Task;
use App\Models\Item;
use App\Models\Battle;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class AdminController extends Controller
{
    /**
     * Display admin dashboard with analytics.
     */
    public function index()
    {
        // User stats
        $totalUsers = User::count();
        $activeUsers = User::whereHas('tasks', function($q) {
            $q->where('completed_at', '>=', Carbon::now()->subWeek());
        })->count();
        $totalAdmins = User::where('is_admin', true)->count();

        // Task stats
        $totalTasks = Task::count();
        $completedTasks = Task::where('status', 'completed')->count();
        $completionRate = $totalTasks > 0 ? ($completedTasks / $totalTasks) * 100 : 0;
        $tasksToday = Task::whereDate('completed_at', today())->count();
        $tasksThisWeek = Task::whereBetween('completed_at', [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()])->count();

        // Game stats
        $avgXP = User::avg('xp');
        $avgGold = User::avg('gold');
        $avgLevel = User::avg('level');
        $totalBattles = Battle::count();
        $battlesWon = Battle::where('result', 'win')->count();
        $winRate = $totalBattles > 0 ? ($battlesWon / $totalBattles) * 100 : 0;

        // Popular items
        $popularItems = Item::withCount('userItems')
            ->orderBy('user_items_count', 'desc')
            ->limit(5)
            ->get();

        // Leaderboard (top 10)
        $leaderboard = User::orderBy('level', 'desc')
            ->orderBy('xp', 'desc')
            ->limit(10)
            ->get();

        // Category completion rates
        $categoryStats = DB::table('categories')
            ->leftJoin('tasks', 'categories.id', '=', 'tasks.category_id')
            ->select('categories.name', DB::raw('COUNT(tasks.id) as total_tasks'), 
                    DB::raw('SUM(CASE WHEN tasks.status = "completed" THEN 1 ELSE 0 END) as completed_tasks'))
            ->groupBy('categories.id', 'categories.name')
            ->get()
            ->map(function($cat) {
                $cat->completion_rate = $cat->total_tasks > 0 ? ($cat->completed_tasks / $cat->total_tasks) * 100 : 0;
                return $cat;
            })
            ->sortByDesc('completion_rate');

        return view('admin.dashboard', compact(
            'totalUsers', 'activeUsers', 'totalAdmins',
            'totalTasks', 'completedTasks', 'completionRate', 'tasksToday', 'tasksThisWeek',
            'avgXP', 'avgGold', 'avgLevel',
            'totalBattles', 'battlesWon', 'winRate',
            'popularItems', 'leaderboard', 'categoryStats'
        ));
    }

    /**
     * Display user management.
     */
    public function users()
    {
        $users = User::withCount(['tasks', 'battles'])
            ->orderBy('level', 'desc')
            ->paginate(20);

        return view('admin.users', compact('users'));
    }

    /**
     * Toggle admin status for a user.
     */
    public function toggleAdmin(User $user)
    {
        $user->update(['is_admin' => !$user->is_admin]);

        return redirect()->back()
            ->with('success', "Admin status " . ($user->is_admin ? 'granted' : 'revoked') . " for {$user->name}");
    }
}
