<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Arena3Background;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class Arena3BackgroundController extends Controller
{
    /**
     * Display a listing of backgrounds.
     */
    public function index()
    {
        $backgrounds = Arena3Background::orderBy('sort_order')->orderBy('name')->paginate(20);
        return view('admin.arena3-backgrounds.index', compact('backgrounds'));
    }

    /**
     * Show the form for creating a new background.
     */
    public function create()
    {
        return view('admin.arena3-backgrounds.create');
    }

    /**
     * Store a newly created background.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'image' => ['required', 'image', 'max:10240'],
            'shop_image' => ['nullable', 'image', 'max:10240'],
            'cost_gold' => ['nullable', 'integer', 'min:0'],
            'cost_premium' => ['nullable', 'integer', 'min:0'],
            'unlock_level' => ['nullable', 'integer', 'min:1'],
            'is_vip_only' => ['nullable', 'boolean'],
            'is_default' => ['nullable', 'boolean'],
            'is_active' => ['nullable', 'boolean'],
            'sort_order' => ['nullable', 'integer'],
        ]);

        // Generate slug
        $validated['slug'] = Str::slug($validated['name']);

        // Handle image uploads
        if ($request->hasFile('image')) {
            $validated['image_path'] = $request->file('image')->store('arena3/backgrounds', 'public');
        }
        
        if ($request->hasFile('shop_image')) {
            $validated['shop_image_path'] = $request->file('shop_image')->store('arena3/backgrounds', 'public');
        }

        // Set defaults
        $validated['cost_gold'] = $validated['cost_gold'] ?? null;
        $validated['cost_premium'] = $validated['cost_premium'] ?? null;
        $validated['unlock_level'] = $validated['unlock_level'] ?? null;
        $validated['is_vip_only'] = $request->has('is_vip_only') ? true : false;
        $validated['is_default'] = $request->has('is_default') ? true : false;
        $validated['is_active'] = $request->has('is_active') ? true : false;
        $validated['sort_order'] = $validated['sort_order'] ?? 0;

        // If this is set as default, unset other defaults
        if ($validated['is_default']) {
            Arena3Background::where('is_default', true)->update(['is_default' => false]);
        }

        Arena3Background::create($validated);

        return redirect()->route('admin.arena3-backgrounds.index')
            ->with('success', 'Background created successfully!');
    }

    /**
     * Show the form for editing the specified background.
     */
    public function edit(Arena3Background $arena3Background)
    {
        return view('admin.arena3-backgrounds.edit', compact('arena3Background'));
    }

    /**
     * Update the specified background.
     */
    public function update(Request $request, Arena3Background $arena3Background)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'image' => ['nullable', 'image', 'max:10240'],
            'shop_image' => ['nullable', 'image', 'max:10240'],
            'cost_gold' => ['nullable', 'integer', 'min:0'],
            'cost_premium' => ['nullable', 'integer', 'min:0'],
            'unlock_level' => ['nullable', 'integer', 'min:1'],
            'is_vip_only' => ['nullable', 'boolean'],
            'is_default' => ['nullable', 'boolean'],
            'is_active' => ['nullable', 'boolean'],
            'sort_order' => ['nullable', 'integer'],
        ]);

        // Handle image uploads
        if ($request->hasFile('image')) {
            if ($arena3Background->image_path) {
                Storage::disk('public')->delete($arena3Background->image_path);
            }
            $validated['image_path'] = $request->file('image')->store('arena3/backgrounds', 'public');
        }
        
        if ($request->hasFile('shop_image')) {
            if ($arena3Background->shop_image_path) {
                Storage::disk('public')->delete($arena3Background->shop_image_path);
            }
            $validated['shop_image_path'] = $request->file('shop_image')->store('arena3/backgrounds', 'public');
        }

        // Set defaults
        $validated['cost_gold'] = $validated['cost_gold'] ?? null;
        $validated['cost_premium'] = $validated['cost_premium'] ?? null;
        $validated['unlock_level'] = $validated['unlock_level'] ?? null;
        $validated['is_vip_only'] = $request->has('is_vip_only') ? true : false;
        $validated['is_default'] = $request->has('is_default') ? true : false;
        $validated['is_active'] = $request->has('is_active') ? true : false;

        // If this is set as default, unset other defaults
        if ($validated['is_default'] && !$arena3Background->is_default) {
            Arena3Background::where('id', '!=', $arena3Background->id)
                ->where('is_default', true)
                ->update(['is_default' => false]);
        }

        $arena3Background->update($validated);

        return redirect()->route('admin.arena3-backgrounds.index')
            ->with('success', 'Background updated successfully!');
    }

    /**
     * Remove the specified background.
     */
    public function destroy(Arena3Background $arena3Background)
    {
        // Delete images
        if ($arena3Background->image_path) {
            Storage::disk('public')->delete($arena3Background->image_path);
        }
        if ($arena3Background->shop_image_path) {
            Storage::disk('public')->delete($arena3Background->shop_image_path);
        }

        $arena3Background->delete();

        return redirect()->route('admin.arena3-backgrounds.index')
            ->with('success', 'Background deleted successfully!');
    }
}

