<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Character;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class CharacterController extends Controller
{
    /**
     * Display a listing of characters.
     */
    public function index()
    {
        $characters = Character::orderBy('sort_order')->orderBy('name')->paginate(20);
        return view('admin.characters.index', compact('characters'));
    }

    /**
     * Show the form for creating a new character.
     */
    public function create()
    {
        return view('admin.characters.create');
    }

    /**
     * Store a newly created character.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'unlock_level' => ['nullable', 'integer', 'min:1'],
            'cost_gold' => ['nullable', 'integer', 'min:0'],
            'cost_premium' => ['nullable', 'integer', 'min:0'],
            'is_vip_only' => ['nullable', 'boolean'],
            'hp_bonus' => ['nullable', 'integer', 'min:0'],
            'attack_bonus' => ['nullable', 'integer', 'min:0'],
            'defense_bonus' => ['nullable', 'integer', 'min:0'],
            'stamina_bonus' => ['nullable', 'integer', 'min:0'],
            'base_hp' => ['nullable', 'integer', 'min:1'],
            'base_attack' => ['nullable', 'integer', 'min:1'],
            'base_defense' => ['nullable', 'integer', 'min:1'],
            'rarity' => ['required', 'in:common,rare,epic,legendary'],
            'sort_order' => ['nullable', 'integer'],
            'is_active' => ['nullable', 'boolean'],
            'image_normal' => ['nullable', 'image', 'max:5120'],
            'image_win' => ['nullable', 'image', 'max:5120'],
            'image_loss' => ['nullable', 'image', 'max:5120'],
            'sprite_source' => ['nullable', 'string', 'max:500'],
            'video_url' => ['nullable', 'url', 'max:500'],
            'video_win' => ['nullable', 'url', 'max:500'],
            'video_loss' => ['nullable', 'url', 'max:500'],
        ]);

        // Generate slug
        $validated['slug'] = Str::slug($validated['name']);
        $characterSlug = $validated['slug'];

        // Handle PNG sequence uploads for Arena 3
        $this->handlePngSequenceUploads($request, $characterSlug);

        // Handle image uploads
        if ($request->hasFile('image_normal')) {
            $validated['image_normal'] = $request->file('image_normal')->store('characters', 'public');
        }
        if ($request->hasFile('image_idle')) {
            $validated['image_idle'] = $request->file('image_idle')->store('characters', 'public');
        }
        if ($request->hasFile('image_idle_animation')) {
            $validated['image_idle_animation'] = $request->file('image_idle_animation')->store('characters', 'public');
        }
        if ($request->hasFile('image_win')) {
            $validated['image_win'] = $request->file('image_win')->store('characters', 'public');
        }
        if ($request->hasFile('image_winning')) {
            $validated['image_winning'] = $request->file('image_winning')->store('characters', 'public');
        }
        if ($request->hasFile('image_winning_animation')) {
            $validated['image_winning_animation'] = $request->file('image_winning_animation')->store('characters', 'public');
        }
        if ($request->hasFile('image_loss')) {
            $validated['image_loss'] = $request->file('image_loss')->store('characters', 'public');
        }
        if ($request->hasFile('image_lost')) {
            $validated['image_lost'] = $request->file('image_lost')->store('characters', 'public');
        }
        if ($request->hasFile('image_lost_animation')) {
            $validated['image_lost_animation'] = $request->file('image_lost_animation')->store('characters', 'public');
        }

        // Set defaults for optional fields
        $validated['hp_bonus'] = $validated['hp_bonus'] ?? 0;
        $validated['attack_bonus'] = $validated['attack_bonus'] ?? 0;
        $validated['defense_bonus'] = $validated['defense_bonus'] ?? 0;
        $validated['stamina_bonus'] = $validated['stamina_bonus'] ?? 0;
        $validated['sort_order'] = $validated['sort_order'] ?? 0;
        $validated['is_vip_only'] = $request->has('is_vip_only') ? true : false;
        $validated['is_active'] = $request->has('is_active') ? true : false;

        Character::create($validated);

        return redirect()->route('admin.characters.index')
            ->with('success', 'Character created successfully!');
    }

    /**
     * Display the specified character.
     */
    public function show(Character $character)
    {
        // Redirect to edit page for now, or create a show view if needed
        return redirect()->route('admin.characters.edit', $character);
    }

    /**
     * Show the form for editing the specified character.
     */
    public function edit(Character $character)
    {
        return view('admin.characters.edit', compact('character'));
    }

    /**
     * Update the specified character.
     */
    public function update(Request $request, Character $character)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'unlock_level' => ['nullable', 'integer', 'min:1'],
            'cost_gold' => ['nullable', 'integer', 'min:0'],
            'cost_premium' => ['nullable', 'integer', 'min:0'],
            'is_vip_only' => ['nullable', 'boolean'],
            'hp_bonus' => ['nullable', 'integer', 'min:0'],
            'attack_bonus' => ['nullable', 'integer', 'min:0'],
            'defense_bonus' => ['nullable', 'integer', 'min:0'],
            'stamina_bonus' => ['nullable', 'integer', 'min:0'],
            'base_hp' => ['nullable', 'integer', 'min:1'],
            'base_attack' => ['nullable', 'integer', 'min:1'],
            'base_defense' => ['nullable', 'integer', 'min:1'],
            'rarity' => ['required', 'in:common,rare,epic,legendary'],
            'sort_order' => ['nullable', 'integer'],
            'is_active' => ['nullable', 'boolean'],
            'image_normal' => ['nullable', 'image', 'max:5120'],
            'image_idle' => ['nullable', 'image', 'max:5120'],
            'image_idle_animation' => ['nullable', 'mimes:gif,mp4,webm', 'max:10240'],
            'image_win' => ['nullable', 'image', 'max:5120'],
            'image_winning' => ['nullable', 'image', 'max:5120'],
            'image_winning_animation' => ['nullable', 'mimes:gif,mp4,webm', 'max:10240'],
            'image_loss' => ['nullable', 'image', 'max:5120'],
            'image_lost' => ['nullable', 'image', 'max:5120'],
            'image_lost_animation' => ['nullable', 'mimes:gif,mp4,webm', 'max:10240'],
            'sprite_source' => ['nullable', 'string', 'max:500'],
            'video_url' => ['nullable', 'url', 'max:500'],
            'video_win' => ['nullable', 'url', 'max:500'],
            'video_loss' => ['nullable', 'url', 'max:500'],
        ]);

        // Generate slug if name changed
        if ($character->name !== $validated['name']) {
            $validated['slug'] = Str::slug($validated['name']);
        }
        $characterSlug = $validated['slug'] ?? $character->slug;

        // Handle PNG sequence uploads for Arena 3
        $this->handlePngSequenceUploads($request, $characterSlug);

        // Handle image uploads (only update if new file provided)
        if ($request->hasFile('image_normal')) {
            if ($character->image_normal) {
                Storage::disk('public')->delete($character->image_normal);
            }
            $validated['image_normal'] = $request->file('image_normal')->store('characters', 'public');
        }
        if ($request->hasFile('image_idle')) {
            if ($character->image_idle) {
                Storage::disk('public')->delete($character->image_idle);
            }
            $validated['image_idle'] = $request->file('image_idle')->store('characters', 'public');
        }
        if ($request->hasFile('image_idle_animation')) {
            if ($character->image_idle_animation) {
                Storage::disk('public')->delete($character->image_idle_animation);
            }
            $validated['image_idle_animation'] = $request->file('image_idle_animation')->store('characters', 'public');
        }
        if ($request->hasFile('image_win')) {
            if ($character->image_win) {
                Storage::disk('public')->delete($character->image_win);
            }
            $validated['image_win'] = $request->file('image_win')->store('characters', 'public');
        }
        if ($request->hasFile('image_winning')) {
            if ($character->image_winning) {
                Storage::disk('public')->delete($character->image_winning);
            }
            $validated['image_winning'] = $request->file('image_winning')->store('characters', 'public');
        }
        if ($request->hasFile('image_winning_animation')) {
            if ($character->image_winning_animation) {
                Storage::disk('public')->delete($character->image_winning_animation);
            }
            $validated['image_winning_animation'] = $request->file('image_winning_animation')->store('characters', 'public');
        }
        if ($request->hasFile('image_loss')) {
            if ($character->image_loss) {
                Storage::disk('public')->delete($character->image_loss);
            }
            $validated['image_loss'] = $request->file('image_loss')->store('characters', 'public');
        }
        if ($request->hasFile('image_lost')) {
            if ($character->image_lost) {
                Storage::disk('public')->delete($character->image_lost);
            }
            $validated['image_lost'] = $request->file('image_lost')->store('characters', 'public');
        }
        if ($request->hasFile('image_lost_animation')) {
            if ($character->image_lost_animation) {
                Storage::disk('public')->delete($character->image_lost_animation);
            }
            $validated['image_lost_animation'] = $request->file('image_lost_animation')->store('characters', 'public');
        }

        // Set defaults for optional fields
        $validated['hp_bonus'] = $validated['hp_bonus'] ?? $character->hp_bonus ?? 0;
        $validated['attack_bonus'] = $validated['attack_bonus'] ?? $character->attack_bonus ?? 0;
        $validated['defense_bonus'] = $validated['defense_bonus'] ?? $character->defense_bonus ?? 0;
        $validated['stamina_bonus'] = $validated['stamina_bonus'] ?? $character->stamina_bonus ?? 0;
        $validated['sort_order'] = $validated['sort_order'] ?? $character->sort_order ?? 0;
        $validated['is_vip_only'] = $request->has('is_vip_only') ? true : false;
        $validated['is_active'] = $request->has('is_active') ? true : false;

        $character->update($validated);

        return redirect()->route('admin.characters.index')
            ->with('success', 'Character updated successfully!');
    }

    /**
     * Remove the specified character.
     */
    public function destroy(Character $character)
    {
        // Delete images
        $imageFields = [
            'image_normal', 'image_idle', 'image_idle_animation',
            'image_win', 'image_winning', 'image_winning_animation',
            'image_loss', 'image_lost', 'image_lost_animation'
        ];
        
        foreach ($imageFields as $field) {
            if ($character->$field) {
                Storage::disk('public')->delete($character->$field);
            }
        }

        $character->delete();

        return redirect()->route('admin.characters.index')
            ->with('success', 'Character deleted successfully!');
    }

    /**
     * Handle PNG sequence uploads for Arena 3 animations.
     * 
     * @param Request $request
     * @param string $characterSlug
     * @return void
     */
    private function handlePngSequenceUploads(Request $request, string $characterSlug): void
    {
        $states = [
            'png_idle' => 'idle',
            'png_fight' => 'fight',
            'png_win' => 'win',
            'png_lose' => 'lose',
        ];

        foreach ($states as $inputName => $folderName) {
            if ($request->hasFile($inputName)) {
                $files = $request->file($inputName);
                
                // Ensure we have exactly 5 files
                if (count($files) !== 5) {
                    continue; // Skip if not exactly 5 files
                }

                // Create directory if it doesn't exist
                $directory = public_path("assets/characters/{$characterSlug}/{$folderName}");
                if (!is_dir($directory)) {
                    mkdir($directory, 0755, true);
                }

                // Sort files by name to ensure correct order
                $sortedFiles = collect($files)->sortBy(function ($file) {
                    return $file->getClientOriginalName();
                })->values()->all();

                // Delete existing files in the directory
                $existingFiles = glob("{$directory}/*.png");
                foreach ($existingFiles as $existingFile) {
                    unlink($existingFile);
                }

                // Save files with numbered names (01.png, 02.png, etc.)
                foreach ($sortedFiles as $index => $file) {
                    if ($file->isValid() && $file->getMimeType() === 'image/png') {
                        $frameNumber = str_pad($index + 1, 2, '0', STR_PAD_LEFT);
                        $fileName = "{$frameNumber}.png";
                        $file->move($directory, $fileName);
                    }
                }
            }
        }
    }
}
