<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Enemy;
use App\Models\Item;
use Illuminate\Http\Request;

class EnemyController extends Controller
{
    /**
     * Display a listing of enemies.
     */
    public function index()
    {
        $enemies = Enemy::with('dropItem')->orderBy('level')->paginate(20);
        return view('admin.enemies.index', compact('enemies'));
    }

    /**
     * Show the form for creating a new enemy.
     */
    public function create()
    {
        $items = Item::all();
        return view('admin.enemies.create', compact('items'));
    }

    /**
     * Store a newly created enemy.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'level' => ['required', 'integer', 'min:1'],
            'attack' => ['required', 'integer', 'min:1'],
            'defense' => ['required', 'integer', 'min:0'],
            'hp' => ['required', 'integer', 'min:1'],
            'image_url' => ['nullable', 'url'],
            'xp_reward' => ['required', 'integer', 'min:0'],
            'gold_reward' => ['required', 'integer', 'min:0'],
            'drop_item_id' => ['nullable', 'exists:items,id'],
        ]);

        Enemy::create($validated);

        return redirect()->route('admin.enemies.index')
            ->with('success', 'Enemy created successfully.');
    }

    /**
     * Show the form for editing an enemy.
     */
    public function edit(Enemy $enemy)
    {
        $items = Item::all();
        return view('admin.enemies.edit', compact('enemy', 'items'));
    }

    /**
     * Update an enemy.
     */
    public function update(Request $request, Enemy $enemy)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'level' => ['required', 'integer', 'min:1'],
            'attack' => ['required', 'integer', 'min:1'],
            'defense' => ['required', 'integer', 'min:0'],
            'hp' => ['required', 'integer', 'min:1'],
            'image_url' => ['nullable', 'url'],
            'xp_reward' => ['required', 'integer', 'min:0'],
            'gold_reward' => ['required', 'integer', 'min:0'],
            'drop_item_id' => ['nullable', 'exists:items,id'],
        ]);

        $enemy->update($validated);

        return redirect()->route('admin.enemies.index')
            ->with('success', 'Enemy updated successfully.');
    }

    /**
     * Remove an enemy.
     */
    public function destroy(Enemy $enemy)
    {
        $enemy->delete();

        return redirect()->route('admin.enemies.index')
            ->with('success', 'Enemy deleted successfully.');
    }
}
