<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Item;
use Illuminate\Http\Request;

class ItemController extends Controller
{
    /**
     * Display a listing of items.
     */
    public function index()
    {
        $items = Item::orderBy('required_level')->orderBy('cost')->paginate(20);
        return view('admin.items.index', compact('items'));
    }

    /**
     * Show the form for creating a new item.
     */
    public function create()
    {
        return view('admin.items.create');
    }

    /**
     * Store a newly created item.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'type' => ['required', 'in:weapon,armor,potion,tool'],
            'attack' => ['required', 'integer', 'min:0'],
            'defense' => ['required', 'integer', 'min:0'],
            'magic' => ['required', 'integer', 'min:0'],
            'stamina' => ['required', 'integer', 'min:0'],
            'cost' => ['required', 'integer', 'min:0'],
            'image_url' => ['nullable', 'url'],
            'required_level' => ['required', 'integer', 'min:1'],
        ]);

        Item::create($validated);

        return redirect()->route('admin.items.index')
            ->with('success', 'Item created successfully.');
    }

    /**
     * Show the form for editing an item.
     */
    public function edit(Item $item)
    {
        return view('admin.items.edit', compact('item'));
    }

    /**
     * Update an item.
     */
    public function update(Request $request, Item $item)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'type' => ['required', 'in:weapon,armor,potion,tool'],
            'attack' => ['required', 'integer', 'min:0'],
            'defense' => ['required', 'integer', 'min:0'],
            'magic' => ['required', 'integer', 'min:0'],
            'stamina' => ['required', 'integer', 'min:0'],
            'cost' => ['required', 'integer', 'min:0'],
            'image_url' => ['nullable', 'url'],
            'required_level' => ['required', 'integer', 'min:1'],
        ]);

        $item->update($validated);

        return redirect()->route('admin.items.index')
            ->with('success', 'Item updated successfully.');
    }

    /**
     * Remove an item.
     */
    public function destroy(Item $item)
    {
        $item->delete();

        return redirect()->route('admin.items.index')
            ->with('success', 'Item deleted successfully.');
    }
}
