<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Mob;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class MobController extends Controller
{
    /**
     * Display a listing of mobs.
     */
    public function index()
    {
        $mobs = Mob::orderBy('level')->orderBy('name')->paginate(20);
        return view('admin.mobs.index', compact('mobs'));
    }

    /**
     * Show the form for creating a new mob.
     */
    public function create()
    {
        return view('admin.mobs.create');
    }

    /**
     * Store a newly created mob.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'level' => ['required', 'integer', 'min:1', 'max:80'],
            'hp' => ['required', 'integer', 'min:1'],
            'attack' => ['required', 'integer', 'min:1'],
            'defense' => ['required', 'integer', 'min:0'],
            'speed' => ['required', 'integer', 'min:1'],
            'luck' => ['required', 'integer', 'min:0', 'max:20'],
            'xp_reward' => ['required', 'integer', 'min:0'],
            'gold_reward' => ['required', 'integer', 'min:0'],
            'premium_coin_chance' => ['nullable', 'numeric', 'min:0', 'max:1'],
            'rarity' => ['required', 'in:common,rare,epic,legendary'],
            'ai_type' => ['required', 'in:passive,aggressive,strategic'],
            'image_normal' => ['nullable', 'image', 'max:2048'],
            'image_win' => ['nullable', 'image', 'max:2048'],
            'image_loss' => ['nullable', 'image', 'max:2048'],
        ]);

        $slug = Str::slug($validated['name'] . '-' . time());

        // Handle image uploads
        if ($request->hasFile('image_normal')) {
            $validated['image_normal'] = $request->file('image_normal')->store('mobs', 'public');
        }
        if ($request->hasFile('image_win')) {
            $validated['image_win'] = $request->file('image_win')->store('mobs', 'public');
        }
        if ($request->hasFile('image_loss')) {
            $validated['image_loss'] = $request->file('image_loss')->store('mobs', 'public');
        }

        $validated['slug'] = $slug;
        $validated['sprite_source'] = $validated['image_normal'] ?? "/assets/sprites/mobs/{$slug}.png";
        $validated['premium_coin_chance'] = $validated['premium_coin_chance'] ?? 0.005;

        Mob::create($validated);

        return redirect()->route('admin.mobs.index')
            ->with('success', 'Mob created successfully.');
    }

    /**
     * Show the form for editing a mob.
     */
    public function edit(Mob $mob)
    {
        return view('admin.mobs.edit', compact('mob'));
    }

    /**
     * Update a mob.
     */
    public function update(Request $request, Mob $mob)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'level' => ['required', 'integer', 'min:1', 'max:80'],
            'hp' => ['required', 'integer', 'min:1'],
            'attack' => ['required', 'integer', 'min:1'],
            'defense' => ['required', 'integer', 'min:0'],
            'speed' => ['required', 'integer', 'min:1'],
            'luck' => ['required', 'integer', 'min:0', 'max:20'],
            'xp_reward' => ['required', 'integer', 'min:0'],
            'gold_reward' => ['required', 'integer', 'min:0'],
            'premium_coin_chance' => ['nullable', 'numeric', 'min:0', 'max:1'],
            'rarity' => ['required', 'in:common,rare,epic,legendary'],
            'ai_type' => ['required', 'in:passive,aggressive,strategic'],
            'image_normal' => ['nullable', 'image', 'max:2048'],
            'image_win' => ['nullable', 'image', 'max:2048'],
            'image_loss' => ['nullable', 'image', 'max:2048'],
        ]);

        // Handle image uploads
        if ($request->hasFile('image_normal')) {
            // Delete old image
            if ($mob->image_normal) {
                Storage::disk('public')->delete($mob->image_normal);
            }
            $validated['image_normal'] = $request->file('image_normal')->store('mobs', 'public');
        }
        if ($request->hasFile('image_win')) {
            if ($mob->image_win) {
                Storage::disk('public')->delete($mob->image_win);
            }
            $validated['image_win'] = $request->file('image_win')->store('mobs', 'public');
        }
        if ($request->hasFile('image_loss')) {
            if ($mob->image_loss) {
                Storage::disk('public')->delete($mob->image_loss);
            }
            $validated['image_loss'] = $request->file('image_loss')->store('mobs', 'public');
        }

        $mob->update($validated);

        return redirect()->route('admin.mobs.index')
            ->with('success', 'Mob updated successfully.');
    }

    /**
     * Remove a mob.
     */
    public function destroy(Mob $mob)
    {
        // Delete images
        if ($mob->image_normal) {
            Storage::disk('public')->delete($mob->image_normal);
        }
        if ($mob->image_win) {
            Storage::disk('public')->delete($mob->image_win);
        }
        if ($mob->image_loss) {
            Storage::disk('public')->delete($mob->image_loss);
        }

        $mob->delete();

        return redirect()->route('admin.mobs.index')
            ->with('success', 'Mob deleted successfully.');
    }
}
