<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Mob;
use App\Services\BattleService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class BattleController extends Controller
{
    protected BattleService $battleService;

    public function __construct(BattleService $battleService)
    {
        $this->battleService = $battleService;
    }

    /**
     * Start a new battle.
     */
    public function start(Request $request)
    {
        $request->validate([
            'mob_id' => 'required|exists:mobs,id',
            'mode' => 'nullable|in:auto,manual',
        ]);

        $user = Auth::user();
        $mob = Mob::findOrFail($request->mob_id);
        $mode = $request->input('mode', 'auto');

        try {
            $result = $this->battleService->startBattle($user, $mob, $mode);
            return response()->json($result);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }

    /**
     * Perform a turn in manual mode.
     */
    public function turn(Request $request)
    {
        $request->validate([
            'battle_id' => 'required|string',
            'action' => 'nullable|in:attack,use_item,skill,attempt_escape',
        ]);

        $battleId = $request->input('battle_id');
        $action = $request->input('action', 'attack');

        try {
            $result = $this->battleService->performTurn($battleId, $action);
            return response()->json($result);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }

    /**
     * Get battle result.
     */
    public function result(string $battleId)
    {
        $battleInstance = \Illuminate\Support\Facades\Cache::get("battle_instance:{$battleId}");
        
        if (!$battleInstance) {
            return response()->json(['error' => 'Battle not found'], 404);
        }

        return response()->json($battleInstance);
    }

    /**
     * Get current battle state.
     */
    public function show(string $battleId)
    {
        $battleInstance = \Illuminate\Support\Facades\Cache::get("battle_instance:{$battleId}");
        
        if (!$battleInstance) {
            return response()->json(['error' => 'Battle not found'], 404);
        }

        return response()->json($battleInstance);
    }
}
