<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\ScrollCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class QuestController extends Controller
{
    /**
     * Get quests organized by scrolls with view mode support.
     */
    public function index(Request $request)
    {
        $viewMode = $request->get('view', 'cards');
        $scrollId = $request->get('scroll');

        // Get scrolls for the user
        $scrollsQuery = ScrollCategory::forUser(Auth::id())
            ->where('is_active', true)
            ->with(['tasks' => function($query) {
                $query->where('user_id', Auth::id())
                      ->with(['category', 'scrollCategory', 'reminders']);
            }])
            ->orderBy('sort_order');

        // Filter by specific scroll if requested
        if ($scrollId) {
            $scrollsQuery->where('id', $scrollId);
        }

        $scrolls = $scrollsQuery->get();

        // Apply additional filters
        foreach ($scrolls as $scroll) {
            $taskQuery = $scroll->tasks();

            if ($request->has('status')) {
                $taskQuery->where('status', $request->status);
            }

            if ($request->has('category_id')) {
                $taskQuery->where('category_id', $request->category_id);
            }

            if ($request->has('difficulty')) {
                $taskQuery->where('difficulty_level', $request->difficulty);
            }

            $scroll->setRelation('tasks', $taskQuery->orderBy('due_date', 'asc')
                ->orderBy('created_at', 'desc')
                ->get());
        }

        return response()->json([
            'view_mode' => $viewMode,
            'scrolls' => $scrolls->map(function($scroll) {
                return [
                    'id' => $scroll->id,
                    'title' => $scroll->title,
                    'icon' => $scroll->icon,
                    'color' => $scroll->color,
                    'description' => $scroll->description,
                    'rewards' => $scroll->rewards,
                    'quests' => $scroll->tasks->map(function($task) {
                        return [
                            'id' => $task->id,
                            'title' => $task->title,
                            'description' => $task->description,
                            'points' => $task->points,
                            'coins' => $task->coins,
                            'difficulty_level' => $task->difficulty_level,
                            'status' => $task->status,
                            'due_date' => $task->due_date?->toIso8601String(),
                            'category' => $task->category ? [
                                'id' => $task->category->id,
                                'name' => $task->category->name,
                            ] : null,
                        ];
                    }),
                ];
            }),
        ]);
    }
}
