<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\ShopItem;
use App\Services\ShopService;
use App\Services\CurrencyService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ShopController extends Controller
{
    protected ShopService $shopService;
    protected CurrencyService $currencyService;

    public function __construct(ShopService $shopService, CurrencyService $currencyService)
    {
        $this->shopService = $shopService;
        $this->currencyService = $currencyService;
    }

    /**
     * List shop items with filters.
     */
    public function index(Request $request)
    {
        $type = $request->input('type');
        $rarity = $request->input('rarity');
        $perPage = $request->input('per_page', 20);

        $query = ShopItem::query();

        if ($type) {
            $query->where('type', $type);
        }

        if ($rarity) {
            $query->where('rarity', $rarity);
        }

        $items = $query->paginate($perPage);

        return response()->json($items);
    }

    /**
     * Get specific item details.
     */
    public function show(string $id)
    {
        $item = ShopItem::findOrFail($id);
        return response()->json($item);
    }

    /**
     * Purchase an item.
     */
    public function buy(Request $request)
    {
        $request->validate([
            'item_id' => 'required|exists:shop_items,id',
            'currency' => 'required|in:gold,premium',
        ]);

        $user = Auth::user();
        $item = ShopItem::findOrFail($request->item_id);

        try {
            // Check if already owned
            if ($user->shopItems()->where('shop_item_id', $item->id)->exists()) {
                return response()->json([
                    'success' => false,
                    'error' => 'You already own this item.',
                ], 400);
            }

            // Check VIP requirement
            if ($item->is_vip_only && !$user->isVip()) {
                return response()->json([
                    'success' => false,
                    'error' => 'VIP membership required for this item.',
                ], 400);
            }

            // Purchase with gold
            if ($request->currency === 'gold') {
                if ($item->cost_gold <= 0) {
                    return response()->json([
                        'success' => false,
                        'error' => 'This item cannot be purchased with gold.',
                    ], 400);
                }

                if ($user->gold < $item->cost_gold) {
                    return response()->json([
                        'success' => false,
                        'error' => 'Insufficient gold.',
                    ], 400);
                }

                $this->currencyService->deductGold($user, $item->cost_gold);
            } else {
                // Purchase with premium coins
                if ($item->cost_premium <= 0) {
                    return response()->json([
                        'success' => false,
                        'error' => 'This item cannot be purchased with premium coins.',
                    ], 400);
                    }

                if (!$this->currencyService->hasEnoughPremiumCoins($user, $item->cost_premium)) {
                    return response()->json([
                        'success' => false,
                        'error' => 'Insufficient premium coins.',
                    ], 400);
                }

                $this->currencyService->deductPremiumCoins($user, $item->cost_premium);
            }

            // Add item to user's inventory
            $user->shopItems()->attach($item->id, [
                'equipped' => false,
                'purchased_at' => now(),
            ]);

            return response()->json([
                'success' => true,
                'message' => "Successfully purchased {$item->name}!",
                'item' => $item,
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }
}
