<?php

namespace App\Http\Controllers;

use App\Models\Mob;
use App\Models\Battle;
use App\Services\BattleService;
use App\Services\MobService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class BattleController extends Controller
{
    protected BattleService $battleService;
    protected MobService $mobService;

    public function __construct(BattleService $battleService, MobService $mobService)
    {
        $this->battleService = $battleService;
        $this->mobService = $mobService;
    }

    /**
     * Display available mobs for battle.
     */
    public function index()
    {
        $user = Auth::user()->load('character');
        $mobs = $this->battleService->getAvailableMobs($user);
        
        return view('battle.index', compact('mobs', 'user'));
    }

    /**
     * Start a new battle.
     */
    public function start(Request $request, $mobId)
    {
        $user = Auth::user();
        
        // Find the mob by ID
        $mob = Mob::find($mobId);
        
        if (!$mob) {
            return redirect()->route('battle.index')
                ->with('error', 'Mob not found.');
        }

        // Check cooldown
        if (!$this->battleService->canStartBattle($user)) {
            $cooldownSeconds = $this->battleService->getBattleCooldown($user);
            $cooldownMinutes = ceil($cooldownSeconds / 60);
            return redirect()->route('battle.index')
                ->with('error', "Battle cooldown active. Please wait {$cooldownMinutes} minute(s) before starting another battle.");
        }

        // Verify mob is appropriate level
        $availableMobs = $this->battleService->getAvailableMobs($user);
        $mobIds = collect($availableMobs)->pluck('id')->toArray();
        if (!in_array($mob->id, $mobIds)) {
            return redirect()->route('battle.index')
                ->with('error', 'This mob is not available for your level.');
        }

        try {
            // Start battle in auto mode for web interface
            $result = $this->battleService->startBattle($user, $mob, 'auto');
            
            // Reload user to get updated stats
            $user->refresh();
            
            // Redirect to a battle result page or show the result
            return redirect()->route('battle.index')
                ->with('battle_result', $result)
                ->with('battle_mob_id', $mob->id);
        } catch (\Exception $e) {
            \Log::error('Battle start error: ' . $e->getMessage(), [
                'user_id' => $user->id,
                'mob_id' => $mob->id,
                'trace' => $e->getTraceAsString()
            ]);
            return redirect()->route('battle.index')
                ->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }

    /**
     * Display the battle interface.
     * Note: This is kept for backward compatibility with old Battle model.
     * New battles use the API endpoints.
     */
    public function show(Battle $battle)
    {
        if ($battle->user_id !== Auth::id()) {
            abort(403);
        }

        $battle->load(['user', 'enemy']);

        // If battle is complete, show result
        if ($battle->result !== null) {
            return view('battle.result', compact('battle'));
        }

        $userStats = $battle->user->getTotalStats();
        
        // Calculate HP percentages
        $userMaxHP = max(1, $userStats['hp'] ?? 100);
        $enemyMaxHP = max(1, $battle->enemy->hp ?? 100);
        $userHP = $battle->user_hp_remaining > 0 ? $battle->user_hp_remaining : $userMaxHP;
        $enemyHP = $battle->enemy_hp_remaining > 0 ? $battle->enemy_hp_remaining : $enemyMaxHP;
        
        $userHPPercent = ($userHP / $userMaxHP) * 100;
        $enemyHPPercent = ($enemyHP / $enemyMaxHP) * 100;

        return view('battle.show', compact('battle', 'userStats', 'userHP', 'enemyHP', 'userMaxHP', 'enemyMaxHP', 'userHPPercent', 'enemyHPPercent'));
    }

    /**
     * Execute an attack turn.
     * Note: This is kept for backward compatibility with old Battle model.
     * New battles use the API endpoints.
     */
    public function attack(Battle $battle)
    {
        if ($battle->user_id !== Auth::id()) {
            abort(403);
        }

        // Old battle system - redirect to new system
        return redirect()->route('battle.index')
            ->with('info', 'Please use the new battle system via the battle selection page.');
    }

    /**
     * Display battle history.
     */
    public function history()
    {
        // Show both old battles and new battle histories
        $oldBattles = Auth::user()->battles()
            ->with('enemy')
            ->orderBy('created_at', 'desc')
            ->paginate(10);
            
        $newBattles = Auth::user()->battleHistories()
            ->with('mob')
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('battle.history', compact('oldBattles', 'newBattles'));
    }
}
