<?php

namespace App\Http\Controllers;

use App\Models\Character;
use App\Services\CurrencyService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class CharacterController extends Controller
{
    protected CurrencyService $currencyService;

    public function __construct(CurrencyService $currencyService)
    {
        $this->currencyService = $currencyService;
    }

    /**
     * Display available characters (shop and owned).
     */
    public function index()
    {
        $user = Auth::user()->load('character');
        
        // Get all active characters
        $allCharacters = Character::where('is_active', true)
            ->orderBy('sort_order')
            ->orderBy('unlock_level')
            ->get();
        
        // Get owned character IDs
        $ownedCharacterIds = $user->characters()->pluck('characters.id')->toArray();
        
        // Separate characters
        $ownedCharacters = $allCharacters->whereIn('id', $ownedCharacterIds);
        $shopCharacters = $allCharacters->whereNotIn('id', $ownedCharacterIds)
            ->filter(function($character) use ($user) {
                // Show if purchasable or unlockable by level
                return $character->isPurchasable() || 
                       ($character->unlock_level && $user->level >= $character->unlock_level);
            });
        
        $lockedCharacters = $allCharacters->whereNotIn('id', $ownedCharacterIds)
            ->filter(function($character) use ($user) {
                return !$character->isPurchasable() && 
                       ($character->unlock_level && $user->level < $character->unlock_level);
            });
        
        return view('characters.index', compact('ownedCharacters', 'shopCharacters', 'lockedCharacters', 'user'));
    }

    /**
     * Purchase a character.
     */
    public function purchase(Character $character)
    {
        $user = Auth::user();
        
        // Check if already owned
        if ($user->characters()->where('character_id', $character->id)->exists()) {
            return redirect()->route('characters.index')
                ->with('error', 'You already own this character!');
        }
        
        // Check if purchasable
        if (!$character->isPurchasable()) {
            return redirect()->route('characters.index')
                ->with('error', 'This character is not available for purchase.');
        }
        
        // Check VIP requirement
        if ($character->is_vip_only && !$user->isVip()) {
            return redirect()->route('characters.index')
                ->with('error', 'This character requires VIP membership.');
        }
        
        // Determine payment method and check funds
        $paymentMethod = null;
        if ($character->cost_premium && $character->cost_premium > 0) {
            if ($user->premium_coins < $character->cost_premium) {
                return redirect()->route('characters.index')
                    ->with('error', 'Insufficient Premium Coins.');
            }
            $paymentMethod = 'premium';
        } elseif ($character->cost_gold && $character->cost_gold > 0) {
            if ($user->gold < $character->cost_gold) {
                return redirect()->route('characters.index')
                    ->with('error', 'Insufficient Sacred Gold.');
            }
            $paymentMethod = 'gold';
        }
        
        if (!$paymentMethod) {
            return redirect()->route('characters.index')
                ->with('error', 'Character cannot be purchased.');
        }
        
        // Process purchase
        DB::transaction(function() use ($user, $character, $paymentMethod) {
            if ($paymentMethod === 'premium') {
                $this->currencyService->deductPremiumCoins($user, $character->cost_premium);
            } else {
                $this->currencyService->deductGold($user, $character->cost_gold);
            }
            
            // Add character to user's collection
            $user->characters()->attach($character->id, [
                'unlocked_at' => now(),
                'unlock_method' => 'purchase_' . $paymentMethod,
            ]);
        });
        
        return redirect()->route('characters.index')
            ->with('success', "You have unlocked {$character->name}!");
    }

    /**
     * Equip a character.
     */
    public function equip(Character $character)
    {
        $user = Auth::user();
        
        // Check if user owns this character
        if (!$user->characters()->where('character_id', $character->id)->exists()) {
            return redirect()->route('characters.index')
                ->with('error', 'You do not own this character.');
        }
        
        $user->character_id = $character->id;
        $user->save();
        
        return redirect()->route('characters.index')
            ->with('success', "Equipped {$character->name}!");
    }

    /**
     * Unequip current character.
     */
    public function unequip()
    {
        $user = Auth::user();
        $user->character_id = null;
        $user->save();
        
        return redirect()->route('characters.index')
            ->with('success', 'Character unequipped.');
    }

    /**
     * Test page for character animations.
     */
    public function testAnimations()
    {
        $characters = Character::where('is_active', true)
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get()
            ->filter(function($character) {
                // Only include characters that have at least one PNG sequence animation
                $states = $character->getPngSequenceStates('right');
                return count($states) > 0;
            })
            ->map(function($character) {
                return [
                    'character' => $character,
                    'states' => $character->getAllAnimationStates(),
                ];
            });
        
        $allStates = config('character_animations.states', []);
        
        return view('characters.test-animations', compact('characters', 'allStates'));
    }

    /**
     * Test page for fight animations (character vs mob).
     */
    public function testFight()
    {
        $user = Auth::user();
        
        $characters = Character::where('is_active', true)
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get()
            ->filter(function($character) {
                // Only include characters that have at least one PNG sequence animation
                $states = $character->getPngSequenceStates('right');
                return count($states) > 0;
            });
        
        $mobs = \App\Models\Mob::orderBy('level')
            ->orderBy('name')
            ->get()
            ->filter(function($mob) {
                // Only include mobs that have at least one PNG sequence animation
                $states = $mob->getPngSequenceStates('left');
                return count($states) > 0;
            });
        
        // Get available backgrounds for user
        $allBackgrounds = \App\Models\Arena3Background::where('is_active', true)
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();
        
        // Get user's owned backgrounds
        $ownedBackgroundIds = $user->arena3Backgrounds()->pluck('arena3_background_id')->toArray();
        
        // Get default background if user has none
        $defaultBackground = \App\Models\Arena3Background::where('is_default', true)
            ->where('is_active', true)
            ->first();
        
        // If user has no backgrounds and there's a default, give it to them
        if (empty($ownedBackgroundIds) && $defaultBackground) {
            $user->arena3Backgrounds()->attach($defaultBackground->id, [
                'is_selected' => true,
                'unlocked_at' => now(),
                'unlock_method' => 'default'
            ]);
            $ownedBackgroundIds = [$defaultBackground->id];
        }
        
        // Get user's selected background
        $selectedBackground = $user->arena3Backgrounds()
            ->wherePivot('is_selected', true)
            ->first();
        
        // If no selected background, use default or first owned
        if (!$selectedBackground) {
            if ($defaultBackground && in_array($defaultBackground->id, $ownedBackgroundIds)) {
                $selectedBackground = $defaultBackground;
            } else {
                $selectedBackground = \App\Models\Arena3Background::whereIn('id', $ownedBackgroundIds)
                    ->where('is_active', true)
                    ->first();
            }
        }
        
        $allStates = config('character_animations.states', []);
        
        return view('characters.test-fight', compact('characters', 'mobs', 'allStates', 'allBackgrounds', 'ownedBackgroundIds', 'selectedBackground'));
    }

    /**
     * Arena fight page with immersive game experience.
     */
    public function arenaFight(Request $request)
    {
        $characters = Character::where('is_active', true)
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get()
            ->filter(function($character) {
                $states = $character->getPngSequenceStates('right');
                return count($states) > 0;
            });
        
        $mobs = \App\Models\Mob::orderBy('level')
            ->orderBy('name')
            ->get()
            ->filter(function($mob) {
                $states = $mob->getPngSequenceStates('left');
                return count($states) > 0;
            });
        
        // Pre-encode data for JavaScript
        $charactersData = [];
        foreach ($characters as $char) {
            $charactersData[$char->slug] = [
                'slug' => $char->slug,
                'name' => $char->name,
                'states' => $char->getAllAnimationStates(),
                'attack_power' => $char->attack ?? 50
            ];
        }
        
        $mobsData = [];
        foreach ($mobs as $mobItem) {
            $mobsData[$mobItem->slug] = [
                'slug' => $mobItem->slug,
                'name' => $mobItem->name,
                'states' => [
                    'png' => $mobItem->getPngSequenceStates('left')
                ],
                'attack_power' => $mobItem->attack ?? 40
            ];
        }
        
        return view('characters.arena-fight', compact('characters', 'mobs', 'charactersData', 'mobsData'));
    }

    /**
     * Award resources when character wins arena fight.
     */
    public function arenaVictory(Request $request)
    {
        $user = Auth::user();
        
        // Award resource increases
        $attackIncrease = 1;
        $defenseIncrease = 1;
        $staminaIncrease = 1;
        
        $user->attack = ($user->attack ?? 12) + $attackIncrease;
        $user->defense = ($user->defense ?? 5) + $defenseIncrease;
        $user->stamina = ($user->stamina ?? 100) + $staminaIncrease;
        $user->save();
        
        return response()->json([
            'success' => true,
            'rewards' => [
                'attack' => $attackIncrease,
                'defense' => $defenseIncrease,
                'stamina' => $staminaIncrease,
            ],
            'new_stats' => [
                'attack' => $user->attack,
                'defense' => $user->defense,
                'stamina' => $user->stamina,
            ],
        ]);
    }

    /**
     * Serve the Fight-Game-master game (Arena 2) with integrated characters and mobs.
     */
    public function arenaFight2(Request $request)
    {
        $characterSlug = $request->get('character', '');
        $mobSlug = $request->get('mob', '');
        
        // If no character/mob selected, redirect to test-fight page
        if (empty($characterSlug) || empty($mobSlug)) {
            return redirect()->route('characters.test-fight')
                ->with('error', 'Please select both a character and a mob to enter Arena 2.');
        }
        
        $character = Character::where('slug', $characterSlug)->where('is_active', true)->first();
        $mob = \App\Models\Mob::where('slug', $mobSlug)->first();
        
        if (!$character || !$mob) {
            return redirect()->route('characters.test-fight')
                ->with('error', 'Character or mob not found.');
        }
        
        // Get character animation data
        $characterData = [
            'slug' => $character->slug,
            'name' => $character->name,
            'health' => 100, // Base health, can be adjusted based on character stats
            'attack' => $character->attack_bonus ?? 0,
            'defense' => $character->defense_bonus ?? 0,
            'sprites' => $this->getCharacterSpritesForGame($character),
        ];
        
        // Get mob animation data
        $mobData = [
            'slug' => $mob->slug,
            'name' => $mob->name,
            'health' => $mob->hp ?? 100,
            'attack' => $mob->attack ?? 40,
            'defense' => $mob->defense ?? 20,
            'sprites' => $this->getMobSpritesForGame($mob),
        ];
        
        $gamePath = public_path('Fight-Game-master/index.html');
        
        if (!file_exists($gamePath)) {
            abort(404, 'Game not found');
        }
        
        $html = file_get_contents($gamePath);
        
        // Add base tag to make all relative paths resolve correctly
        $basePath = rtrim(asset('Fight-Game-master'), '/');
        $baseTag = '<base href="' . $basePath . '/">';
        
        // Add Quest of Ra fonts and CSS
        $questOfRaFonts = '<link rel="preconnect" href="https://fonts.googleapis.com">
        <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
        <link href="https://fonts.googleapis.com/css2?family=Cinzel+Decorative:wght@400;700&family=Noto+Serif:wght@400;600;700&family=Orbitron:wght@400;600;700&display=swap" rel="stylesheet">';
        
        $fontStyle = '<style>
            * {
                font-family: "Cinzel Decorative", "Noto Serif", serif !important;
            }
            body {
                font-family: "Cinzel Decorative", "Noto Serif", serif !important;
            }
            html {
                font-family: "Cinzel Decorative", "Noto Serif", serif !important;
            }
            #timer, #countdownNumber, #gameStatus {
                font-family: "Orbitron", sans-serif !important;
            }
        </style>';
        
        // Insert base tag, fonts, and style right after <head>
        $html = str_replace('<head>', '<head>' . "\n    " . $baseTag . "\n    " . $questOfRaFonts . "\n    " . $fontStyle, $html);
        
        // Remove the Press Start 2P font link
        $html = str_replace('<link href="https://fonts.googleapis.com/css2?family=Press+Start+2P&display=swap" rel="stylesheet">', '', $html);
        
        // Also update script src paths to use absolute paths
        $html = str_replace('src="index.js"', 'src="' . $basePath . '/index.js"', $html);
        $html = str_replace('src="js/', 'src="' . $basePath . '/js/', $html);
        $html = str_replace('href="index.css"', 'href="' . $basePath . '/index.css"', $html);
        
        // Inject character and mob data as JSON before closing </head>
        // Set useDynamic = true to enable dynamic character/mob sprites
        // Set useDynamic = false or omit to use original hardcoded images
        $gameData = json_encode([
            'useDynamic' => false, // Set to true to use character/mob sprites, false for original images
            'character' => $characterData,
            'mob' => $mobData,
        ]);
        $dataScript = '<script>window.GAME_DATA = ' . $gameData . ';</script>';
        $html = str_replace('</head>', $dataScript . "\n</head>", $html);
        
        // Extract body content from the HTML
        preg_match('/<body[^>]*>(.*?)<\/body>/is', $html, $bodyMatches);
        $bodyContent = $bodyMatches[1] ?? $html;
        
        // Extract head content (scripts, styles, etc.)
        preg_match('/<head[^>]*>(.*?)<\/head>/is', $html, $headMatches);
        $headContent = $headMatches[1] ?? '';
        
        // Return the HTML to be embedded in the Blade view
        return view('characters.arena-fight-2', [
            'gameBody' => $bodyContent,
            'gameHead' => $headContent,
            'character' => $character,
            'mob' => $mob,
        ]);
    }

    /**
     * Serve the Arena 3 game (arean 3) with integrated characters and mobs.
     */
    public function arenaFight3(Request $request)
    {
        $user = Auth::user();
        $characterSlug = $request->get('character', '');
        $mobSlug = $request->get('mob', '');
        $backgroundId = $request->get('background', null);
        // Cast to integer if not null
        if ($backgroundId !== null) {
            $backgroundId = (int) $backgroundId;
        }
        $initialState = $request->get('state', 'idle'); // Get initial character state, default to 'idle'
        $initialMobState = $request->get('mobState', 'idle'); // Get initial mob state, default to 'idle'
        
        // Get available characters and mobs for selection
        $characters = Character::where('is_active', true)
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get()
            ->filter(function($character) {
                // Only include characters that have at least one PNG sequence animation
                $states = $character->getPngSequenceStates('right');
                return count($states) > 0;
            });
        
        $mobs = \App\Models\Mob::orderBy('level')
            ->orderBy('name')
            ->get()
            ->filter(function($mob) {
                // Only include mobs that have at least one PNG sequence animation
                $states = $mob->getPngSequenceStates('left');
                return count($states) > 0;
            });
        
        // Get available backgrounds for user
        $allBackgrounds = \App\Models\Arena3Background::where('is_active', true)
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();
        
        // Get user's owned backgrounds
        $ownedBackgroundIds = $user->arena3Backgrounds()->pluck('arena3_background_id')->toArray();
        
        // Get default background if user has none
        $defaultBackground = \App\Models\Arena3Background::where('is_default', true)
            ->where('is_active', true)
            ->first();
        
        // If user has no backgrounds and there's a default, give it to them
        if (empty($ownedBackgroundIds) && $defaultBackground) {
            $user->arena3Backgrounds()->attach($defaultBackground->id, [
                'is_selected' => true,
                'unlocked_at' => now(),
                'unlock_method' => 'default'
            ]);
            $ownedBackgroundIds = [$defaultBackground->id];
        }
        
        // Get user's selected background
        $selectedBackground = $user->arena3Backgrounds()
            ->wherePivot('is_selected', true)
            ->first();
        
        // If no selected background, use default or first owned
        if (!$selectedBackground) {
            if ($defaultBackground && in_array($defaultBackground->id, $ownedBackgroundIds)) {
                $selectedBackground = $defaultBackground;
            } else {
                $selectedBackground = \App\Models\Arena3Background::whereIn('id', $ownedBackgroundIds)
                    ->where('is_active', true)
                    ->first();
            }
        }
        
        // If no character/mob selected, show selection UI
        if (empty($characterSlug) || empty($mobSlug)) {
            return view('characters.arena-3', [
                'character' => null,
                'mob' => null,
                'characters' => $characters,
                'mobs' => $mobs,
                'allBackgrounds' => $allBackgrounds,
                'ownedBackgroundIds' => $ownedBackgroundIds,
                'selectedBackground' => $selectedBackground,
                'showSelection' => true,
            ]);
        }
        
        $character = Character::where('slug', $characterSlug)->where('is_active', true)->first();
        $mob = \App\Models\Mob::where('slug', $mobSlug)->first();
        
        if (!$character || !$mob) {
            // If character/mob not found, show selection UI with error
            return view('characters.arena-3', [
                'character' => null,
                'mob' => null,
                'characters' => $characters,
                'mobs' => $mobs,
                'allBackgrounds' => $allBackgrounds,
                'ownedBackgroundIds' => $ownedBackgroundIds,
                'selectedBackground' => $selectedBackground,
                'showSelection' => true,
                'error' => 'Character or mob not found.',
            ]);
        }
        
        // Get selected background or default
        $background = null;
        if ($backgroundId) {
            // Verify user owns this background
            $background = $user->arena3Backgrounds()
                ->where('arena3_backgrounds.id', $backgroundId)
                ->where('is_active', true)
                ->first();
            
            \Log::info('Arena 3 Background Selection', [
                'requested_background_id' => $backgroundId,
                'found_background' => $background ? $background->id : null,
                'background_name' => $background ? $background->name : null,
            ]);
        }
        
        // If no background selected or user doesn't own it, use default or first owned
        if (!$background) {
            $background = $user->arena3Backgrounds()
                ->wherePivot('is_selected', true)
                ->where('is_active', true)
                ->first();
            
            if (!$background) {
                $background = \App\Models\Arena3Background::where('is_default', true)
                    ->where('is_active', true)
                    ->first();
            }
        }
        
        // Fallback to default path if no background found
        $backgroundPath = $background && $background->image_path 
            ? asset('storage/' . $background->image_path)
            : asset('arean 3/backgrounds/Background1.png');
        
        \Log::info('Arena 3 Background Path', [
            'background_id' => $background ? $background->id : null,
            'image_path' => $background ? $background->image_path : null,
            'final_background_path' => $backgroundPath,
        ]);
        
        $shopPath = $background && $background->shop_image_path
            ? asset('storage/' . $background->shop_image_path)
            : asset('arean 3/backgrounds/shop_anim.png');
        
        // Get character animation paths
        $characterSprites = $this->getCharacterSpritesForArena3($character);
        
        // Get mob animation paths
        $mobSprites = $this->getMobSpritesForArena3($mob);
        
        // Calculate character stats (base + bonuses)
        // Use character-specific base stats if set, otherwise use defaults
        $baseHP = $character->base_hp ?? 300; // Default 300 HP for balanced fights
        $baseAttack = $character->base_attack ?? 45; // Default 45 attack
        $baseDefense = $character->base_defense ?? 25; // Default 25 defense
        
        // Special case: farmer-peasant gets 100 power if no base_attack is set
        if ($character->slug === 'farmer-peasant' && !$character->base_attack) {
            $baseAttack = 100;
        }
        
        // Final stats = base + bonuses
        $hpBonus = $character->hp_bonus ?? 0;
        $attackBonus = $character->attack_bonus ?? 0;
        $defenseBonus = $character->defense_bonus ?? 0;
        
        $characterStats = [
            'hp' => $baseHP + $hpBonus,
            'attack' => $baseAttack + $attackBonus,
            'defense' => $baseDefense + $defenseBonus,
        ];
        
        // Log character stats for verification
        \Log::info('Arena 3 - Character Stats', [
            'character_id' => $character->id,
            'character_name' => $character->name,
            'character_slug' => $character->slug,
            'base_hp' => $baseHP,
            'hp_bonus' => $hpBonus,
            'final_hp' => $characterStats['hp'],
            'base_attack' => $baseAttack,
            'attack_bonus' => $attackBonus,
            'final_attack' => $characterStats['attack'],
            'base_defense' => $baseDefense,
            'defense_bonus' => $defenseBonus,
            'final_defense' => $characterStats['defense'],
        ]);
        
        // Get mob stats from database - use actual values set in admin panel
        // Balance mob stats to be within 10% of character stats for fair fights
        $originalMobHP = $mob->hp ?? 300;
        $originalMobAttack = $mob->attack ?? 45;
        $originalMobDefense = $mob->defense ?? 25;
        
        $mobHP = $originalMobHP;
        $mobAttack = $originalMobAttack;
        $mobDefense = $originalMobDefense;
        
        // Balance mob stats to be within 10% of character stats for fair fights
        // HP: Mob should have 90-110% of character HP
        $hpRatio = $mobHP / max($characterStats['hp'], 1);
        $hpWasBalanced = false;
        if ($hpRatio < 0.9) {
            $mobHP = (int)($characterStats['hp'] * 0.9); // At least 90% of character HP
            $hpWasBalanced = true;
        } elseif ($hpRatio > 1.1) {
            $mobHP = (int)($characterStats['hp'] * 1.1); // At most 110% of character HP
            $hpWasBalanced = true;
        }
        
        // Attack: Mob should have 90-110% of character attack
        $attackRatio = $mobAttack / max($characterStats['attack'], 1);
        $attackWasBalanced = false;
        if ($attackRatio < 0.9) {
            $mobAttack = (int)($characterStats['attack'] * 0.9); // At least 90% of character attack
            $attackWasBalanced = true;
        } elseif ($attackRatio > 1.1) {
            $mobAttack = (int)($characterStats['attack'] * 1.1); // At most 110% of character attack
            $attackWasBalanced = true;
        }
        
        // Defense: Mob should have 90-110% of character defense
        $defenseRatio = $mobDefense / max($characterStats['defense'], 1);
        $defenseWasBalanced = false;
        if ($defenseRatio < 0.9) {
            $mobDefense = (int)($characterStats['defense'] * 0.9); // At least 90% of character defense
            $defenseWasBalanced = true;
        } elseif ($defenseRatio > 1.1) {
            $mobDefense = (int)($characterStats['defense'] * 1.1); // At most 110% of character defense
            $defenseWasBalanced = true;
        }
        
        $mobStats = [
            'hp' => $mobHP,
            'attack' => $mobAttack,
            'defense' => $mobDefense,
        ];
        
        // Log mob stats for verification
        \Log::info('Arena 3 - Mob Stats', [
            'mob_id' => $mob->id,
            'mob_name' => $mob->name,
            'mob_slug' => $mob->slug,
            'original_hp' => $originalMobHP,
            'final_hp' => $mobHP,
            'hp_was_balanced' => $hpWasBalanced,
            'hp_ratio' => round($hpRatio, 2),
            'original_attack' => $originalMobAttack,
            'final_attack' => $mobAttack,
            'attack_was_balanced' => $attackWasBalanced,
            'attack_ratio' => round($attackRatio, 2),
            'original_defense' => $originalMobDefense,
            'final_defense' => $mobDefense,
            'defense_was_balanced' => $defenseWasBalanced,
            'defense_ratio' => round($defenseRatio, 2),
        ]);
        
        // Get state modifiers for display (use character/mob-specific if available)
        $charStateMod = $character->getStateModifier($initialState ?? 'idle');
        $mobStateMod = $mob->getStateModifier($initialMobState ?? 'idle');
        
        return view('characters.arena-3', [
            'character' => $character,
            'mob' => $mob,
            'characterSprites' => $characterSprites,
            'mobSprites' => $mobSprites,
            'characterStats' => $characterStats,
            'mobStats' => $mobStats,
            'initialState' => $initialState ?? 'idle',
            'initialMobState' => $initialMobState ?? 'idle',
            'characterStateMod' => $charStateMod,
            'mobStateMod' => $mobStateMod,
            'backgroundPath' => $backgroundPath ?? asset('arean 3/backgrounds/Background1.png'),
            'shopPath' => $shopPath ?? asset('arean 3/backgrounds/shop_anim.png'),
            'showSelection' => false,
            'characters' => $characters,
            'mobs' => $mobs,
            'allBackgrounds' => $allBackgrounds,
            'ownedBackgroundIds' => $ownedBackgroundIds,
            'selectedBackground' => $selectedBackground,
        ]);
    }

    /**
     * Get character sprites formatted for Arena 3 game.
     */
    private function getCharacterSpritesForArena3(Character $character): array
    {
        // Map game states to character animation states
        $stateMap = [
            'idle' => 'idle',
            'run' => 'idle',
            'jump' => 'idle',
            'fall' => 'idle',
            'attack1' => 'attack', // Maps to 'fight' folder via getPngSequence
            'takeHit' => 'hurt', // Maps to 'hurt' folder
            'death' => 'lose',
            'win' => 'win', // Maps to 'win' folder
        ];
        
        $sprites = [];
        foreach ($stateMap as $gameState => $charState) {
            $frames = $character->getPngSequence($charState, 'right');
            if (empty($frames) && $gameState !== 'idle') {
                $frames = $character->getPngSequence('idle', 'right');
            }
            
            if (!empty($frames)) {
                // For death animation, limit to first 5 frames (01.png to 05.png)
                if ($gameState === 'death') {
                    $frames = array_slice($frames, 0, 5);
                }
                
                // getPngSequence already returns web paths like /assets/characters/{slug}/...
                $sprites[$gameState] = [
                    'images' => $frames,
                    'framesMax' => count($frames),
                ];
            }
        }
        
        return $sprites;
    }
    
    /**
     * Get mob sprites formatted for Arena 3 game.
     */
    private function getMobSpritesForArena3(\App\Models\Mob $mob): array
    {
        // Map game states to mob animation states
        $stateMap = [
            'idle' => 'idle',
            'run' => 'idle',
            'jump' => 'idle',
            'fall' => 'idle',
            'attack1' => 'attack',
            'takeHit' => 'idle', // Fallback to idle
            'death' => 'lose',
        ];
        
        $sprites = [];
        foreach ($stateMap as $gameState => $mobState) {
            $frames = $mob->getPngSequence($mobState, 'left');
            if (empty($frames) && $gameState !== 'idle') {
                $frames = $mob->getPngSequence('idle', 'left');
            }
            
            if (!empty($frames)) {
                // getPngSequence already returns web paths like /assets/mobs/{slug}/...
                $sprites[$gameState] = [
                    'images' => $frames,
                    'framesMax' => count($frames),
                ];
            }
        }
        
        return $sprites;
    }
    
    /**
     * Get character sprites formatted for the game.
     */
    private function getCharacterSpritesForGame(Character $character): array
    {
        // Map game states to character animation states
        // Note: 'attack' in character system maps to 'fight' folder
        $stateMap = [
            'idle' => 'idle',
            'run' => 'idle', // Fallback to idle if run doesn't exist
            'jump' => 'idle', // Fallback to idle
            'fall' => 'idle', // Fallback to idle
            'attack1' => 'attack', // Maps to 'fight' folder in character system
            'takeHit' => 'idle', // Fallback, could check for 'hurt' state if it exists
            'death' => 'lose',
        ];
        
        $sprites = [];
        foreach ($stateMap as $gameState => $charState) {
            $frames = $character->getPngSequence($charState, 'right');
            if (empty($frames) && $gameState !== 'idle') {
                // Fallback to idle if state doesn't exist
                $frames = $character->getPngSequence('idle', 'right');
            }
            
            if (!empty($frames)) {
                $sprites[$gameState] = [
                    'frames' => $frames,
                    'framesMax' => count($frames),
                    'isSequence' => true, // Flag to indicate PNG sequence vs sprite sheet
                ];
            }
        }
        
        return $sprites;
    }
    
    /**
     * Get mob sprites formatted for the game.
     */
    private function getMobSpritesForGame(\App\Models\Mob $mob): array
    {
        // Map game states to mob animation states
        $stateMap = [
            'idle' => 'idle',
            'run' => 'idle', // Fallback to idle
            'jump' => 'idle', // Fallback to idle
            'fall' => 'idle', // Fallback to idle
            'attack1' => 'attack', // Maps to 'attack' folder in mob system
            'takeHit' => 'idle', // Fallback, could check for 'hurt' state if it exists
            'death' => 'lose',
        ];
        
        $sprites = [];
        foreach ($stateMap as $gameState => $mobState) {
            $frames = $mob->getPngSequence($mobState, 'left');
            if (empty($frames) && $gameState !== 'idle') {
                // Fallback to idle if state doesn't exist
                $frames = $mob->getPngSequence('idle', 'left');
            }
            
            if (!empty($frames)) {
                $sprites[$gameState] = [
                    'frames' => $frames,
                    'framesMax' => count($frames),
                    'isSequence' => true, // Flag to indicate PNG sequence vs sprite sheet
                ];
            }
        }
        
        return $sprites;
    }
}
