<?php

namespace App\Http\Controllers;

use App\Models\CoinPack;
use App\Models\Transaction;
use App\Services\PayPalPaymentService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CoinsController extends Controller
{
    protected PayPalPaymentService $paypalService;

    public function __construct(PayPalPaymentService $paypalService)
    {
        $this->paypalService = $paypalService;
    }

    /**
     * Display available coin packs.
     */
    public function index()
    {
        $coinPacks = CoinPack::active()->orderBy('price_usd')->get();
        $user = Auth::user();

        return view('coins.index', compact('coinPacks', 'user'));
    }

    /**
     * Create PayPal payment for coin pack purchase.
     */
    public function buy(Request $request)
    {
        $validated = $request->validate([
            'pack_id' => ['required', 'exists:coin_packs,id'],
        ]);

        $user = Auth::user();
        $pack = CoinPack::findOrFail($validated['pack_id']);

        if (!$pack->is_active) {
            return redirect()->back()
                ->with('error', 'This coin pack is not available.');
        }

        // Create PayPal payment
        $result = $this->paypalService->createCoinPackPayment($user, $pack);

        if (!$result['success']) {
            return redirect()->back()
                ->with('error', $result['error'] ?? 'Failed to create payment. Please try again.');
        }

        // Redirect to PayPal checkout
        return redirect($result['approval_url']);
    }

    /**
     * Handle successful coin pack payment return from PayPal.
     */
    public function success(Request $request)
    {
        $orderId = $request->query('token') ?? $request->query('paymentId');
        $payerId = $request->query('PayerID');
        $transactionId = $request->query('transaction_id');

        if (!$payerId || !$transactionId) {
            return redirect()->route('coins.index')
                ->with('error', 'Invalid payment return parameters.');
        }

        // Get transaction to find order ID
        $transaction = Transaction::find($transactionId);
        if (!$transaction || !$transaction->paypal_order_id) {
            return redirect()->route('coins.index')
                ->with('error', 'Transaction not found.');
        }

        $orderId = $transaction->paypal_order_id;

        // Capture payment
        $result = $this->paypalService->capturePayment($orderId, $payerId);

        if (!$result['success']) {
            return redirect()->route('coins.index')
                ->with('error', 'Payment capture failed. Please contact support.');
        }

        // Find transaction
        $transaction = Transaction::find($transactionId);

        if ($transaction) {
            $user = $transaction->user;
            $this->paypalService->handleSuccessfulCoinsPayment($user, $transaction);
        }

        return redirect()->route('coins.index')
            ->with('success', 'Premium coins added to your account!');
    }

    /**
     * Handle cancelled coin pack payment.
     */
    public function cancel(Request $request)
    {
        $transactionId = $request->query('transaction_id');

        if ($transactionId) {
            $transaction = Transaction::find($transactionId);
            if ($transaction) {
                $transaction->status = Transaction::STATUS_CANCELLED;
                $transaction->save();
            }
        }

        return redirect()->route('coins.index')
            ->with('error', 'Payment was cancelled.');
    }
}
