<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Services\GameService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class DashboardController extends Controller
{
    protected GameService $gameService;

    public function __construct(GameService $gameService)
    {
        $this->gameService = $gameService;
    }

    /**
     * Display the user dashboard.
     */
    public function index()
    {
        $user = Auth::user()->load('character');
        
        // Regenerate HP if needed (for offline time)
        $this->regenerateHPIfNeeded($user);
        
        // Calculate XP progress
        $currentLevelXP = $this->gameService->xpNeededForLevel($user->level);
        $nextLevelXP = $this->gameService->xpNeededForLevel($user->level + 1);
        $xpInCurrentLevel = $user->xp - $currentLevelXP;
        $xpNeededForNextLevel = $nextLevelXP - $currentLevelXP;
        $xpProgress = $xpNeededForNextLevel > 0 ? ($xpInCurrentLevel / $xpNeededForNextLevel) * 100 : 100;

        // Get user stats
        $totalStats = $this->gameService->getUserTotalStats($user);
        
        // Get recent tasks
        $recentTasks = $user->tasks()
            ->where('status', 'completed')
            ->orderBy('completed_at', 'desc')
            ->limit(5)
            ->get();

        // Get achievements
        $achievements = $user->achievements()->orderBy('unlocked_at', 'desc')->get();
        
        // Get tasks stats
        $tasksCompletedToday = $user->tasks()
            ->where('status', 'completed')
            ->whereDate('completed_at', today())
            ->count();
            
        $tasksCompletedThisWeek = $user->tasks()
            ->where('status', 'completed')
            ->whereBetween('completed_at', [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()])
            ->count();

        $tasksCompletedThisMonth = $user->tasks()
            ->where('status', 'completed')
            ->whereBetween('completed_at', [Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()])
            ->count();

        // Leaderboard (top 10 by level, then XP)
        $leaderboard = User::orderBy('level', 'desc')
            ->orderBy('xp', 'desc')
            ->limit(10)
            ->get();

        // Calculate daily quest streak (Blessings Meter)
        $streakData = $this->gameService->calculateDailyQuestStreak($user);

        return view('dashboard', compact(
            'user',
            'xpProgress',
            'currentLevelXP',
            'nextLevelXP',
            'xpInCurrentLevel',
            'xpNeededForNextLevel',
            'totalStats',
            'recentTasks',
            'achievements',
            'tasksCompletedToday',
            'tasksCompletedThisWeek',
            'tasksCompletedThisMonth',
            'leaderboard',
            'streakData'
        ));
    }

    /**
     * Regenerate HP for offline time if needed.
     */
    protected function regenerateHPIfNeeded(User $user): void
    {
        $gameService = app(\App\Services\GameService::class);
        $totalStats = $gameService->getUserTotalStats($user);
        $maxHP = $totalStats['hp'];
        
        $currentHP = $user->current_hp ?? 0;
        
        // If already at max HP, no need to regenerate
        if ($currentHP >= $maxHP) {
            return;
        }
        
        // Calculate minutes since last regeneration
        $minutesSinceRegen = 0;
        if ($user->last_hp_regen_at) {
            $minutesSinceRegen = now()->diffInMinutes($user->last_hp_regen_at);
        } else {
            // If never regenerated, assume some time has passed
            $minutesSinceRegen = 1;
        }
        
        // Regenerate HP for all minutes passed (1 HP per minute)
        // Cap at max HP
        if ($minutesSinceRegen > 0) {
            $hpToRegenerate = min($minutesSinceRegen, $maxHP - $currentHP);
            $newHP = $currentHP + $hpToRegenerate;
            
            if ($newHP > $currentHP) {
                $user->current_hp = $newHP;
                $user->last_hp_regen_at = now();
                $user->save();
            }
        }
    }
}
