<?php

namespace App\Http\Controllers;

use App\Models\Membership;
use App\Models\Transaction;
use App\Services\VipService;
use App\Services\PayPalPaymentService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class MembershipController extends Controller
{
    protected VipService $vipService;
    protected PayPalPaymentService $paypalService;

    public function __construct(VipService $vipService, PayPalPaymentService $paypalService)
    {
        $this->vipService = $vipService;
        $this->paypalService = $paypalService;
    }

    /**
     * Display all membership plans.
     */
    public function plans()
    {
        $memberships = Membership::all();
        $user = Auth::user();

        return view('membership.plans', compact('memberships', 'user'));
    }

    /**
     * Subscribe to a membership via PayPal.
     */
    public function subscribe(Request $request)
    {
        $validated = $request->validate([
            'membership_id' => ['required', 'exists:memberships,id'],
            'period' => ['required', 'in:monthly,yearly'],
        ]);

        $user = Auth::user();
        $membership = Membership::findOrFail($validated['membership_id']);

        // Create PayPal payment
        $result = $this->paypalService->createVipSubscriptionPayment($user, $membership, $validated['period']);

        if (!$result['success']) {
            return redirect()->back()
                ->with('error', $result['error'] ?? 'Failed to create payment. Please try again.');
        }

        // Redirect to PayPal checkout
        return redirect($result['approval_url']);
    }

    /**
     * Handle successful VIP payment return from PayPal.
     */
    public function vipSuccess(Request $request)
    {
        $orderId = $request->query('token') ?? $request->query('paymentId');
        $payerId = $request->query('PayerID');
        $transactionId = $request->query('transaction_id');

        if (!$payerId || !$transactionId) {
            return redirect()->route('membership.status')
                ->with('error', 'Invalid payment return parameters.');
        }

        // Get transaction to find order ID
        $transaction = Transaction::find($transactionId);
        if (!$transaction || !$transaction->paypal_order_id) {
            return redirect()->route('membership.status')
                ->with('error', 'Transaction not found.');
        }

        $orderId = $transaction->paypal_order_id;

        // Capture payment
        $result = $this->paypalService->capturePayment($orderId, $payerId);

        if (!$result['success']) {
            return redirect()->route('membership.status')
                ->with('error', 'Payment capture failed. Please contact support.');
        }

        // Find transaction
        $transaction = Transaction::find($transactionId);

        if ($transaction) {
            $user = $transaction->user;
            $this->paypalService->handleSuccessfulVipPayment($user, $transaction);
        }

        return redirect()->route('membership.status')
            ->with('success', 'VIP membership activated successfully!');
    }

    /**
     * Handle cancelled VIP payment.
     */
    public function vipCancel(Request $request)
    {
        $transactionId = $request->query('transaction_id');

        if ($transactionId) {
            $transaction = Transaction::find($transactionId);
            if ($transaction) {
                $transaction->status = Transaction::STATUS_CANCELLED;
                $transaction->save();
            }
        }

        return redirect()->route('membership.plans')
            ->with('error', 'Payment was cancelled.');
    }

    /**
     * Cancel VIP membership.
     */
    public function cancel()
    {
        $user = Auth::user();

        if (!$this->vipService->isVip($user)) {
            return redirect()->route('membership.status')
                ->with('error', 'You are not currently a VIP member.');
        }

        $this->vipService->deactivateVip($user);

        return redirect()->route('membership.status')
            ->with('success', 'VIP membership cancelled.');
    }

    /**
     * Show current membership status.
     */
    public function status()
    {
        $user = Auth::user()->load('membership');
        $isVip = $this->vipService->isVip($user);
        $membership = $isVip ? $this->vipService->getActiveMembership($user) : null;

        return view('membership.status', compact('user', 'isVip', 'membership'));
    }
}
