<?php

namespace App\Http\Controllers;

use App\Models\ScrollCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Http\Controllers\Controller;

class ScrollCategoryController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $scrollCategories = Auth::user()->scrollCategories()->orderBy('sort_order')->orderBy('created_at', 'desc')->get();
        return view('scroll-categories.index', compact('scrollCategories'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('scroll-categories.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => ['required', 'string', 'max:255'],
            'icon' => ['nullable', 'string', 'max:255'],
            'color' => ['nullable', 'string', 'max:7'],
            'description' => ['nullable', 'string'],
            'sort_order' => ['nullable', 'integer', 'min:0'],
        ]);

        $validated['user_id'] = Auth::id();
        $validated['is_active'] = true;
        $validated['sort_order'] = $validated['sort_order'] ?? 0;

        $scrollCategory = ScrollCategory::create($validated);

        return redirect()->route('scroll-categories.index')
            ->with('success', 'Scroll category created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(ScrollCategory $scrollCategory)
    {
        $this->authorize('view', $scrollCategory);
        $scrollCategory->load('tasks');
        return view('scroll-categories.show', compact('scrollCategory'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ScrollCategory $scrollCategory)
    {
        $this->authorize('update', $scrollCategory);
        return view('scroll-categories.edit', compact('scrollCategory'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, ScrollCategory $scrollCategory)
    {
        $this->authorize('update', $scrollCategory);

        $validated = $request->validate([
            'title' => ['required', 'string', 'max:255'],
            'icon' => ['nullable', 'string', 'max:255'],
            'color' => ['nullable', 'string', 'max:7'],
            'description' => ['nullable', 'string'],
            'sort_order' => ['nullable', 'integer', 'min:0'],
            'is_active' => ['nullable', 'boolean'],
        ]);

        $scrollCategory->update($validated);

        return redirect()->route('scroll-categories.index')
            ->with('success', 'Scroll category updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ScrollCategory $scrollCategory)
    {
        $this->authorize('delete', $scrollCategory);
        $scrollCategory->delete();

        return redirect()->route('scroll-categories.index')
            ->with('success', 'Scroll category deleted successfully.');
    }
}
