<?php

namespace App\Http\Controllers;

use App\Models\Item;
use App\Services\ShopService;
use App\Services\VipService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ShopController extends Controller
{
    protected ShopService $shopService;
    protected VipService $vipService;

    public function __construct(ShopService $shopService, VipService $vipService)
    {
        $this->shopService = $shopService;
        $this->vipService = $vipService;
    }

    /**
     * Display the shop.
     */
    public function index()
    {
        $user = Auth::user();
        
        // Get all items
        $allItems = Item::where('required_level', '<=', $user->level)
            ->orderBy('required_level')
            ->orderBy('cost')
            ->get();

        // Separate items by type
        $regularItems = $allItems->where('is_premium', false)->where('is_vip_only', false);
        $premiumItems = $allItems->where('is_premium', true);
        $vipItems = $allItems->where('is_vip_only', true);

        // Check which items user already owns
        $ownedItemIds = $user->userItems()->pluck('item_id')->toArray();

        $isVip = $this->vipService->isVip($user);

        return view('shop.index', compact('regularItems', 'premiumItems', 'vipItems', 'ownedItemIds', 'user', 'isVip'));
    }

    /**
     * Purchase an item (with gold or premium coins).
     */
    public function purchase(Item $item)
    {
        $user = Auth::user();

        // Check if premium item
        if ($item->is_premium && $item->premium_coins_cost) {
            return $this->buyPremiumItem($item);
        }

        // Regular purchase with gold
        $success = $this->shopService->buyWithGold($user, $item);

        if (!$success) {
            $canPurchase = $this->shopService->canPurchaseItem($user, $item);
            $reason = !empty($canPurchase['reasons']) ? implode(', ', $canPurchase['reasons']) : 'Unable to purchase item.';
            
            return redirect()->route('shop.index')
                ->with('error', $reason);
        }

        return redirect()->route('shop.index')
            ->with('success', "Purchased {$item->name} for {$item->cost} gold!");
    }

    /**
     * Purchase a premium item with premium coins.
     */
    public function buyPremiumItem(Item $item)
    {
        $user = Auth::user();

        $success = $this->shopService->buyWithPremiumCoins($user, $item);

        if (!$success) {
            $canPurchase = $this->shopService->canPurchaseItem($user, $item);
            $reason = !empty($canPurchase['reasons']) ? implode(', ', $canPurchase['reasons']) : 'Unable to purchase item.';
            
            return redirect()->route('shop.index')
                ->with('error', $reason);
        }

        return redirect()->route('shop.index')
            ->with('success', "Purchased {$item->name} for {$item->premium_coins_cost} premium coins!");
    }

    /**
     * List premium items (API endpoint if needed).
     */
    public function listPremiumItems()
    {
        $items = $this->shopService->getPremiumItems();
        return response()->json($items);
    }

    /**
     * List VIP items (API endpoint if needed).
     */
    public function listVipItems()
    {
        $items = $this->shopService->getVipExclusiveItems();
        return response()->json($items);
    }
}
