<?php

namespace App\Http\Controllers;

use App\Models\Task;
use App\Models\TaskChecklist;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class TaskChecklistController extends Controller
{
    /**
     * Store a newly created checklist item.
     */
    public function store(Request $request, Task $task)
    {
        $this->authorize('update', $task);

        $validated = $request->validate([
            'title' => ['required', 'string', 'max:255'],
        ]);

        $maxOrder = $task->checklists()->max('order') ?? 0;

        $task->checklists()->create([
            'title' => $validated['title'],
            'completed' => false,
            'order' => $maxOrder + 1,
        ]);

        return redirect()->back()->with('success', 'Checklist item added.');
    }

    /**
     * Update the specified checklist item.
     */
    public function update(Request $request, Task $task, TaskChecklist $checklist)
    {
        $this->authorize('update', $task);

        if ($checklist->task_id !== $task->id) {
            abort(404);
        }

        $validated = $request->validate([
            'title' => ['sometimes', 'required', 'string', 'max:255'],
            'completed' => ['sometimes', 'boolean'],
        ]);

        $checklist->update($validated);

        return redirect()->back()->with('success', 'Checklist item updated.');
    }

    /**
     * Toggle completion status of a checklist item.
     */
    public function toggle(Task $task, TaskChecklist $checklist)
    {
        $this->authorize('update', $task);

        if ($checklist->task_id !== $task->id) {
            abort(404);
        }

        $checklist->update(['completed' => !$checklist->completed]);

        return response()->json(['completed' => $checklist->completed]);
    }

    /**
     * Remove the specified checklist item.
     */
    public function destroy(Task $task, TaskChecklist $checklist)
    {
        $this->authorize('update', $task);

        if ($checklist->task_id !== $task->id) {
            abort(404);
        }

        $checklist->delete();

        return redirect()->back()->with('success', 'Checklist item deleted.');
    }
}
