<?php

namespace App\Http\Controllers;

use App\Models\Task;
use App\Models\Category;
use App\Services\TaskService;
use App\Services\ReminderService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class TaskController extends Controller
{
    protected TaskService $taskService;
    protected ReminderService $reminderService;

    public function __construct(TaskService $taskService, ReminderService $reminderService)
    {
        $this->taskService = $taskService;
        $this->reminderService = $reminderService;
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $viewMode = $request->get('view', session('quest_view_mode', 'cards'));
        session(['quest_view_mode' => $viewMode]);

        // Use Categories as Quest Scrolls
        $categoriesQuery = Auth::user()->categories()->orderBy('created_at', 'desc');

        // Filter by specific category/scroll if requested
        if ($request->has('scroll') && $request->scroll) {
            $categoriesQuery->where('id', $request->scroll);
        }

        $categories = $categoriesQuery->get();

        // Load tasks for each category (scroll) with filters - exclude completed by default
        foreach ($categories as $category) {
            $taskQuery = Task::where('user_id', Auth::id())
                ->where('category_id', $category->id)
                ->with(['category', 'reminders']);

            // Status filter - if not specified, exclude completed by default
            if ($request->has('status') && $request->status) {
                $taskQuery->where('status', $request->status);
            } else {
                // Exclude completed by default
                $taskQuery->where('status', '!=', 'completed');
            }

            // Difficulty filter
            if ($request->has('difficulty') && $request->difficulty) {
                $taskQuery->where('difficulty_level', $request->difficulty);
            }

            // Due date filter
            if ($request->has('due_date')) {
                if ($request->due_date === 'today') {
                    $taskQuery->whereDate('due_date', today());
                } elseif ($request->due_date === 'overdue') {
                    $taskQuery->where('due_date', '<', now())->where('status', '!=', 'completed');
                }
            }

            // Set filtered tasks
            $category->setRelation('tasks', $taskQuery->orderBy('due_date', 'asc')
                ->orderBy('created_at', 'desc')
                ->get());

            // Also load completed tasks count for stats
            $completedCount = Task::where('user_id', Auth::id())
                ->where('category_id', $category->id)
                ->where('status', 'completed')
                ->count();
            $category->completed_count = $completedCount;
        }

        // Also get tasks without category_id (uncategorized) - exclude completed by default
        $uncategorizedTasksQuery = Task::where('user_id', Auth::id())
            ->whereNull('category_id')
            ->with(['category', 'reminders']);

        // Apply same filters
        if ($request->has('status') && $request->status) {
            $uncategorizedTasksQuery->where('status', $request->status);
        } else {
            // Exclude completed by default
            $uncategorizedTasksQuery->where('status', '!=', 'completed');
        }
        if ($request->has('difficulty') && $request->difficulty) {
            $uncategorizedTasksQuery->where('difficulty_level', $request->difficulty);
        }
        if ($request->has('due_date')) {
            if ($request->due_date === 'today') {
                $uncategorizedTasksQuery->whereDate('due_date', today());
            } elseif ($request->due_date === 'overdue') {
                $uncategorizedTasksQuery->where('due_date', '<', now())->where('status', '!=', 'completed');
            }
        }

        $uncategorizedTasks = $uncategorizedTasksQuery->orderBy('due_date', 'asc')
            ->orderBy('created_at', 'desc')
            ->get();

        // Create a virtual category for uncategorized tasks if there are any
        if ($uncategorizedTasks->count() > 0) {
            $uncategorizedCategory = new Category([
                'id' => 0,
                'name' => 'Uncategorized Quests',
                'icon' => '📋',
                'color' => '#666666',
                'repeat_type' => 'one-time',
            ]);
            $uncategorizedCategory->setRelation('tasks', $uncategorizedTasks);
            $uncategorizedCategory->completed_count = Task::where('user_id', Auth::id())
                ->whereNull('category_id')
                ->where('status', 'completed')
                ->count();
            $categories->push($uncategorizedCategory);
        }

        return view('tasks.index', compact('categories', 'viewMode'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $categories = Auth::user()->categories()->get();
        $selectedCategoryId = $request->query('category_id');
        return view('tasks.create', compact('categories', 'selectedCategoryId'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'title' => ['required', 'string', 'max:255'],
                'description' => ['nullable', 'string'],
                'category_id' => ['required', 'exists:categories,id'],
                'due_date' => ['nullable'],
                'difficulty_level' => ['required', 'in:easy,medium,hard,expert'],
                'repeat_rule' => ['nullable', 'array'],
                'reminders' => ['nullable', 'array'],
                'reminders.*.enabled' => ['nullable'],
                'reminders.*.type' => ['nullable', 'in:green,yellow,red'],
                'reminders.*.schedule_type' => ['nullable', 'in:weekly,daily,custom'],
                'reminders.*.time' => ['nullable', 'date_format:H:i'],
                'reminders.*.days' => ['nullable', 'array'],
                'reminders.*.days_before' => ['nullable', 'integer'],
                'reminders.*.hours_before' => ['nullable', 'integer', 'min:0', 'max:48'],
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        }

        // Verify category belongs to user
        $category = Category::findOrFail($validated['category_id']);
        if ($category->user_id !== Auth::id()) {
            return redirect()->back()->withErrors(['category_id' => 'Invalid category.']);
        }


        // Get difficulty level configuration
        $difficultyConfig = config('difficulty_levels.levels.' . $validated['difficulty_level']);
        
        // Auto-calculate points and coins based on difficulty level
        $validated['points'] = $difficultyConfig['points'];
        $validated['coins'] = $difficultyConfig['coins'];

        // Handle due_date format conversion (datetime-local to datetime)
        if (!empty($validated['due_date'])) {
            try {
                $validated['due_date'] = \Carbon\Carbon::parse($validated['due_date'])->format('Y-m-d H:i:s');
            } catch (\Exception $e) {
                return redirect()->back()
                    ->withErrors(['due_date' => 'Invalid date format.'])
                    ->withInput();
            }
        }

        // Extract reminders before creating task
        $reminders = $validated['reminders'] ?? [];
        unset($validated['reminders']);

        try {
            $task = Auth::user()->tasks()->create($validated);

            // Create reminders if provided
            if (!empty($reminders)) {
                try {
                    $this->reminderService->createRemindersForTask($task, $reminders);
                } catch (\Exception $e) {
                    // Log error but don't fail the task creation
                    \Log::error('Failed to create reminders: ' . $e->getMessage());
                }
            }

            return redirect()->route('tasks.show', $task)
                ->with('success', 'Task created successfully.');
        } catch (\Exception $e) {
            \Log::error('Task creation failed: ' . $e->getMessage());
            return redirect()->back()
                ->withInput()
                ->withErrors(['error' => 'Failed to create task. Please try again.']);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Task $task)
    {
        $this->authorize('view', $task);
        $task->load(['category', 'checklists', 'comments.user', 'images', 'reminders']);
        return view('tasks.show', compact('task'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Task $task)
    {
        $this->authorize('update', $task);
        $categories = Auth::user()->categories()->get();
        $task->load('reminders');
        return view('tasks.edit', compact('task', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Task $task)
    {
        $this->authorize('update', $task);

        $validated = $request->validate([
            'title' => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'category_id' => ['required', 'exists:categories,id'],
            'due_date' => ['nullable'],
            'difficulty_level' => ['required', 'in:easy,medium,hard,expert'],
            'repeat_rule' => ['nullable', 'array'],
            'status' => ['nullable', 'in:pending,completed,skipped'],
            'reminders' => ['nullable', 'array'],
            'reminders.*.enabled' => ['nullable', 'boolean'],
            'reminders.*.type' => ['nullable', 'in:green,yellow,red'],
            'reminders.*.schedule_type' => ['nullable', 'in:weekly,daily,custom'],
            'reminders.*.time' => ['nullable', 'date_format:H:i'],
            'reminders.*.days' => ['nullable', 'array'],
            'reminders.*.days_before' => ['nullable', 'integer'],
            'reminders.*.hours_before' => ['nullable', 'integer', 'min:0', 'max:48'],
        ]);


        // Get difficulty level configuration
        $difficultyConfig = config('difficulty_levels.levels.' . $validated['difficulty_level']);
        
        // Auto-calculate points and coins based on difficulty level
        $validated['points'] = $difficultyConfig['points'];
        $validated['coins'] = $difficultyConfig['coins'];

        // Handle due_date format conversion
        if (!empty($validated['due_date'])) {
            try {
                $validated['due_date'] = \Carbon\Carbon::parse($validated['due_date'])->format('Y-m-d H:i:s');
            } catch (\Exception $e) {
                return redirect()->back()
                    ->withErrors(['due_date' => 'Invalid date format.'])
                    ->withInput();
            }
        }

        // Extract reminders before updating task
        $reminders = $validated['reminders'] ?? [];
        unset($validated['reminders']);

        $task->update($validated);

        // Update reminders if provided
        if (isset($reminders)) {
            $this->reminderService->createRemindersForTask($task, $reminders);
        }

        return redirect()->route('tasks.show', $task)
            ->with('success', 'Task updated successfully.');
    }

    /**
     * Display completed quests organized by categories.
     */
    public function completed(Request $request)
    {
        $viewMode = $request->get('view', session('quest_view_mode', 'cards'));
        session(['quest_view_mode' => $viewMode]);

        // Use Categories as Quest Scrolls
        $categoriesQuery = Auth::user()->categories()->orderBy('created_at', 'desc');

        // Filter by specific category/scroll if requested
        if ($request->has('scroll') && $request->scroll) {
            $categoriesQuery->where('id', $request->scroll);
        }

        $categories = $categoriesQuery->get();

        // Load completed tasks for each category (scroll) with filters
        foreach ($categories as $category) {
            $taskQuery = Task::where('user_id', Auth::id())
                ->where('category_id', $category->id)
                ->where('status', 'completed')
                ->with(['category', 'reminders']);

            // Difficulty filter
            if ($request->has('difficulty') && $request->difficulty) {
                $taskQuery->where('difficulty_level', $request->difficulty);
            }

            // Date filter
            if ($request->has('completed_date')) {
                if ($request->completed_date === 'today') {
                    $taskQuery->whereDate('completed_at', today());
                } elseif ($request->completed_date === 'this_week') {
                    $taskQuery->whereBetween('completed_at', [\Carbon\Carbon::now()->startOfWeek(), \Carbon\Carbon::now()->endOfWeek()]);
                } elseif ($request->completed_date === 'this_month') {
                    $taskQuery->whereBetween('completed_at', [\Carbon\Carbon::now()->startOfMonth(), \Carbon\Carbon::now()->endOfMonth()]);
                }
            }

            // Set filtered tasks
            $category->setRelation('tasks', $taskQuery->orderBy('completed_at', 'desc')
                ->get());
        }

        // Also get completed tasks without category_id (uncategorized)
        $uncategorizedTasksQuery = Task::where('user_id', Auth::id())
            ->whereNull('category_id')
            ->where('status', 'completed')
            ->with(['category', 'reminders']);

        // Apply same filters
        if ($request->has('difficulty') && $request->difficulty) {
            $uncategorizedTasksQuery->where('difficulty_level', $request->difficulty);
        }
        if ($request->has('completed_date')) {
            if ($request->completed_date === 'today') {
                $uncategorizedTasksQuery->whereDate('completed_at', today());
            } elseif ($request->completed_date === 'this_week') {
                $uncategorizedTasksQuery->whereBetween('completed_at', [\Carbon\Carbon::now()->startOfWeek(), \Carbon\Carbon::now()->endOfWeek()]);
            } elseif ($request->completed_date === 'this_month') {
                $uncategorizedTasksQuery->whereBetween('completed_at', [\Carbon\Carbon::now()->startOfMonth(), \Carbon\Carbon::now()->endOfMonth()]);
            }
        }

        $uncategorizedTasks = $uncategorizedTasksQuery->orderBy('completed_at', 'desc')->get();

        // Create a virtual category for uncategorized completed tasks if there are any
        if ($uncategorizedTasks->count() > 0) {
            $uncategorizedCategory = new Category([
                'id' => 0,
                'name' => 'Uncategorized Quests',
                'icon' => '📋',
                'color' => '#666666',
                'repeat_type' => 'one-time',
            ]);
            $uncategorizedCategory->setRelation('tasks', $uncategorizedTasks);
            $categories->push($uncategorizedCategory);
        }

        return view('tasks.completed', compact('categories', 'viewMode'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Task $task)
    {
        $this->authorize('delete', $task);
        $task->delete();

        return redirect()->route('tasks.index')
            ->with('success', 'Task deleted successfully.');
    }

    /**
     * Complete a task.
     */
    public function complete(Task $task)
    {
        $this->authorize('update', $task);

        if ($task->status === 'completed') {
            return redirect()->back()->withErrors(['error' => 'Task already completed.']);
        }

        $result = $this->taskService->completeTask($task);

        $message = "Quest completed! Earned {$result['xp_earned']} ✨ Divine Energy";
        if ($result['vip_xp_bonus'] > 0) {
            $message .= " (+{$result['vip_xp_bonus']} VIP Bonus!)";
        }
        $message .= " and {$result['gold_earned']} 💰 Sacred Gold";
        if ($result['vip_gold_bonus'] > 0) {
            $message .= " (+{$result['vip_gold_bonus']} VIP Bonus!)";
        }
        $message .= ".";
        
        if ($result['streak_bonus'] > 0) {
            $message .= " Ra's Blessing: +{$result['streak_bonus']}% bonus from your {$result['streak_days']}-day streak!";
        }
        
        if (isset($result['coins_earned']) && $result['coins_earned'] > 0) {
            $message .= " Bonus: {$result['coins_earned']} premium coins earned!";
        }

        return redirect()->back()
            ->with('success', $message)
            ->with('level_up', $result['level_up'] ?? false)
            ->with('new_level', $result['new_level'] ?? null);
    }
}
