<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class Arena3Background extends Model
{
    protected $fillable = [
        'name',
        'slug',
        'description',
        'image_path',
        'shop_image_path',
        'cost_gold',
        'cost_premium',
        'unlock_level',
        'is_vip_only',
        'is_default',
        'is_active',
        'sort_order',
    ];

    protected function casts(): array
    {
        return [
            'cost_gold' => 'integer',
            'cost_premium' => 'integer',
            'unlock_level' => 'integer',
            'is_vip_only' => 'boolean',
            'is_default' => 'boolean',
            'is_active' => 'boolean',
            'sort_order' => 'integer',
        ];
    }

    /**
     * Get users who own this background.
     */
    public function users(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'user_arena3_backgrounds')
            ->withPivot('is_selected', 'unlocked_at', 'unlock_method')
            ->withTimestamps();
    }

    /**
     * Check if background can be unlocked by level.
     */
    public function canUnlockByLevel(int $level): bool
    {
        return $this->unlock_level !== null && $level >= $this->unlock_level;
    }

    /**
     * Check if background is purchasable.
     */
    public function isPurchasable(): bool
    {
        return ($this->cost_gold !== null && $this->cost_gold > 0) || 
               ($this->cost_premium !== null && $this->cost_premium > 0);
    }

    /**
     * Get the full image URL.
     */
    public function getImageUrlAttribute(): string
    {
        return asset($this->image_path);
    }

    /**
     * Get the full shop image URL.
     */
    public function getShopImageUrlAttribute(): ?string
    {
        return $this->shop_image_path ? asset($this->shop_image_path) : null;
    }
}

