<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Character extends Model
{
    protected $fillable = [
        'name',
        'slug',
        'description',
        'image_normal',
        'image_idle',
        'image_idle_animation',
        'image_win',
        'image_winning',
        'image_winning_animation',
        'image_loss',
        'image_lost',
        'image_lost_animation',
        'sprite_source',
        'video_url',
        'video_win',
        'video_loss',
        'unlock_level',
        'cost_gold',
        'cost_premium',
        'is_vip_only',
        'hp_bonus',
        'attack_bonus',
        'defense_bonus',
        'stamina_bonus',
        'base_hp',
        'base_attack',
        'base_defense',
        'rarity',
        'sort_order',
        'is_active',
    ];

    protected function casts(): array
    {
        return [
            'unlock_level' => 'integer',
            'cost_gold' => 'integer',
            'cost_premium' => 'integer',
            'is_vip_only' => 'boolean',
            'hp_bonus' => 'integer',
            'attack_bonus' => 'integer',
            'defense_bonus' => 'integer',
            'stamina_bonus' => 'integer',
            'base_hp' => 'integer',
            'base_attack' => 'integer',
            'base_defense' => 'integer',
            'sort_order' => 'integer',
            'is_active' => 'boolean',
        ];
    }

    /**
     * Get users who own this character.
     */
    public function users(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'user_characters')
            ->withPivot('unlocked_at', 'unlock_method')
            ->withTimestamps();
    }

    /**
     * Get users who have this character equipped.
     */
    public function equippedUsers(): HasMany
    {
        return $this->hasMany(User::class);
    }

    /**
     * Check if character can be unlocked by level.
     */
    public function canUnlockByLevel(int $level): bool
    {
        return $this->unlock_level !== null && $level >= $this->unlock_level;
    }

    /**
     * Check if character is purchasable.
     */
    public function isPurchasable(): bool
    {
        return ($this->cost_gold !== null && $this->cost_gold > 0) || 
               ($this->cost_premium !== null && $this->cost_premium > 0);
    }

    /**
     * Get the fallback PNG image path for a given state.
     * 
     * @param string $state The animation state
     * @return string|null The path to the PNG file, or null if it doesn't exist
     */
    public function animationFallbackImage(string $state): ?string
    {
        $pngPath = public_path("assets/characters/{$this->slug}/{$state}/01.png");
        
        if (file_exists($pngPath)) {
            return "/assets/characters/{$this->slug}/{$state}/01.png";
        }

        return null;
    }

    /**
     * Check if character has PNG sequence for a given state.
     * 
     * @param string $state The animation state
     * @return bool True if PNG sequence exists
     */
    public function hasPngSequence(string $state, ?string $direction = 'right'): bool
    {
        // Map state to folder name (e.g., 'attack' -> 'fight')
        $folderMap = [
            'attack' => 'fight',
        ];
        
        $folderName = $folderMap[$state] ?? $state;
        $direction = $direction ?? 'right'; // Default to right (facing mob)
        
        // Check for direction-specific folder first, then fallback to base folder
        $sequencePath = public_path("assets/characters/{$this->slug}/{$folderName}/{$direction}");
        if (!is_dir($sequencePath)) {
            // Fallback to base folder (no direction)
            $sequencePath = public_path("assets/characters/{$this->slug}/{$folderName}");
        }
        
        if (!is_dir($sequencePath)) {
            return false;
        }
        
        // Check if at least one PNG file exists
        $files = glob("{$sequencePath}/*.png");
        return count($files) > 0;
    }

    /**
     * Get PNG sequence frames for a given state.
     * 
     * @param string $state The animation state
     * @return array Array of PNG file paths, sorted by filename
     */
    public function getPngSequence(string $state, ?string $direction = 'right'): array
    {
        // Map state to folder name
        $folderMap = [
            'attack' => 'fight',
        ];
        
        $folderName = $folderMap[$state] ?? $state;
        $direction = $direction ?? 'right'; // Default to right (facing mob)
        
        // Check for direction-specific folder first, then fallback to base folder
        $sequencePath = public_path("assets/characters/{$this->slug}/{$folderName}/{$direction}");
        $useDirection = is_dir($sequencePath);
        
        if (!$useDirection) {
            // Fallback to base folder (no direction)
            $sequencePath = public_path("assets/characters/{$this->slug}/{$folderName}");
        }
        
        if (!is_dir($sequencePath)) {
            return [];
        }
        
        // Get all PNG files and sort them
        $files = glob("{$sequencePath}/*.png");
        natsort($files); // Natural sort (01.png, 02.png, etc.)
        
        // Convert to web paths
        $paths = [];
        $pathPrefix = $useDirection 
            ? "/assets/characters/{$this->slug}/{$folderName}/{$direction}"
            : "/assets/characters/{$this->slug}/{$folderName}";
            
        foreach ($files as $file) {
            $paths[] = $pathPrefix . "/" . basename($file);
        }
        
        return $paths;
    }

    /**
     * Get all states that have PNG sequences.
     * 
     * @return array Array of state names that have PNG sequences
     */
    public function getPngSequenceStates(?string $direction = 'right'): array
    {
        $states = [];
        $allStates = config('character_animations.states', []);
        
        foreach ($allStates as $state) {
            if ($this->hasPngSequence($state, $direction)) {
                $states[] = $state;
            }
        }
        
        return $states;
    }

    /**
     * Get all available animation states (PNG sequences only).
     * 
     * @return array Array with 'png' key containing state arrays
     */
    public function getAllAnimationStates(): array
    {
        return [
            'png' => $this->getPngSequenceStates(),
        ];
    }

    /**
     * Get state modifiers for this character, falling back to global config if not set.
     */
    public function getStateModifiers(): array
    {
        if ($this->state_modifiers && !empty($this->state_modifiers)) {
            return $this->state_modifiers;
        }
        
        // Fall back to global config
        return config('character_animations.state_modifiers', []);
    }
    
    /**
     * Get modifier for a specific state.
     */
    public function getStateModifier(string $state): array
    {
        $modifiers = $this->getStateModifiers();
        $stateMod = $modifiers[$state] ?? $modifiers['idle'] ?? [];
        
        // Convert percentage-based modifiers to multipliers if needed
        // If modifiers are in percentage format, convert to multiplier
        if (isset($stateMod['hp_percent']) || isset($stateMod['attack_percent']) || isset($stateMod['defense_percent'])) {
            return [
                'hp_multiplier' => 1.0 + (($stateMod['hp_percent'] ?? 0) / 100),
                'attack_multiplier' => 1.0 + (($stateMod['attack_percent'] ?? 0) / 100),
                'defense_multiplier' => 1.0 + (($stateMod['defense_percent'] ?? 0) / 100),
            ];
        }
        
        // If already in multiplier format, return as-is
        return [
            'hp_multiplier' => $stateMod['hp_multiplier'] ?? 1.0,
            'attack_multiplier' => $stateMod['attack_multiplier'] ?? 1.0,
            'defense_multiplier' => $stateMod['defense_multiplier'] ?? 1.0,
        ];
    }
}
