<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Builder;

class Mob extends Model
{
    protected $fillable = [
        'name',
        'slug',
        'level',
        'hp',
        'attack',
        'defense',
        'speed',
        'luck',
        'xp_reward',
        'gold_reward',
        'premium_coin_chance',
        'rarity',
        'loot_table',
        'ai_type',
        'sprite_source',
        'image_normal',
        'image_win',
        'image_loss',
    ];

    protected function casts(): array
    {
        return [
            'level' => 'integer',
            'hp' => 'integer',
            'attack' => 'integer',
            'defense' => 'integer',
            'speed' => 'integer',
            'luck' => 'integer',
            'xp_reward' => 'integer',
            'gold_reward' => 'integer',
            'premium_coin_chance' => 'decimal:4',
            'loot_table' => 'array',
        ];
    }

    /**
     * Get battle histories for this mob.
     */
    public function battleHistories(): HasMany
    {
        return $this->hasMany(BattleHistory::class);
    }

    /**
     * Scope a query to only include mobs at a specific level.
     */
    public function scopeByLevel(Builder $query, int $level): Builder
    {
        return $query->where('level', $level);
    }

    /**
     * Scope a query to only include mobs of a specific rarity.
     */
    public function scopeByRarity(Builder $query, string $rarity): Builder
    {
        return $query->where('rarity', $rarity);
    }

    /**
     * Scope a query to only include active mobs (placeholder for future use).
     */
    public function scopeActive(Builder $query): Builder
    {
        return $query;
    }

    /**
     * Check if mob has PNG sequence for a given state.
     * 
     * @param string $state The animation state
     * @param string|null $direction The direction ('left' or 'right')
     * @return bool True if PNG sequence exists
     */
    public function hasPngSequence(string $state, ?string $direction = 'left'): bool
    {
        // Mobs face left (towards character)
        $direction = $direction ?? 'left';
        
        // Check for direction-specific folder first, then fallback to base folder
        $sequencePath = public_path("assets/mobs/{$this->slug}/{$state}/{$direction}");
        if (!is_dir($sequencePath)) {
            // Fallback to base folder (no direction)
            $sequencePath = public_path("assets/mobs/{$this->slug}/{$state}");
        }
        
        if (!is_dir($sequencePath)) {
            return false;
        }
        
        // Check if at least one PNG file exists
        $files = glob("{$sequencePath}/*.png");
        return count($files) > 0;
    }

    /**
     * Get PNG sequence frames for a given state.
     * 
     * @param string $state The animation state
     * @param string|null $direction The direction ('left' or 'right')
     * @return array Array of PNG file paths, sorted by filename
     */
    public function getPngSequence(string $state, ?string $direction = 'left'): array
    {
        // Mobs face left (towards character)
        $direction = $direction ?? 'left';
        
        // Check for direction-specific folder first, then fallback to base folder
        $sequencePath = public_path("assets/mobs/{$this->slug}/{$state}/{$direction}");
        $useDirection = is_dir($sequencePath);
        
        if (!$useDirection) {
            // Fallback to base folder (no direction)
            $sequencePath = public_path("assets/mobs/{$this->slug}/{$state}");
        }
        
        if (!is_dir($sequencePath)) {
            return [];
        }
        
        // Get all PNG files and sort them
        $files = glob("{$sequencePath}/*.png");
        natsort($files); // Natural sort (01.png, 02.png, etc.)
        
        // Convert to web paths
        $paths = [];
        $pathPrefix = $useDirection 
            ? "/assets/mobs/{$this->slug}/{$state}/{$direction}"
            : "/assets/mobs/{$this->slug}/{$state}";
            
        foreach ($files as $file) {
            $paths[] = $pathPrefix . "/" . basename($file);
        }
        
        return $paths;
    }

    /**
     * Get all states that have PNG sequences.
     * 
     * @param string|null $direction The direction ('left' or 'right')
     * @return array Array of state names that have PNG sequences
     */
    public function getPngSequenceStates(?string $direction = 'left'): array
    {
        $states = [];
        $allStates = ['idle', 'attack', 'magic', 'win', 'lose'];
        
        foreach ($allStates as $state) {
            if ($this->hasPngSequence($state, $direction)) {
                $states[] = $state;
            }
        }
        
        return $states;
    }

    /**
     * Get state modifiers for this mob, falling back to global config if not set.
     */
    public function getStateModifiers(): array
    {
        if ($this->state_modifiers && !empty($this->state_modifiers)) {
            return $this->state_modifiers;
        }
        
        // Fall back to global config
        return config('character_animations.state_modifiers', []);
    }
    
    /**
     * Get modifier for a specific state.
     */
    public function getStateModifier(string $state): array
    {
        $modifiers = $this->getStateModifiers();
        $stateMod = $modifiers[$state] ?? $modifiers['idle'] ?? [];
        
        // Convert percentage-based modifiers to multipliers if needed
        // If modifiers are in percentage format, convert to multiplier
        if (isset($stateMod['hp_percent']) || isset($stateMod['attack_percent']) || isset($stateMod['defense_percent'])) {
            return [
                'hp_multiplier' => 1.0 + (($stateMod['hp_percent'] ?? 0) / 100),
                'attack_multiplier' => 1.0 + (($stateMod['attack_percent'] ?? 0) / 100),
                'defense_multiplier' => 1.0 + (($stateMod['defense_percent'] ?? 0) / 100),
            ];
        }
        
        // If already in multiplier format, return as-is
        return [
            'hp_multiplier' => $stateMod['hp_multiplier'] ?? 1.0,
            'attack_multiplier' => $stateMod['attack_multiplier'] ?? 1.0,
            'defense_multiplier' => $stateMod['defense_multiplier'] ?? 1.0,
        ];
    }
}
