<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Builder;

class ShopItem extends Model
{
    protected $fillable = [
        'name',
        'slug',
        'type',
        'rarity',
        'attack_bonus',
        'defense_bonus',
        'hp_bonus',
        'xp_bonus_percent',
        'gold_bonus_percent',
        'cost_gold',
        'cost_premium',
        'is_vip_only',
        'image_url',
    ];

    protected function casts(): array
    {
        return [
            'attack_bonus' => 'integer',
            'defense_bonus' => 'integer',
            'hp_bonus' => 'integer',
            'xp_bonus_percent' => 'integer',
            'gold_bonus_percent' => 'integer',
            'cost_gold' => 'integer',
            'cost_premium' => 'integer',
            'is_vip_only' => 'boolean',
        ];
    }

    /**
     * Get users who own this item.
     */
    public function users(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'user_shop_items')
            ->withPivot('equipped', 'purchased_at')
            ->withTimestamps();
    }

    /**
     * Scope a query to only include items of a specific type.
     */
    public function scopeByType(Builder $query, string $type): Builder
    {
        return $query->where('type', $type);
    }

    /**
     * Scope a query to only include items of a specific rarity.
     */
    public function scopeByRarity(Builder $query, string $rarity): Builder
    {
        return $query->where('rarity', $rarity);
    }

    /**
     * Scope a query to only include available items.
     */
    public function scopeAvailable(Builder $query): Builder
    {
        return $query;
    }
}
