<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Task extends Model
{
    protected $fillable = [
        'title',
        'description',
        'category_id',
        'scroll_id',
        'user_id',
        'repeat_rule',
        'due_date',
        'completed_at',
        'points',
        'difficulty_level',
        'coins',
        'status',
    ];

    protected function casts(): array
    {
        return [
            'repeat_rule' => 'array',
            'due_date' => 'datetime',
            'completed_at' => 'datetime',
            'points' => 'integer',
            'coins' => 'integer',
            'status' => 'string',
        ];
    }

    /**
     * Get the user that owns the task.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the category that owns the task.
     */
    public function category(): BelongsTo
    {
        return $this->belongsTo(Category::class);
    }

    /**
     * Get the scroll category for this task.
     */
    public function scrollCategory(): BelongsTo
    {
        return $this->belongsTo(ScrollCategory::class, 'scroll_id');
    }

    /**
     * Get the checklists for the task.
     */
    public function checklists(): HasMany
    {
        return $this->hasMany(TaskChecklist::class);
    }

    /**
     * Get the comments for the task.
     */
    public function comments(): HasMany
    {
        return $this->hasMany(TaskComment::class);
    }

    /**
     * Get the images for the task.
     */
    public function images(): HasMany
    {
        return $this->hasMany(TaskImage::class);
    }

    /**
     * Get the reminders for the task.
     */
    public function reminders(): HasMany
    {
        return $this->hasMany(TaskReminder::class);
    }
}
