<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;

class TaskReminder extends Model
{
    protected $fillable = [
        'task_id',
        'reminder_type',
        'schedule_type',
        'schedule_data',
        'reminder_time',
        'days_before',
        'hours_before',
        'last_sent_at',
        'sent_count',
        'is_active',
    ];

    protected function casts(): array
    {
        return [
            'schedule_data' => 'array',
            'reminder_time' => 'datetime',
            'days_before' => 'integer',
            'hours_before' => 'integer',
            'last_sent_at' => 'datetime',
            'sent_count' => 'integer',
            'is_active' => 'boolean',
        ];
    }

    /**
     * Get the task that owns the reminder.
     */
    public function task(): BelongsTo
    {
        return $this->belongsTo(Task::class);
    }

    /**
     * Check if reminder should be sent now.
     */
    public function shouldSend(): bool
    {
        if (!$this->is_active || !$this->task->due_date) {
            return false;
        }

        // Don't send if task is already completed
        if ($this->task->status === 'completed') {
            return false;
        }

        $now = Carbon::now();
        $dueDate = Carbon::parse($this->task->due_date);
        
        // For hour-based reminders, we don't need to check reminder_time strictly
        // For day-based reminders, check if it's the right time of day
        if ($this->hours_before === null) {
            $currentTime = $now->format('H:i:s');
            $reminderTime = Carbon::parse($this->reminder_time)->format('H:i:s');
            
            // Check if it's the right time of day (within 1 minute window)
            if (abs(strtotime($currentTime) - strtotime($reminderTime)) > 60) {
                return false;
            }
        }

        // Check if we already sent this reminder recently (within last hour for hours_before, or today for days_before)
        if ($this->last_sent_at) {
            $lastSent = Carbon::parse($this->last_sent_at);
            if ($this->hours_before !== null) {
                // For hour-based reminders, check if sent within last hour
                if ($lastSent->diffInHours($now) < 1) {
                    return false;
                }
            } else {
                // For day-based reminders, check if sent today
                if ($lastSent->isToday()) {
                    return false;
                }
            }
        }

        // Calculate time until due date
        $daysUntilDue = $now->diffInDays($dueDate, false);
        $hoursUntilDue = $now->diffInHours($dueDate, false);
        $isSameDay = $now->isSameDay($dueDate);
        $isNearDays = $daysUntilDue >= 0 && $daysUntilDue <= 2; // Within 2 days

        // For one-time tasks with due dates
        if ($this->task->category->repeat_type === 'once' && $this->task->due_date) {
            if ($this->schedule_type === 'weekly') {
                // Weekly reminders - check if today is one of the scheduled days
                $scheduledDays = $this->schedule_data['days'] ?? [];
                $todayDayOfWeek = $now->dayOfWeek; // 0 = Sunday, 6 = Saturday
                
                if (!in_array($todayDayOfWeek, $scheduledDays)) {
                    return false;
                }

                // Only send if we're still far enough from due date (more than 7 days)
                return $daysUntilDue > 7;
            } elseif ($this->schedule_type === 'daily') {
                // Check hours_before first (for same day or near days)
                if ($this->hours_before !== null && ($isSameDay || $isNearDays)) {
                    // Calculate target reminder time (due date minus hours_before)
                    $targetTime = $dueDate->copy()->subHours($this->hours_before);
                    
                    // Check if current time is at or past the target time, but before due date
                    if ($now->gte($targetTime) && $now->lt($dueDate)) {
                        // Check if we're within a reasonable window (5 minutes) of the target time
                        $timeDiff = abs($now->diffInMinutes($targetTime));
                        return $timeDiff <= 5;
                    }
                    return false;
                }
                
                // Daily reminders - check days_before
                if ($this->days_before !== null) {
                    // If hours_before is also set, prioritize hours_before logic
                    if ($this->hours_before === null) {
                        return $daysUntilDue == $this->days_before || 
                               ($daysUntilDue < $this->days_before && $daysUntilDue >= 0);
                    }
                }
            }
        }

        // For repetitive tasks
        if (in_array($this->task->category->repeat_type, ['daily', 'weekly', 'custom'])) {
            if ($this->schedule_type === 'daily') {
                // Send reminder daily at specified time
                return true;
            } elseif ($this->schedule_type === 'weekly') {
                $scheduledDays = $this->schedule_data['days'] ?? [];
                $todayDayOfWeek = $now->dayOfWeek;
                return in_array($todayDayOfWeek, $scheduledDays);
            }
        }

        return false;
    }

    /**
     * Mark reminder as sent.
     */
    public function markAsSent(): void
    {
        $this->last_sent_at = now();
        $this->sent_count++;
        $this->save();
    }

    /**
     * Get reminder type color class.
     */
    public function getColorClass(): string
    {
        return match($this->reminder_type) {
            'green' => 'text-green-400 bg-green-900/30 border-green-500',
            'yellow' => 'text-yellow-400 bg-yellow-900/30 border-yellow-500',
            'red' => 'text-red-400 bg-red-900/30 border-red-500',
            default => 'text-gray-400 bg-gray-900/30 border-gray-500',
        };
    }

    /**
     * Get reminder type label.
     */
    public function getTypeLabel(): string
    {
        return match($this->reminder_type) {
            'green' => 'Plenty of Time',
            'yellow' => 'Running Out of Time',
            'red' => 'Urgent',
            default => 'Reminder',
        };
    }
}
