<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'xp',
        'gold',
        'level',
        'max_hp',
        'current_hp',
        'attack',
        'defense',
        'stamina',
        'luck',
        'avatar_url',
        'is_admin',
        'is_vip',
        'vip_expires_at',
        'premium_coins',
        'membership_id',
        'last_vip_reward_claimed_at',
        'last_battle_at',
        'last_hp_regen_at',
        'character_id',
        'character_image_normal',
        'character_image_win',
        'character_image_loss',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'xp' => 'integer',
            'gold' => 'integer',
            'level' => 'integer',
            'max_hp' => 'integer',
            'current_hp' => 'integer',
            'attack' => 'integer',
            'defense' => 'integer',
            'stamina' => 'integer',
            'luck' => 'integer',
            'is_admin' => 'boolean',
            'is_vip' => 'boolean',
            'vip_expires_at' => 'datetime',
            'premium_coins' => 'integer',
            'membership_id' => 'integer',
            'last_vip_reward_claimed_at' => 'datetime',
            'last_battle_at' => 'datetime',
            'last_hp_regen_at' => 'datetime',
        ];
    }

    /**
     * Get the categories for the user.
     */
    public function categories()
    {
        return $this->hasMany(Category::class);
    }

    /**
     * Get the scroll categories for the user.
     */
    public function scrollCategories()
    {
        return $this->hasMany(ScrollCategory::class);
    }

    /**
     * Get the tasks for the user.
     */
    public function tasks()
    {
        return $this->hasMany(Task::class);
    }

    /**
     * Get the user items.
     */
    public function userItems()
    {
        return $this->hasMany(UserItem::class);
    }

    /**
     * Get the items through user items.
     */
    public function items()
    {
        return $this->belongsToMany(Item::class, 'user_items')
            ->withPivot('equipped', 'purchased_at')
            ->withTimestamps();
    }

    /**
     * Get the battles for the user.
     */
    public function battles()
    {
        return $this->hasMany(Battle::class);
    }

    /**
     * Get the user achievements.
     */
    public function userAchievements()
    {
        return $this->hasMany(UserAchievement::class);
    }

    /**
     * Get the achievements through user achievements.
     */
    public function achievements()
    {
        return $this->belongsToMany(Achievement::class, 'user_achievements')
            ->withPivot('unlocked_at')
            ->withTimestamps();
    }

    /**
     * Get the membership for the user.
     */
    public function membership()
    {
        return $this->belongsTo(Membership::class);
    }

    /**
     * Get the level data for the user.
     */
    public function levelData()
    {
        return $this->belongsTo(Level::class, 'level', 'level');
    }

    /**
     * Get battle histories for the user.
     */
    public function battleHistories()
    {
        return $this->hasMany(BattleHistory::class);
    }

    /**
     * Get the currently equipped character.
     */
    public function character()
    {
        return $this->belongsTo(Character::class);
    }

    /**
     * Get all characters owned by the user.
     */
    public function characters()
    {
        return $this->belongsToMany(Character::class, 'user_characters')
            ->withPivot('unlocked_at', 'unlock_method')
            ->withTimestamps();
    }

    /**
     * Get Arena 3 backgrounds owned by the user.
     */
    public function arena3Backgrounds()
    {
        return $this->belongsToMany(Arena3Background::class, 'user_arena3_backgrounds')
            ->withPivot('is_selected', 'unlocked_at', 'unlock_method')
            ->withTimestamps();
    }

    /**
     * Get shop items owned by the user.
     */
    public function shopItems()
    {
        return $this->belongsToMany(ShopItem::class, 'user_shop_items')
            ->withPivot('equipped', 'purchased_at')
            ->withTimestamps();
    }

    /**
     * Check if user is currently VIP.
     */
    public function isVip(): bool
    {
        return $this->is_vip && $this->vip_expires_at && $this->vip_expires_at->isFuture();
    }

    /**
     * Add XP to user and check for level up.
     */
    public function giveXp(int $amount): void
    {
        $this->xp += $amount;
        $this->save();
        $this->levelUpIfNeeded();
    }

    /**
     * Check if user has enough XP to level up and level up if needed.
     */
    public function levelUpIfNeeded(): void
    {
        $levelService = app(\App\Services\LevelService::class);
        $currentLevel = $this->level;
        
        // Calculate total XP needed to reach next level
        $totalXpNeeded = 0;
        for ($i = 1; $i <= $currentLevel; $i++) {
            $totalXpNeeded += $levelService->xpForNextLevel($i);
        }
        
        // Check if we have enough XP for next level
        if ($this->xp >= $totalXpNeeded) {
            $oldLevel = $this->level;
            $this->level++;
            
            // Get old and new level data
            $oldLevelData = Level::findByLevel($oldLevel);
            $newLevelData = Level::findByLevel($this->level);
            
            if ($newLevelData) {
                // Calculate stat increases (cumulative bonuses)
                $hpIncrease = $newLevelData->hp_bonus - ($oldLevelData ? $oldLevelData->hp_bonus : 0);
                $attackIncrease = $newLevelData->attack_bonus - ($oldLevelData ? $oldLevelData->attack_bonus : 0);
                $defenseIncrease = $newLevelData->defense_bonus - ($oldLevelData ? $oldLevelData->defense_bonus : 0);
                $staminaIncrease = $newLevelData->stamina_bonus - ($oldLevelData ? $oldLevelData->stamina_bonus : 0);
                
                $this->max_hp += $hpIncrease;
                $this->attack += $attackIncrease;
                $this->defense += $defenseIncrease;
                $this->stamina += $staminaIncrease;
            }
            
            // Full heal on level up
            $this->current_hp = $this->max_hp;
            
            // Increment luck every 3 levels
            if ($this->level % 3 === 0) {
                $this->luck++;
            }
            
            $this->save();
            
            // Unlock characters for this level
            $this->unlockCharactersForLevel($this->level);
            
            // Recursively check for additional level ups
            $this->levelUpIfNeeded();
        }
    }

    /**
     * Unlock characters that are available at the given level.
     */
    public function unlockCharactersForLevel(int $level): void
    {
        $charactersToUnlock = \App\Models\Character::where('unlock_level', $level)
            ->where('is_active', true)
            ->whereDoesntHave('users', function($query) {
                $query->where('user_id', $this->id);
            })
            ->get();
        
        foreach ($charactersToUnlock as $character) {
            $this->characters()->attach($character->id, [
                'unlocked_at' => now(),
                'unlock_method' => 'level_up',
            ]);
        }
    }

    /**
     * Calculate total stats from base stats, level bonuses, and equipped items.
     */
    public function applyStatBonusesFromItems(): array
    {
        $baseStats = [
            'attack' => $this->attack ?? 12,
            'defense' => $this->defense ?? 5,
            'hp' => $this->max_hp ?? 120,
            'stamina' => $this->stamina ?? 100,
            'luck' => $this->luck ?? 3,
        ];

        // Add level bonuses
        $levelData = Level::findByLevel($this->level);
        if ($levelData) {
            $baseStats['attack'] += $levelData->attack_bonus;
            $baseStats['defense'] += $levelData->defense_bonus;
            $baseStats['hp'] += $levelData->hp_bonus;
            $baseStats['stamina'] += $levelData->stamina_bonus;
        }

        // Add item bonuses from equipped items
        $equippedItems = $this->shopItems()->wherePivot('equipped', true)->get();
        foreach ($equippedItems as $item) {
            $baseStats['attack'] += $item->attack_bonus;
            $baseStats['defense'] += $item->defense_bonus;
            $baseStats['hp'] += $item->hp_bonus;
        }

        return $baseStats;
    }

    /**
     * Get total stats (base + level + items).
     */
    public function getTotalStats(): array
    {
        return $this->applyStatBonusesFromItems();
    }
}
