<?php

namespace App\Services;

use App\Models\User;
use App\Models\Mob;
use App\Models\BattleHistory;
use App\Services\LevelService;
use App\Services\LootService;
use App\Services\CurrencyService;
use App\Services\VipService;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class BattleService
{
    protected LevelService $levelService;
    protected LootService $lootService;
    protected CurrencyService $currencyService;
    protected VipService $vipService;
    protected int $baseCooldownMinutes = 5;

    public function __construct(
        LevelService $levelService,
        LootService $lootService,
        CurrencyService $currencyService,
        VipService $vipService
    ) {
        $this->levelService = $levelService;
        $this->lootService = $lootService;
        $this->currencyService = $currencyService;
        $this->vipService = $vipService;
    }

    /**
     * Get battle cooldown in seconds for a user.
     */
    public function getBattleCooldown(User $user): int
    {
        $cooldownMinutes = $this->baseCooldownMinutes;

        // Apply VIP reduction if user is VIP
        if ($this->vipService->isVip($user)) {
            $membership = $this->vipService->getActiveMembership($user);
            if ($membership && $membership->battle_cooldown_reduction > 0) {
                $reductionPercent = $membership->battle_cooldown_reduction / 100;
                $cooldownMinutes = (int) round($cooldownMinutes * (1 - $reductionPercent));
            }
        }

        return $cooldownMinutes * 60; // Convert to seconds
    }

    /**
     * Check if user can start a battle (cooldown has passed).
     */
    public function canStartBattle(User $user): bool
    {
        if (!$user->last_battle_at) {
            return true;
        }

        $cooldownSeconds = $this->getBattleCooldown($user);
        $cooldownUntil = $user->last_battle_at->copy()->addSeconds($cooldownSeconds);

        return now()->greaterThanOrEqualTo($cooldownUntil);
    }

    /**
     * Set battle cooldown for a user.
     */
    public function setBattleCooldown(User $user): void
    {
        $user->last_battle_at = now();
        $user->save();
    }

    /**
     * Start a new battle between a user and a mob.
     */
    public function startBattle(User $user, Mob $mob, string $mode = 'auto'): array
    {
        // Check cooldown
        if (!$this->canStartBattle($user)) {
            throw new \Exception('Battle cooldown has not expired yet.');
        }

        $userStats = $user->getTotalStats();
        $userHP = $user->current_hp ?? $userStats['hp'];
        $mobHP = $mob->hp;

        // Generate battle instance ID
        $battleId = Str::uuid()->toString();

        // Create battle instance
        $battleInstance = [
            'battle_id' => $battleId,
            'user_id' => $user->id,
            'mob_id' => $mob->id,
            'mode' => $mode,
            'user_hp' => $userHP,
            'user_max_hp' => $userStats['hp'],
            'mob_hp' => $mobHP,
            'mob_max_hp' => $mob->hp,
            'turns' => [],
            'result' => null,
            'rewards' => null,
            'created_at' => now()->toIso8601String(),
        ];

        // Store in cache for 1 hour
        Cache::put("battle_instance:{$battleId}", $battleInstance, 3600);

        // Set cooldown
        $this->setBattleCooldown($user);

        Log::info("Battle started between user {$user->id} and mob {$mob->id} in {$mode} mode");

        // If auto mode, simulate immediately
        if ($mode === 'auto') {
            return $this->simulateBattle($user, $mob, $battleId);
        }

        // Return initial state for manual mode
        return [
            'battle_id' => $battleId,
            'mode' => $mode,
            'user_id' => $user->id,
            'mob' => [
                'id' => $mob->id,
                'name' => $mob->name,
                'hp' => $mobHP,
                'max_hp' => $mob->hp,
                'sprite' => $mob->sprite_source,
            ],
            'player' => [
                'hp' => $userHP,
                'max_hp' => $userStats['hp'],
                'attack' => $userStats['attack'],
                'defense' => $userStats['defense'],
                'speed' => $userStats['stamina'] ?? 100, // Using stamina as speed for now
                'luck' => $userStats['luck'],
            ],
            'turns' => [],
            'result' => null,
            'rewards' => null,
        ];
    }

    /**
     * Simulate a full automated battle.
     */
    public function simulateBattle(User $user, Mob $mob, ?string $battleId = null): array
    {
        if (!$battleId) {
            $battleId = Str::uuid()->toString();
        }

        $userStats = $user->getTotalStats();
        $userHP = $user->current_hp ?? $userStats['hp'];
        $mobHP = $mob->hp;
        $turns = [];
        $turnNumber = 0;
        $maxTurns = 100; // Safety limit

        while ($userHP > 0 && $mobHP > 0 && $turnNumber < $maxTurns) {
            $turnNumber++;
            
            // Determine turn order based on speed
            $turnOrder = $this->determineTurnOrder($user, $mob, $userStats);
            
            // Player turn
            if ($turnOrder['player_first']) {
                $playerTurn = $this->performPlayerAttack($user, $mob, $userStats, $userHP, $mobHP, $turnNumber);
                $userHP = $playerTurn['user_hp'];
                $mobHP = $playerTurn['mob_hp'];
                $turns[] = $playerTurn['turn_log'];
                
                if ($mobHP <= 0) {
                    break; // Player won
                }
            }

            // Mob turn
            $mobTurn = $this->performMobAttack($mob, $user, $userStats, $userHP, $mobHP, $turnNumber);
            $userHP = $mobTurn['user_hp'];
            $mobHP = $mobTurn['mob_hp'];
            $turns[] = $mobTurn['turn_log'];
            
            if ($userHP <= 0) {
                break; // Mob won
            }

            // If player didn't go first, they go now
            if (!$turnOrder['player_first']) {
                $playerTurn = $this->performPlayerAttack($user, $mob, $userStats, $userHP, $mobHP, $turnNumber);
                $userHP = $playerTurn['user_hp'];
                $mobHP = $playerTurn['mob_hp'];
                $turns[] = $playerTurn['turn_log'];
                
                if ($mobHP <= 0) {
                    break; // Player won
                }
            }
        }

        // Determine result
        $userWon = $userHP > 0 && $mobHP <= 0;
        $result = $userWon ? 'win' : 'lose';

        // Generate rewards if won
        $rewards = null;
        if ($userWon) {
            $rewards = $this->lootService->generateLoot($mob, $user);
            
            // Award rewards
            $this->levelService->addXpToUser($user, $rewards['xp']);
            $this->currencyService->addGold($user, $rewards['gold']);
            if ($rewards['premium_coins'] > 0) {
                $this->currencyService->addPremiumCoins($user, $rewards['premium_coins']);
            }
        }

        // Save to battle history
        $this->saveBattleHistory($user, $mob, $result, $turnNumber, $rewards, $turns);

        // Update user HP
        $user->current_hp = max(0, $userHP);
        $user->save();

        return [
            'battle_id' => $battleId,
            'mode' => 'auto',
            'user_id' => $user->id,
            'mob' => [
                'id' => $mob->id,
                'name' => $mob->name,
                'hp' => max(0, $mobHP),
                'max_hp' => $mob->hp,
                'sprite' => $mob->sprite_source,
                'image_normal' => $mob->image_normal,
                'image_win' => $mob->image_win,
                'image_loss' => $mob->image_loss,
            ],
            'player' => [
                'hp' => max(0, $userHP),
                'max_hp' => $userStats['hp'],
                'attack' => $userStats['attack'],
                'defense' => $userStats['defense'],
                'speed' => $userStats['stamina'] ?? 100,
                'luck' => $userStats['luck'],
            ],
            'turns' => $turns,
            'result' => $result,
            'rewards' => $rewards,
        ];
    }

    /**
     * Perform a single turn in manual mode.
     */
    public function performTurn(string $battleId, ?string $playerAction = 'attack'): array
    {
        $battleInstance = Cache::get("battle_instance:{$battleId}");
        
        if (!$battleInstance) {
            throw new \Exception('Battle instance not found or expired.');
        }

        $user = User::find($battleInstance['user_id']);
        $mob = Mob::find($battleInstance['mob_id']);
        $userStats = $user->getTotalStats();
        $userHP = $battleInstance['user_hp'];
        $mobHP = $battleInstance['mob_hp'];
        $turns = $battleInstance['turns'] ?? [];
        $turnNumber = count($turns) + 1;

        // Determine turn order
        $turnOrder = $this->determineTurnOrder($user, $mob, $userStats);
        
        // Player action
        if ($turnOrder['player_first'] || count($turns) === 0) {
            $playerTurn = $this->performPlayerAction($user, $mob, $userStats, $userHP, $mobHP, $playerAction, $turnNumber);
            $userHP = $playerTurn['user_hp'];
            $mobHP = $playerTurn['mob_hp'];
            $turns[] = $playerTurn['turn_log'];
            
            if ($mobHP <= 0) {
                // Player won
                $rewards = $this->lootService->generateLoot($mob, $user);
                $this->levelService->addXpToUser($user, $rewards['xp']);
                $this->currencyService->addGold($user, $rewards['gold']);
                if ($rewards['premium_coins'] > 0) {
                    $this->currencyService->addPremiumCoins($user, $rewards['premium_coins']);
                }
                
                $this->saveBattleHistory($user, $mob, 'win', $turnNumber, $rewards, $turns);
                $user->current_hp = max(0, $userHP);
                $user->save();
                
                Cache::forget("battle_instance:{$battleId}");
                
                return [
                    'battle_id' => $battleId,
                    'mode' => 'manual',
                    'turns' => $turns,
                    'result' => 'win',
                    'rewards' => $rewards,
                ];
            }
        }

        // Mob action
        $mobTurn = $this->performMobAttack($mob, $user, $userStats, $userHP, $mobHP, $turnNumber);
        $userHP = $mobTurn['user_hp'];
        $mobHP = $mobTurn['mob_hp'];
        $turns[] = $mobTurn['turn_log'];
        
        if ($userHP <= 0) {
            // Mob won
            $this->saveBattleHistory($user, $mob, 'lose', $turnNumber, null, $turns);
            $user->current_hp = 0;
            $user->save();
            
            Cache::forget("battle_instance:{$battleId}");
            
            return [
                'battle_id' => $battleId,
                'mode' => 'manual',
                'turns' => $turns,
                'result' => 'lose',
                'rewards' => null,
            ];
        }

        // Update battle instance
        $battleInstance['user_hp'] = $userHP;
        $battleInstance['mob_hp'] = $mobHP;
        $battleInstance['turns'] = $turns;
        Cache::put("battle_instance:{$battleId}", $battleInstance, 3600);

        return [
            'battle_id' => $battleId,
            'mode' => 'manual',
            'mob' => [
                'id' => $mob->id,
                'name' => $mob->name,
                'hp' => $mobHP,
                'max_hp' => $mob->hp,
            ],
            'player' => [
                'hp' => $userHP,
                'max_hp' => $userStats['hp'],
            ],
            'turns' => $turns,
            'result' => null,
            'rewards' => null,
        ];
    }

    /**
     * Calculate damage with crit chance.
     */
    public function calculateDamage(int $attackerAttack, int $defenderDefense, int $attackerLuck): int
    {
        $base = $attackerAttack - floor($defenderDefense / 2);
        if ($base < 1) {
            $base = 1;
        }

        $randomAdd = random_int(1, 6);
        $damage = max(1, $base + $randomAdd);

        // Calculate crit rate
        $critRate = max(5, min(40, 5 + floor($attackerLuck / 2)));
        
        // Check for crit
        if (random_int(1, 100) <= $critRate) {
            $damage *= 2;
        }

        return $damage;
    }

    /**
     * Determine turn order based on speed.
     */
    public function determineTurnOrder(User $user, Mob $mob, array $userStats): array
    {
        $userSpeed = $userStats['stamina'] ?? 100;
        $mobSpeed = $mob->speed;

        // If tie, player goes first
        $playerFirst = $userSpeed >= $mobSpeed;

        return [
            'player_first' => $playerFirst,
            'player_speed' => $userSpeed,
            'mob_speed' => $mobSpeed,
        ];
    }

    /**
     * Perform player attack.
     */
    protected function performPlayerAttack(User $user, Mob $mob, array $userStats, int $userHP, int $mobHP, int $turnNumber = 0): array
    {
        $baseDamage = $userStats['attack'] - floor($mob->defense / 2);
        if ($baseDamage < 1) $baseDamage = 1;
        
        $randomAdd = random_int(1, 6);
        $damage = max(1, $baseDamage + $randomAdd);
        
        // Check for crit
        $critRate = max(5, min(40, 5 + floor($userStats['luck'] / 2)));
        $isCrit = random_int(1, 100) <= $critRate;
        
        if ($isCrit) {
            $damage *= 2;
        }
        
        $newMobHP = max(0, $mobHP - $damage);

        return [
            'user_hp' => $userHP,
            'mob_hp' => $newMobHP,
            'turn_log' => [
                'turn' => $turnNumber,
                'actor' => 'player',
                'action' => 'attack',
                'damage' => $damage,
                'is_crit' => $isCrit,
                'mob_hp' => $newMobHP,
                'player_hp' => $userHP,
                'log' => $isCrit 
                    ? "You critically strike {$mob->name} for {$damage} damage!" 
                    : "You strike {$mob->name} for {$damage} damage",
            ],
        ];
    }

    /**
     * Perform player action (for manual mode).
     */
    protected function performPlayerAction(User $user, Mob $mob, array $userStats, int $userHP, int $mobHP, string $action, int $turnNumber = 0): array
    {
        // For now, only attack is implemented
        if ($action === 'attack') {
            return $this->performPlayerAttack($user, $mob, $userStats, $userHP, $mobHP, $turnNumber);
        }

        // Placeholder for other actions (use_item, skill, escape)
        return [
            'user_hp' => $userHP,
            'mob_hp' => $mobHP,
            'turn_log' => [
                'turn' => $turnNumber,
                'actor' => 'player',
                'action' => $action,
                'log' => "Action {$action} not yet implemented",
            ],
        ];
    }

    /**
     * Perform mob attack.
     */
    protected function performMobAttack(Mob $mob, User $user, array $userStats, int $userHP, int $mobHP, int $turnNumber = 0): array
    {
        $baseDamage = $mob->attack - floor($userStats['defense'] / 2);
        if ($baseDamage < 1) $baseDamage = 1;
        
        $randomAdd = random_int(1, 6);
        $damage = max(1, $baseDamage + $randomAdd);
        
        // Check for crit
        $critRate = max(5, min(40, 5 + floor($mob->luck / 2)));
        $isCrit = random_int(1, 100) <= $critRate;
        
        if ($isCrit) {
            $damage *= 2;
        }
        
        $newUserHP = max(0, $userHP - $damage);

        $actionName = match($mob->ai_type) {
            'aggressive' => 'strikes',
            'strategic' => 'attacks',
            default => 'hits',
        };

        return [
            'user_hp' => $newUserHP,
            'mob_hp' => $mobHP,
            'turn_log' => [
                'turn' => $turnNumber,
                'actor' => 'mob',
                'action' => $actionName,
                'damage' => $damage,
                'is_crit' => $isCrit,
                'mob_hp' => $mobHP,
                'player_hp' => $newUserHP,
                'log' => $isCrit
                    ? "{$mob->name} critically {$actionName} you for {$damage} damage!"
                    : "{$mob->name} {$actionName} you for {$damage} damage",
            ],
        ];
    }

    /**
     * Save battle to history.
     */
    protected function saveBattleHistory(User $user, Mob $mob, string $result, int $turns, ?array $rewards, array $turnLog): void
    {
        BattleHistory::create([
            'user_id' => $user->id,
            'mob_id' => $mob->id,
            'result' => $result,
            'turns' => $turns,
            'earned_xp' => $rewards['xp'] ?? 0,
            'earned_gold' => $rewards['gold'] ?? 0,
            'earned_premium_coins' => $rewards['premium_coins'] ?? 0,
            'log' => $turnLog,
        ]);
    }

    /**
     * Get available mobs for a user (compatibility method for web controller).
     * Shows only mobs from user level to user level + 2, sorted by lowest power and level.
     */
    public function getAvailableMobs(User $user): array
    {
        $userLevel = $user->level ?? 1;
        $maxLevel = $userLevel + 2; // Show up to 2 levels above user
        
        // Get mobs in the level range (user level to user level + 2)
        $mobs = \App\Models\Mob::whereBetween('level', [$userLevel, $maxLevel])
            ->get()
            ->map(function($mob) {
                // Calculate power (attack + defense + hp)
                $power = ($mob->attack ?? 0) + ($mob->defense ?? 0) + ($mob->hp ?? 0);
                
                return [
                    'id' => $mob->id,
                    'name' => $mob->name,
                    'level' => $mob->level,
                    'attack' => $mob->attack,
                    'defense' => $mob->defense,
                    'hp' => $mob->hp,
                    'speed' => $mob->speed,
                    'rarity' => $mob->rarity,
                    'sprite_source' => $mob->sprite_source,
                    'xp_reward' => $mob->xp_reward,
                    'gold_reward' => $mob->gold_reward,
                    'power' => $power, // Add power for sorting
                ];
            })
            ->sortBy([
                ['level', 'asc'],      // Sort by level ascending (lowest first)
                ['power', 'asc'],      // Then by power ascending (lowest first)
            ])
            ->values() // Reset array keys
            ->toArray();
        
        return $mobs;
    }
}
