<?php

namespace App\Services;

use App\Models\User;
use App\Events\LevelUp;
use App\Events\TaskCompleted;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class GameService
{
    /**
     * Calculate XP for a task based on category and points.
     */
    public function calculateXPForTask($category, $points): int
    {
        $baseXP = $points;
        
        // Category multipliers
        $multipliers = [
            'daily' => 1.5,
            'weekly' => 2.0,
            'once' => 1.0,
            'custom' => 1.0,
        ];

        $multiplier = $multipliers[$category->repeat_type] ?? 1.0;
        
        return (int) round($baseXP * $multiplier);
    }

    /**
     * Add XP to a user and check for level up.
     */
    public function addXP(User $user, int $amount): array
    {
        $oldLevel = $user->level;
        $user->xp += $amount;
        $user->save();

        $newLevel = $this->calculateLevel($user->xp);
        
        $leveledUp = false;
        if ($newLevel > $oldLevel) {
            $user->level = $newLevel;
            $user->save();
            
            $leveledUp = true;
            event(new LevelUp($user, $oldLevel, $newLevel));
            
            Log::info("User {$user->id} leveled up from {$oldLevel} to {$newLevel}");
        }

        return [
            'leveled_up' => $leveledUp,
            'old_level' => $oldLevel,
            'new_level' => $newLevel,
            'total_xp' => $user->xp,
        ];
    }

    /**
     * Calculate level from total XP.
     * Formula: level = floor(sqrt(total_xp / 100))
     */
    public function calculateLevel(int $totalXP): int
    {
        if ($totalXP <= 0) {
            return 1;
        }
        
        return (int) floor(sqrt($totalXP / 100));
    }

    /**
     * Calculate XP needed for a specific level.
     */
    public function xpNeededForLevel(int $level): int
    {
        return 100 * ($level ** 2);
    }

    /**
     * Add gold to a user.
     */
    public function addGold(User $user, int $amount): void
    {
        $user->gold += $amount;
        $user->save();
    }

    /**
     * Deduct gold from a user.
     */
    public function deductGold(User $user, int $amount): bool
    {
        if ($user->gold < $amount) {
            return false;
        }
        
        $user->gold -= $amount;
        $user->save();
        
        return true;
    }

    /**
     * Calculate gold reward for a task.
     */
    public function calculateGoldForTask(int $points): int
    {
        return (int) round($points * 0.5);
    }

    /**
     * Calculate the current daily quest streak.
     * Returns the number of consecutive days the user has completed at least one daily task.
     */
    public function calculateDailyQuestStreak(User $user): array
    {
        $streak = 0;
        $currentDate = Carbon::today();
        $hasCompletedToday = false;
        
        // Check today first
        $todayCount = $user->tasks()
            ->whereHas('category', function ($query) {
                $query->where('repeat_type', 'daily');
            })
            ->where('status', 'completed')
            ->whereDate('completed_at', '>=', $currentDate->copy()->startOfDay())
            ->whereDate('completed_at', '<=', $currentDate->copy()->endOfDay())
            ->count();
        
        if ($todayCount > 0) {
            $hasCompletedToday = true;
            $streak = 1;
            
            // Check previous days
            $checkDate = $currentDate->copy()->subDay();
            
            while (true) {
                $dayCount = $user->tasks()
                    ->whereHas('category', function ($query) {
                        $query->where('repeat_type', 'daily');
                    })
                    ->where('status', 'completed')
                    ->whereDate('completed_at', '>=', $checkDate->copy()->startOfDay())
                    ->whereDate('completed_at', '<=', $checkDate->copy()->endOfDay())
                    ->count();
                
                if ($dayCount > 0) {
                    $streak++;
                    $checkDate->subDay();
                } else {
                    break;
                }
                
                // Safety limit to prevent infinite loops
                if ($streak > 365) {
                    break;
                }
            }
        } else {
            // Check yesterday - if completed yesterday, streak is maintained but no bonus today
            $yesterdayCount = $user->tasks()
                ->whereHas('category', function ($query) {
                    $query->where('repeat_type', 'daily');
                })
                ->where('status', 'completed')
                ->whereDate('completed_at', '>=', $currentDate->copy()->subDay()->startOfDay())
                ->whereDate('completed_at', '<=', $currentDate->copy()->subDay()->endOfDay())
                ->count();
            
            if ($yesterdayCount > 0) {
                // Calculate streak from yesterday backwards
                $checkDate = $currentDate->copy()->subDay();
                
                while (true) {
                    $dayCount = $user->tasks()
                        ->whereHas('category', function ($query) {
                            $query->where('repeat_type', 'daily');
                        })
                        ->where('status', 'completed')
                        ->whereDate('completed_at', '>=', $checkDate->copy()->startOfDay())
                        ->whereDate('completed_at', '<=', $checkDate->copy()->endOfDay())
                        ->count();
                    
                    if ($dayCount > 0) {
                        $streak++;
                        $checkDate->subDay();
                    } else {
                        break;
                    }
                    
                    if ($streak > 365) {
                        break;
                    }
                }
            }
        }
        
        // Calculate bonus multiplier (1% per day, max 50%)
        $bonusMultiplier = min(0.5, $streak * 0.01);
        $bonusPercent = round($bonusMultiplier * 100);
        
        return [
            'streak' => $streak,
            'bonus_multiplier' => $bonusMultiplier,
            'bonus_percent' => $bonusPercent,
            'has_completed_today' => $hasCompletedToday,
        ];
    }

    /**
     * Apply streak bonus to XP and gold rewards.
     */
    public function applyStreakBonus(int $baseXP, int $baseGold, array $streakData): array
    {
        $multiplier = 1 + $streakData['bonus_multiplier'];
        
        return [
            'xp' => (int) round($baseXP * $multiplier),
            'gold' => (int) round($baseGold * $multiplier),
            'bonus_xp' => (int) round($baseXP * $streakData['bonus_multiplier']),
            'bonus_gold' => (int) round($baseGold * $streakData['bonus_multiplier']),
        ];
    }

    /**
     * Get total stats for a user (base stats + level bonuses + equipped items).
     */
    public function getUserTotalStats(User $user): array
    {
        // Start with base stats from user model
        $stats = [
            'attack' => $user->attack ?? 12,
            'defense' => $user->defense ?? 5,
            'magic' => 0, // Magic is not a base stat, only from items
            'stamina' => $user->stamina ?? 100,
            'hp' => $user->max_hp ?? 120, // Max HP
        ];
        
        // Add level bonuses
        $levelData = \App\Models\Level::findByLevel($user->level);
        if ($levelData) {
            $stats['attack'] += $levelData->attack_bonus ?? 0;
            $stats['defense'] += $levelData->defense_bonus ?? 0;
            $stats['stamina'] += $levelData->stamina_bonus ?? 0;
            $stats['hp'] += $levelData->hp_bonus ?? 0;
        }
        
        // Add equipped items bonuses from shopItems (Blessed Relics)
        $equippedShopItems = $user->shopItems()
            ->wherePivot('equipped', true)
            ->get();
        
        foreach ($equippedShopItems as $shopItem) {
            $stats['attack'] += $shopItem->attack_bonus ?? 0;
            $stats['defense'] += $shopItem->defense_bonus ?? 0;
            $stats['hp'] += $shopItem->hp_bonus ?? 0;
        }
        
        // Also add equipped items from userItems (regular items)
        $equippedItems = $user->userItems()
            ->where('equipped', true)
            ->with('item')
            ->get();
        
        foreach ($equippedItems as $userItem) {
            $item = $userItem->item;
            $stats['attack'] += $item->attack ?? 0;
            $stats['defense'] += $item->defense ?? 0;
            $stats['magic'] += $item->magic ?? 0;
            $stats['stamina'] += $item->stamina ?? 0;
        }
        
        // Add character bonuses if equipped
        if ($user->character) {
            $stats['attack'] += $user->character->attack_bonus ?? 0;
            $stats['defense'] += $user->character->defense_bonus ?? 0;
            $stats['hp'] += $user->character->hp_bonus ?? 0;
            $stats['stamina'] += $user->character->stamina_bonus ?? 0;
        }
        
        return $stats;
    }

    /**
     * Check achievements for a user based on trigger type.
     */
    public function checkAchievements(User $user, string $triggerType): void
    {
        $achievements = \App\Models\Achievement::where('condition_type', $triggerType)->get();
        
        foreach ($achievements as $achievement) {
            // Skip if already unlocked
            if ($user->achievements()->where('achievement_id', $achievement->id)->exists()) {
                continue;
            }
            
            $unlocked = false;
            
            switch ($achievement->condition_type) {
                case 'tasks_completed':
                    $count = $user->tasks()->where('status', 'completed')->count();
                    $unlocked = $count >= $achievement->condition_value;
                    break;
                    
                case 'daily_tasks_completed':
                    $count = $user->tasks()
                        ->whereHas('category', function ($query) {
                            $query->where('repeat_type', 'daily');
                        })
                        ->where('status', 'completed')
                        ->count();
                    $unlocked = $count >= $achievement->condition_value;
                    break;
                    
                case 'level_reached':
                    $unlocked = $user->level >= $achievement->condition_value;
                    break;
                    
                case 'gold_earned':
                    // This would need to be tracked separately or calculated from task completions
                    $unlocked = false; // Placeholder
                    break;
                    
                case 'battles_won':
                    $count = $user->battles()->where('result', 'win')->count();
                    $unlocked = $count >= $achievement->condition_value;
                    break;
            }
            
            if ($unlocked) {
                $user->achievements()->attach($achievement->id, [
                    'unlocked_at' => now(),
                ]);
                
                event(new \App\Events\AchievementUnlocked($user, $achievement));
                
                Log::info("User {$user->id} unlocked achievement: {$achievement->name}");
            }
        }
    }
}
