<?php

namespace App\Services;

use App\Models\User;
use App\Models\Level;
use App\Events\LevelUp;
use Illuminate\Support\Facades\Log;

class LevelService
{
    /**
     * Calculate XP required for next level.
     * Formula: floor(100 * level * (1.12)^(level-1))
     */
    public function xpForNextLevel(int $level): int
    {
        return (int) floor(100 * $level * pow(1.12, $level - 1));
    }

    /**
     * Add XP to user and check for level up.
     */
    public function addXpToUser(User $user, int $xp): array
    {
        $oldLevel = $user->level;
        $user->giveXp($xp);
        $user->refresh();

        $levelUpData = null;
        if ($user->level > $oldLevel) {
            $levelUpData = [
                'old_level' => $oldLevel,
                'new_level' => $user->level,
                'level_data' => Level::findByLevel($user->level),
            ];

            event(new LevelUp($user, $oldLevel, $user->level));

            Log::info("User {$user->id} leveled up from {$oldLevel} to {$user->level}");
        }

        return [
            'xp_added' => $xp,
            'current_xp' => $user->xp,
            'current_level' => $user->level,
            'level_up' => $levelUpData,
        ];
    }

    /**
     * Calculate total stat bonuses for a level.
     */
    public function calculateLevelStats(int $level): array
    {
        $levelData = Level::findByLevel($level);
        
        if (!$levelData) {
            return [
                'hp_bonus' => 0,
                'attack_bonus' => 0,
                'defense_bonus' => 0,
                'stamina_bonus' => 0,
            ];
        }

        return [
            'hp_bonus' => $levelData->hp_bonus,
            'attack_bonus' => $levelData->attack_bonus,
            'defense_bonus' => $levelData->defense_bonus,
            'stamina_bonus' => $levelData->stamina_bonus,
        ];
    }

    /**
     * Determine current level based on total XP.
     */
    public function getLevelByXp(int $totalXp): int
    {
        $level = 1;
        $totalXpNeeded = 0;

        while ($totalXpNeeded <= $totalXp && $level < 200) {
            $totalXpNeeded += $this->xpForNextLevel($level);
            if ($totalXpNeeded <= $totalXp) {
                $level++;
            } else {
                break;
            }
        }

        return $level;
    }
}


