<?php

namespace App\Services;

use App\Models\Mob;
use App\Models\User;
use App\Models\ShopItem;
use App\Services\VipService;
use App\Services\CurrencyService;
use Illuminate\Support\Facades\Log;

class LootService
{
    protected VipService $vipService;
    protected CurrencyService $currencyService;

    public function __construct(VipService $vipService, CurrencyService $currencyService)
    {
        $this->vipService = $vipService;
        $this->currencyService = $currencyService;
    }

    /**
     * Generate loot from mob defeat.
     */
    public function generateLoot(Mob $mob, User $user): array
    {
        $rewards = [
            'xp' => $mob->xp_reward,
            'gold' => $mob->gold_reward,
            'premium_coins' => 0,
            'items' => [],
        ];

        // Roll for premium coins
        $premiumCoins = $this->rollPremiumCoins($mob);
        if ($premiumCoins > 0) {
            $rewards['premium_coins'] = $premiumCoins;
        }

        // Roll for item drops
        $items = $this->rollItemDrops($mob);
        if (!empty($items)) {
            $rewards['items'] = $items;
        }

        // Apply VIP multipliers
        $rewards = $this->applyVipMultipliers($user, $rewards);

        return $rewards;
    }

    /**
     * Roll for premium coins based on mob's chance.
     */
    public function rollPremiumCoins(Mob $mob): int
    {
        $chance = (float) $mob->premium_coin_chance;
        $roll = mt_rand(1, 10000) / 100; // 0.01% precision

        if ($roll <= $chance) {
            return mt_rand(1, 3); // 1-3 premium coins
        }

        return 0;
    }

    /**
     * Roll for item drops based on mob's loot table.
     */
    public function rollItemDrops(Mob $mob): array
    {
        $lootTable = $mob->loot_table ?? [];
        $drops = [];

        // Check common drop
        if (isset($lootTable['common_drop'])) {
            $chance = $lootTable['common_drop']['chance'] ?? 0;
            if (mt_rand(1, 10000) / 100 <= $chance * 100) {
                // In a real implementation, would reference shop item IDs
                $drops[] = [
                    'type' => 'common',
                    'item_id' => $lootTable['common_drop']['item_id'] ?? null,
                ];
            }
        }

        // Check rare drop
        if (isset($lootTable['rare_drop'])) {
            $chance = $lootTable['rare_drop']['chance'] ?? 0;
            if (mt_rand(1, 10000) / 100 <= $chance * 100) {
                $drops[] = [
                    'type' => 'rare',
                    'item_id' => $lootTable['rare_drop']['item_id'] ?? null,
                ];
            }
        }

        return $drops;
    }

    /**
     * Apply VIP multipliers to rewards.
     */
    public function applyVipMultipliers(User $user, array $rewards): array
    {
        if (!$this->vipService->isVip($user)) {
            return $rewards;
        }

        $membership = $this->vipService->getActiveMembership($user);
        if (!$membership) {
            return $rewards;
        }

        // Apply XP and gold multipliers
        $multipliers = $this->vipService->applyVipMultipliers($user, $rewards['xp'], $rewards['gold']);
        
        $rewards['xp'] = $multipliers['xp'];
        $rewards['gold'] = $multipliers['gold'];
        $rewards['xp_bonus'] = $multipliers['xp_bonus'];
        $rewards['gold_bonus'] = $multipliers['gold_bonus'];

        return $rewards;
    }
}


