<?php

namespace App\Services;

use App\Models\User;
use App\Models\Membership;
use App\Models\CoinPack;
use App\Models\Transaction;
use App\Services\VipService;
use App\Services\CurrencyService;
use PayPal\Rest\ApiContext;
use PayPal\Auth\OAuthTokenCredential;
use PayPal\Api\Amount;
use PayPal\Api\Payer;
use PayPal\Api\Payment;
use PayPal\Api\RedirectUrls;
use PayPal\Api\Transaction as PayPalTransaction;
use PayPal\Api\PaymentExecution;
use PayPal\Exception\PayPalConnectionException;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Config;

class PayPalPaymentService
{
    protected VipService $vipService;
    protected CurrencyService $currencyService;
    protected ApiContext $apiContext;

    public function __construct(VipService $vipService, CurrencyService $currencyService)
    {
        $this->vipService = $vipService;
        $this->currencyService = $currencyService;
        $this->initializeApiContext();
    }

    /**
     * Initialize PayPal API Context.
     */
    protected function initializeApiContext(): void
    {
        $clientId = config('paypal.client_id');
        $clientSecret = config('paypal.client_secret');
        $mode = config('paypal.mode', 'sandbox');

        if (empty($clientId) || empty($clientSecret)) {
            throw new \Exception('PayPal credentials are not configured. Please set PAYPAL_CLIENT_ID and PAYPAL_CLIENT_SECRET in your .env file.');
        }

        $this->apiContext = new ApiContext(
            new OAuthTokenCredential($clientId, $clientSecret)
        );

        $this->apiContext->setConfig([
            'mode' => $mode,
            'http.ConnectionTimeOut' => 30,
            'http.Retry' => 1,
            'log.LogEnabled' => true,
            'log.FileName' => storage_path('logs/paypal.log'),
            'log.LogLevel' => 'FINE',
        ]);
    }

    /**
     * Create PayPal payment for VIP subscription.
     */
    public function createVipSubscriptionPayment(User $user, Membership $membership, string $period = 'monthly'): array
    {
        try {
            $amount = $period === 'monthly' ? $membership->monthly_price : ($membership->yearly_price ?? $membership->monthly_price * 12);

            // Create transaction record
            $transaction = Transaction::create([
                'user_id' => $user->id,
                'type' => Transaction::TYPE_VIP,
                'gateway' => 'paypal',
                'amount' => $amount,
                'currency' => config('paypal.currency'),
                'status' => Transaction::STATUS_PENDING,
                'membership_id' => $membership->id,
                'metadata' => [
                    'period' => $period,
                    'membership_name' => $membership->name,
                ],
            ]);

            // Create PayPal payment
            $payer = new Payer();
            $payer->setPaymentMethod('paypal');

            $amountObj = new Amount();
            $amountObj->setCurrency(config('paypal.currency'))
                     ->setTotal(number_format($amount, 2, '.', ''));

            $paypalTransaction = new PayPalTransaction();
            $paypalTransaction->setAmount($amountObj);
            $paypalTransaction->setDescription("VIP Subscription: {$membership->name} ({$period})");

            $redirectUrls = new RedirectUrls();
            $redirectUrls->setReturnUrl(config('paypal.return_url') . '?transaction_id=' . $transaction->id);
            $redirectUrls->setCancelUrl(config('paypal.cancel_url') . '?transaction_id=' . $transaction->id);

            $payment = new Payment();
            $payment->setIntent('sale');
            $payment->setPayer($payer);
            $payment->setRedirectUrls($redirectUrls);
            $payment->setTransactions([$paypalTransaction]);

            $payment->create($this->apiContext);

            // Update transaction with PayPal order ID
            $transaction->paypal_order_id = $payment->getId();
            $transaction->metadata = array_merge($transaction->metadata ?? [], [
                'paypal_payment_id' => $payment->getId(),
                'approval_url' => $payment->getApprovalLink(),
            ]);
            $transaction->save();

            Log::info("Created PayPal payment for VIP subscription", [
                'user_id' => $user->id,
                'transaction_id' => $transaction->id,
                'paypal_order_id' => $payment->getId(),
            ]);

            return [
                'success' => true,
                'approval_url' => $payment->getApprovalLink(),
                'transaction_id' => $transaction->id,
            ];
        } catch (PayPalConnectionException $e) {
            Log::error('PayPal connection error', [
                'error' => $e->getData(),
                'user_id' => $user->id,
            ]);

            return [
                'success' => false,
                'error' => 'Failed to create PayPal payment. Please try again.',
            ];
        } catch (\Exception $e) {
            Log::error('PayPal payment creation error', [
                'error' => $e->getMessage(),
                'user_id' => $user->id,
            ]);

            return [
                'success' => false,
                'error' => 'An error occurred. Please try again.',
            ];
        }
    }

    /**
     * Create PayPal payment for coin pack.
     */
    public function createCoinPackPayment(User $user, CoinPack $pack): array
    {
        try {
            // Create transaction record
            $transaction = Transaction::create([
                'user_id' => $user->id,
                'type' => Transaction::TYPE_COINS,
                'gateway' => 'paypal',
                'amount' => $pack->price_usd,
                'currency' => config('paypal.currency'),
                'status' => Transaction::STATUS_PENDING,
                'coin_pack_id' => $pack->id,
                'coins_amount' => $pack->total_coins,
                'metadata' => [
                    'pack_name' => $pack->name,
                    'base_coins' => $pack->coins_amount,
                    'bonus_coins' => $pack->bonus_coins,
                ],
            ]);

            // Create PayPal payment
            $payer = new Payer();
            $payer->setPaymentMethod('paypal');

            $amountObj = new Amount();
            $amountObj->setCurrency(config('paypal.currency'))
                     ->setTotal(number_format($pack->price_usd, 2, '.', ''));

            $paypalTransaction = new PayPalTransaction();
            $paypalTransaction->setAmount($amountObj);
            $paypalTransaction->setDescription("Premium Coins: {$pack->name} ({$pack->total_coins} coins)");

            $redirectUrls = new RedirectUrls();
            $redirectUrls->setReturnUrl(config('paypal.coins_return_url') . '?transaction_id=' . $transaction->id);
            $redirectUrls->setCancelUrl(config('paypal.coins_cancel_url') . '?transaction_id=' . $transaction->id);

            $payment = new Payment();
            $payment->setIntent('sale');
            $payment->setPayer($payer);
            $payment->setRedirectUrls($redirectUrls);
            $payment->setTransactions([$paypalTransaction]);

            $payment->create($this->apiContext);

            // Update transaction with PayPal order ID
            $transaction->paypal_order_id = $payment->getId();
            $transaction->metadata = array_merge($transaction->metadata ?? [], [
                'paypal_payment_id' => $payment->getId(),
                'approval_url' => $payment->getApprovalLink(),
            ]);
            $transaction->save();

            Log::info("Created PayPal payment for coin pack", [
                'user_id' => $user->id,
                'transaction_id' => $transaction->id,
                'paypal_order_id' => $payment->getId(),
            ]);

            return [
                'success' => true,
                'approval_url' => $payment->getApprovalLink(),
                'transaction_id' => $transaction->id,
            ];
        } catch (PayPalConnectionException $e) {
            Log::error('PayPal connection error for coin pack', [
                'error' => $e->getData(),
                'user_id' => $user->id,
            ]);

            return [
                'success' => false,
                'error' => 'Failed to create PayPal payment. Please try again.',
            ];
        } catch (\Exception $e) {
            Log::error('PayPal coin pack payment creation error', [
                'error' => $e->getMessage(),
                'user_id' => $user->id,
            ]);

            return [
                'success' => false,
                'error' => 'An error occurred. Please try again.',
            ];
        }
    }

    /**
     * Capture PayPal payment.
     */
    public function capturePayment(string $orderId, string $payerId): array
    {
        try {
            $payment = Payment::get($orderId, $this->apiContext);

            $execution = new PaymentExecution();
            $execution->setPayerId($payerId);

            $payment->execute($execution, $this->apiContext);

            $transaction = $payment->getTransactions()[0];
            $relatedResources = $transaction->getRelatedResources()[0];
            $sale = $relatedResources->getSale();

            Log::info("Captured PayPal payment", [
                'order_id' => $orderId,
                'sale_id' => $sale->getId(),
            ]);

            return [
                'success' => true,
                'payment' => $payment,
                'sale_id' => $sale->getId(),
            ];
        } catch (\Exception $e) {
            Log::error('PayPal payment capture error', [
                'error' => $e->getMessage(),
                'order_id' => $orderId,
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    /**
     * Verify webhook signature (simplified - implement full verification in production).
     */
    public function verifyWebhookSignature(array $headers, string $body): bool
    {
        // TODO: Implement full webhook signature verification
        // For now, return true - implement proper verification using PayPal webhook verification API
        return true;
    }

    /**
     * Handle webhook event.
     */
    public function handleWebhookEvent(array $event): void
    {
        $eventType = $event['event_type'] ?? null;

        Log::info('PayPal webhook event received', [
            'event_type' => $eventType,
            'event' => $event,
        ]);

        switch ($eventType) {
            case 'PAYMENT.CAPTURE.COMPLETED':
                $this->handlePaymentCaptureCompleted($event);
                break;
            case 'BILLING.SUBSCRIPTION.CREATED':
                $this->handleSubscriptionCreated($event);
                break;
            case 'BILLING.SUBSCRIPTION.ACTIVATED':
                $this->handleSubscriptionActivated($event);
                break;
            case 'BILLING.SUBSCRIPTION.CANCELLED':
                $this->handleSubscriptionCancelled($event);
                break;
            case 'BILLING.SUBSCRIPTION.EXPIRED':
                $this->handleSubscriptionExpired($event);
                break;
            default:
                Log::info('Unhandled PayPal webhook event type', ['event_type' => $eventType]);
        }
    }

    /**
     * Handle successful VIP payment.
     */
    public function handleSuccessfulVipPayment(User $user, Transaction $transaction): void
    {
        if ($transaction->status === Transaction::STATUS_COMPLETED) {
            Log::warning('Transaction already processed', ['transaction_id' => $transaction->id]);
            return;
        }

        $membership = $transaction->membership;
        $period = $transaction->metadata['period'] ?? 'monthly';

        // Activate VIP
        $this->vipService->activateVip($user, $membership, $period);

        // Grant initial daily rewards
        $this->vipService->grantDailyVipRewards($user);

        // Update transaction
        $transaction->status = Transaction::STATUS_COMPLETED;
        $transaction->save();

        Log::info('VIP payment processed successfully', [
            'user_id' => $user->id,
            'transaction_id' => $transaction->id,
        ]);
    }

    /**
     * Handle successful coins payment.
     */
    public function handleSuccessfulCoinsPayment(User $user, Transaction $transaction): void
    {
        if ($transaction->status === Transaction::STATUS_COMPLETED) {
            Log::warning('Transaction already processed', ['transaction_id' => $transaction->id]);
            return;
        }

        $coinsAmount = $transaction->coins_amount ?? 0;

        if ($coinsAmount > 0) {
            $this->currencyService->addPremiumCoins($user, $coinsAmount);
        }

        // Update transaction
        $transaction->status = Transaction::STATUS_COMPLETED;
        $transaction->save();

        Log::info('Coins payment processed successfully', [
            'user_id' => $user->id,
            'transaction_id' => $transaction->id,
            'coins_amount' => $coinsAmount,
        ]);
    }

    /**
     * Cancel PayPal subscription.
     */
    public function cancelSubscription(string $subscriptionId): bool
    {
        // TODO: Implement subscription cancellation via PayPal API
        // For now, just log it
        Log::info('Subscription cancellation requested', ['subscription_id' => $subscriptionId]);
        return true;
    }

    /**
     * Handle PAYMENT.CAPTURE.COMPLETED webhook event.
     */
    protected function handlePaymentCaptureCompleted(array $event): void
    {
        $resource = $event['resource'] ?? [];
        $orderId = $resource['supplementary_data']['related_ids']['order_id'] ?? null;

        if (!$orderId) {
            Log::warning('No order ID in payment capture event', ['event' => $event]);
            return;
        }

        $transaction = Transaction::where('paypal_order_id', $orderId)->first();

        if (!$transaction) {
            Log::warning('Transaction not found for order ID', ['order_id' => $orderId]);
            return;
        }

        $user = $transaction->user;

        if ($transaction->type === Transaction::TYPE_VIP) {
            $this->handleSuccessfulVipPayment($user, $transaction);
        } elseif ($transaction->type === Transaction::TYPE_COINS) {
            $this->handleSuccessfulCoinsPayment($user, $transaction);
        }
    }

    /**
     * Handle BILLING.SUBSCRIPTION.CREATED webhook event.
     */
    protected function handleSubscriptionCreated(array $event): void
    {
        // Create subscription transaction record
        $resource = $event['resource'] ?? [];
        $subscriptionId = $resource['id'] ?? null;

        Log::info('Subscription created', ['subscription_id' => $subscriptionId]);
    }

    /**
     * Handle BILLING.SUBSCRIPTION.ACTIVATED webhook event.
     */
    protected function handleSubscriptionActivated(array $event): void
    {
        $resource = $event['resource'] ?? [];
        $subscriptionId = $resource['id'] ?? null;

        $transaction = Transaction::where('paypal_subscription_id', $subscriptionId)->first();

        if ($transaction) {
            $user = $transaction->user;
            $membership = $transaction->membership;

            if ($membership) {
                $this->vipService->activateVip($user, $membership, 'monthly');
                $transaction->status = Transaction::STATUS_COMPLETED;
                $transaction->save();
            }
        }
    }

    /**
     * Handle BILLING.SUBSCRIPTION.CANCELLED webhook event.
     */
    protected function handleSubscriptionCancelled(array $event): void
    {
        $resource = $event['resource'] ?? [];
        $subscriptionId = $resource['id'] ?? null;

        $transaction = Transaction::where('paypal_subscription_id', $subscriptionId)->first();

        if ($transaction) {
            $user = $transaction->user;
            $this->vipService->deactivateVip($user);
        }
    }

    /**
     * Handle BILLING.SUBSCRIPTION.EXPIRED webhook event.
     */
    protected function handleSubscriptionExpired(array $event): void
    {
        $this->handleSubscriptionCancelled($event);
    }
}

