<?php

namespace App\Services;

use App\Models\TaskReminder;
use App\Models\Task;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class ReminderService
{
    /**
     * Process and send all due reminders.
     */
    public function processReminders(): int
    {
        $sentCount = 0;
        
        $reminders = TaskReminder::with(['task.category', 'task.user'])
            ->where('is_active', true)
            ->get();

        foreach ($reminders as $reminder) {
            if ($reminder->shouldSend()) {
                $this->sendReminder($reminder);
                $reminder->markAsSent();
                $sentCount++;
            }
        }

        Log::info("Processed reminders: {$sentCount} sent");
        
        return $sentCount;
    }

    /**
     * Send a reminder notification.
     */
    protected function sendReminder(TaskReminder $reminder): void
    {
        $task = $reminder->task;
        $user = $task->user;
        
        // For now, we'll log the reminder. In the future, this could send:
        // - Email notifications
        // - Push notifications
        // - In-app notifications
        // - SMS notifications
        
        Log::info("Reminder sent", [
            'user_id' => $user->id,
            'task_id' => $task->id,
            'reminder_type' => $reminder->reminder_type,
            'task_title' => $task->title,
        ]);

        // Store reminder notification in session or database for in-app display
        // This could be expanded to use Laravel's notification system
    }

    /**
     * Create reminders for a task based on configuration.
     */
    public function createRemindersForTask(Task $task, array $reminderConfigs): void
    {
        // Delete existing reminders for this task
        $task->reminders()->delete();

        foreach ($reminderConfigs as $config) {
            // Check if reminder is enabled (checkbox sends '1' or 'on' when checked, nothing when unchecked)
            $isEnabled = isset($config['enabled']) && ($config['enabled'] === true || $config['enabled'] === '1' || $config['enabled'] === 'on');
            
            if (!$isEnabled) {
                continue;
            }

            try {
                $reminder = new TaskReminder([
                    'task_id' => $task->id,
                    'reminder_type' => $config['type'] ?? 'green',
                    'schedule_type' => $config['schedule_type'] ?? 'daily',
                    'is_active' => true,
                ]);

                // Set reminder time - handle both time string and datetime
                $timeValue = $config['time'] ?? '09:00:00';
                if (is_string($timeValue)) {
                    $reminder->reminder_time = Carbon::createFromTimeString($timeValue);
                } else {
                    $reminder->reminder_time = Carbon::parse($timeValue);
                }

                // Handle schedule-specific data
                if (($config['schedule_type'] ?? 'daily') === 'weekly') {
                    $reminder->schedule_data = [
                        'days' => $config['days'] ?? [],
                    ];
                } else {
                    // Daily schedule
                    $reminder->days_before = isset($config['days_before']) && $config['days_before'] !== '' ? (int)$config['days_before'] : null;
                    $reminder->hours_before = isset($config['hours_before']) && $config['hours_before'] !== '' ? (int)$config['hours_before'] : null;
                }

                $reminder->save();
            } catch (\Exception $e) {
                Log::error('Failed to create reminder: ' . $e->getMessage(), [
                    'task_id' => $task->id,
                    'config' => $config,
                ]);
                // Continue with next reminder instead of failing completely
            }
        }
    }

    /**
     * Determine appropriate reminder schedule based on task type and due date.
     */
    public function getSuggestedReminders(Task $task): array
    {
        $suggestions = [];
        
        if (!$task->due_date) {
            return $suggestions;
        }

        $now = Carbon::now();
        $dueDate = Carbon::parse($task->due_date);
        $daysUntilDue = $now->diffInDays($dueDate, false);

        // For one-time tasks
        if ($task->category->repeat_type === 'once') {
            if ($daysUntilDue > 30) {
                // Far away - suggest weekly reminders
                $suggestions[] = [
                    'type' => 'green',
                    'schedule_type' => 'weekly',
                    'days' => [1, 3, 5], // Monday, Wednesday, Friday
                    'time' => '09:00:00',
                    'label' => 'Weekly reminder (plenty of time)',
                ];
            } elseif ($daysUntilDue > 7) {
                // Medium distance - suggest daily reminders
                $suggestions[] = [
                    'type' => 'green',
                    'schedule_type' => 'daily',
                    'days_before' => null,
                    'time' => '09:00:00',
                    'label' => 'Daily reminder',
                ];
            } else {
                // Close - suggest multiple daily reminders
                if ($daysUntilDue > 3) {
                    $suggestions[] = [
                        'type' => 'yellow',
                        'schedule_type' => 'daily',
                        'days_before' => null,
                        'time' => '09:00:00',
                        'label' => 'Daily reminder (running out of time)',
                    ];
                }
                if ($daysUntilDue > 1) {
                    $suggestions[] = [
                        'type' => 'red',
                        'schedule_type' => 'daily',
                        'days_before' => 1,
                        'time' => '09:00:00',
                        'label' => '1 day before reminder (urgent)',
                    ];
                }
                $suggestions[] = [
                    'type' => 'red',
                    'schedule_type' => 'daily',
                    'days_before' => 0,
                    'time' => '09:00:00',
                    'label' => 'Due date reminder (urgent)',
                ];
            }
        } else {
            // For repetitive tasks
            $suggestions[] = [
                'type' => 'green',
                'schedule_type' => 'daily',
                'days_before' => null,
                'time' => '09:00:00',
                'label' => 'Daily reminder',
            ];
        }

        return $suggestions;
    }
}

