<?php

namespace App\Services;

use App\Models\User;
use App\Models\Item;
use App\Services\CurrencyService;
use App\Services\VipService;
use Illuminate\Support\Facades\Log;

class ShopService
{
    protected CurrencyService $currencyService;
    protected VipService $vipService;

    public function __construct(CurrencyService $currencyService, VipService $vipService)
    {
        $this->currencyService = $currencyService;
        $this->vipService = $vipService;
    }

    /**
     * Buy an item with gold.
     */
    public function buyWithGold(User $user, Item $item): bool
    {
        // Check if VIP-only item
        if ($item->is_vip_only && !$this->vipService->isVip($user)) {
            return false;
        }

        // Check level requirement
        if ($user->level < $item->required_level) {
            return false;
        }

        // Check if user already owns this item
        if ($user->userItems()->where('item_id', $item->id)->exists()) {
            return false;
        }

        // Check if user has enough gold
        if ($user->gold < $item->cost) {
            return false;
        }

        // Deduct gold and add item
        if (!$this->currencyService->deductGold($user, $item->cost)) {
            return false;
        }

        $user->userItems()->create([
            'item_id' => $item->id,
            'equipped' => false,
            'purchased_at' => now(),
        ]);

        Log::info("User {$user->id} purchased item {$item->id} with gold");

        return true;
    }

    /**
     * Buy an item with premium coins.
     */
    public function buyWithPremiumCoins(User $user, Item $item): bool
    {
        // Check if it's a premium item
        if (!$item->is_premium || !$item->premium_coins_cost) {
            return false;
        }

        // Check level requirement
        if ($user->level < $item->required_level) {
            return false;
        }

        // Check if user already owns this item
        if ($user->userItems()->where('item_id', $item->id)->exists()) {
            return false;
        }

        // Check if user has enough premium coins
        if (!$this->currencyService->hasEnoughPremiumCoins($user, $item->premium_coins_cost)) {
            return false;
        }

        // Deduct premium coins and add item
        if (!$this->currencyService->deductPremiumCoins($user, $item->premium_coins_cost)) {
            return false;
        }

        $user->userItems()->create([
            'item_id' => $item->id,
            'equipped' => false,
            'purchased_at' => now(),
        ]);

        Log::info("User {$user->id} purchased item {$item->id} with premium coins");

        return true;
    }

    /**
     * Get VIP exclusive items.
     */
    public function getVipExclusiveItems()
    {
        return Item::where('is_vip_only', true)->get();
    }

    /**
     * Get premium items.
     */
    public function getPremiumItems()
    {
        return Item::where('is_premium', true)->get();
    }

    /**
     * Check if user can purchase an item.
     */
    public function canPurchaseItem(User $user, Item $item): array
    {
        $canPurchase = true;
        $reasons = [];

        // Check level requirement
        if ($user->level < $item->required_level) {
            $canPurchase = false;
            $reasons[] = "Level {$item->required_level} required";
        }

        // Check if already owned
        if ($user->userItems()->where('item_id', $item->id)->exists()) {
            $canPurchase = false;
            $reasons[] = "Already owned";
        }

        // Check VIP requirement
        if ($item->is_vip_only && !$this->vipService->isVip($user)) {
            $canPurchase = false;
            $reasons[] = "VIP membership required";
        }

        // Check currency
        if ($item->is_premium && $item->premium_coins_cost) {
            if (!$this->currencyService->hasEnoughPremiumCoins($user, $item->premium_coins_cost)) {
                $canPurchase = false;
                $reasons[] = "Insufficient premium coins";
            }
        } else {
            if ($user->gold < $item->cost) {
                $canPurchase = false;
                $reasons[] = "Insufficient gold";
            }
        }

        return [
            'can_purchase' => $canPurchase,
            'reasons' => $reasons,
        ];
    }
}

