<?php

namespace App\Services;

use App\Models\Task;
use App\Models\Category;
use App\Services\GameService;
use App\Services\VipService;
use App\Services\CurrencyService;
use App\Events\TaskCompleted;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class TaskService
{
    protected GameService $gameService;
    protected VipService $vipService;
    protected CurrencyService $currencyService;

    public function __construct(GameService $gameService, VipService $vipService, CurrencyService $currencyService)
    {
        $this->gameService = $gameService;
        $this->vipService = $vipService;
        $this->currencyService = $currencyService;
    }

    /**
     * Complete a task and award XP/gold.
     */
    public function completeTask(Task $task): array
    {
        if ($task->status === 'completed') {
            return ['error' => 'Task already completed'];
        }

        $task->status = 'completed';
        $task->completed_at = now();
        $task->save();

        // Calculate XP based on task completion percentage (checklists)
        $baseXP = $this->calculateTaskXP($task);
        $baseGold = $this->gameService->calculateGoldForTask($task->points);

        // Apply streak bonus for daily quests
        $streakBonus = 0;
        $streakData = null;
        if ($task->category->repeat_type === 'daily') {
            $streakData = $this->gameService->calculateDailyQuestStreak($task->user);
            $bonusedRewards = $this->gameService->applyStreakBonus($baseXP, $baseGold, $streakData);
            $xpAmount = $bonusedRewards['xp'];
            $goldAmount = $bonusedRewards['gold'];
            $streakBonus = $streakData['bonus_percent'];
        } else {
            $xpAmount = $baseXP;
            $goldAmount = $baseGold;
        }

        // Apply VIP multipliers if user is VIP
        $vipMultipliers = $this->vipService->applyVipMultipliers($task->user, $xpAmount, $goldAmount);
        $finalXP = $vipMultipliers['xp'];
        $finalGold = $vipMultipliers['gold'];
        $vipXpBonus = $vipMultipliers['xp_bonus'];
        $vipGoldBonus = $vipMultipliers['gold_bonus'];

        // Award XP and gold
        $levelResult = $this->gameService->addXP($task->user, $finalXP);
        $this->gameService->addGold($task->user, $finalGold);

        // Award coins based on task difficulty level
        $coinsEarned = $task->coins ?? 0;
        if ($coinsEarned > 0) {
            $this->currencyService->addPremiumCoins($task->user, $coinsEarned);
        }

        // Fire event
        event(new TaskCompleted($task, $finalXP, $finalGold));

        // Check achievements
        $this->gameService->checkAchievements($task->user, 'tasks_completed');
        $this->gameService->checkAchievements($task->user, 'daily_tasks_completed');

        return [
            'success' => true,
            'xp_earned' => $finalXP,
            'gold_earned' => $finalGold,
            'base_xp' => $baseXP,
            'base_gold' => $baseGold,
            'streak_bonus' => $streakBonus,
            'streak_days' => $streakData['streak'] ?? 0,
            'vip_xp_bonus' => $vipXpBonus,
            'vip_gold_bonus' => $vipGoldBonus,
            'coins_earned' => $coinsEarned,
            'level_up' => $levelResult['leveled_up'],
            'new_level' => $levelResult['new_level'],
        ];
    }

    /**
     * Calculate XP for a task based on completion percentage.
     */
    protected function calculateTaskXP(Task $task): int
    {
        $baseXP = $this->gameService->calculateXPForTask($task->category, $task->points);
        
        // If task has checklists, calculate completion percentage
        if ($task->checklists()->count() > 0) {
            $completed = $task->checklists()->where('completed', true)->count();
            $total = $task->checklists()->count();
            $percentage = $completed / $total;
            
            $baseXP = (int) round($baseXP * $percentage);
        }
        
        return max(1, $baseXP); // Minimum 1 XP
    }

    /**
     * Regenerate daily tasks.
     */
    public function regenerateDailyTasks(): int
    {
        $count = 0;
        
        // Get all users
        $users = \App\Models\User::all();
        
        foreach ($users as $user) {
            // Get all daily categories for this user
            $dailyCategories = $user->categories()->where('repeat_type', 'daily')->get();
            
            foreach ($dailyCategories as $category) {
                // Get completed tasks from yesterday that need regeneration
                $yesterday = Carbon::yesterday();
                
                $completedTasks = $category->tasks()
                    ->where('status', 'completed')
                    ->whereDate('completed_at', $yesterday)
                    ->get();
                
                foreach ($completedTasks as $completedTask) {
                    // Create new task for today
                    $newTask = Task::create([
                        'title' => $completedTask->title,
                        'description' => $completedTask->description,
                        'category_id' => $category->id,
                        'user_id' => $user->id,
                        'repeat_rule' => $completedTask->repeat_rule,
                        'due_date' => Carbon::today()->endOfDay(),
                        'points' => $completedTask->points,
                        'difficulty_level' => $completedTask->difficulty_level ?? 'medium',
                        'coins' => $completedTask->coins ?? 0,
                        'status' => 'pending',
                    ]);
                    
                    // Copy checklists if any
                    foreach ($completedTask->checklists as $checklist) {
                        \App\Models\TaskChecklist::create([
                            'task_id' => $newTask->id,
                            'title' => $checklist->title,
                            'completed' => false,
                            'order' => $checklist->order,
                        ]);
                    }
                    
                    $count++;
                }
            }
        }
        
        Log::info("Regenerated {$count} daily tasks");
        
        return $count;
    }

    /**
     * Regenerate weekly tasks.
     */
    public function regenerateWeeklyTasks(): int
    {
        $count = 0;
        
        // Get all users
        $users = \App\Models\User::all();
        
        foreach ($users as $user) {
            // Get all weekly categories for this user
            $weeklyCategories = $user->categories()->where('repeat_type', 'weekly')->get();
            
            foreach ($weeklyCategories as $category) {
                // Check if we need to regenerate (every Monday)
                if (Carbon::now()->isMonday()) {
                    // Get completed tasks from last week
                    $lastWeekStart = Carbon::now()->subWeek()->startOfWeek();
                    $lastWeekEnd = Carbon::now()->subWeek()->endOfWeek();
                    
                    $completedTasks = $category->tasks()
                        ->where('status', 'completed')
                        ->whereBetween('completed_at', [$lastWeekStart, $lastWeekEnd])
                        ->get();
                    
                    foreach ($completedTasks as $completedTask) {
                        // Create new task for this week
                        $newTask = Task::create([
                            'title' => $completedTask->title,
                            'description' => $completedTask->description,
                            'category_id' => $category->id,
                            'user_id' => $user->id,
                            'repeat_rule' => $completedTask->repeat_rule,
                            'due_date' => Carbon::now()->endOfWeek(),
                            'points' => $completedTask->points,
                            'difficulty_level' => $completedTask->difficulty_level ?? 'medium',
                            'coins' => $completedTask->coins ?? 0,
                            'status' => 'pending',
                        ]);
                        
                        // Copy checklists if any
                        foreach ($completedTask->checklists as $checklist) {
                            \App\Models\TaskChecklist::create([
                                'task_id' => $newTask->id,
                                'title' => $checklist->title,
                                'completed' => false,
                                'order' => $checklist->order,
                            ]);
                        }
                        
                        $count++;
                    }
                }
            }
        }
        
        Log::info("Regenerated {$count} weekly tasks");
        
        return $count;
    }
}

