<?php

namespace App\Services;

use App\Models\User;
use App\Models\Membership;
use App\Services\CurrencyService;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class VipService
{
    protected CurrencyService $currencyService;

    public function __construct(CurrencyService $currencyService)
    {
        $this->currencyService = $currencyService;
    }

    /**
     * Activate VIP for a user with a membership.
     */
    public function activateVip(User $user, Membership $membership, string $period = 'monthly'): void
    {
        $expiresAt = null;
        
        if ($period === 'monthly') {
            $expiresAt = now()->addMonth();
        } elseif ($period === 'yearly') {
            $expiresAt = now()->addYear();
        }

        $user->is_vip = true;
        $user->vip_expires_at = $expiresAt;
        $user->membership_id = $membership->id;
        $user->save();

        Log::info("VIP activated for user {$user->id} with membership {$membership->id}, expires at {$expiresAt}");
    }

    /**
     * Deactivate VIP for a user.
     */
    public function deactivateVip(User $user): void
    {
        $user->is_vip = false;
        $user->vip_expires_at = null;
        $user->membership_id = null;
        $user->save();

        Log::info("VIP deactivated for user {$user->id}");
    }

    /**
     * Check if user is currently VIP.
     */
    public function isVip(User $user): bool
    {
        return $user->isVip();
    }

    /**
     * Grant daily VIP rewards to a user.
     */
    public function grantDailyVipRewards(User $user): bool
    {
        if (!$this->isVip($user)) {
            return false;
        }

        $membership = $this->getActiveMembership($user);
        if (!$membership) {
            return false;
        }

        // Check if already claimed today
        if ($user->last_vip_reward_claimed_at && 
            $user->last_vip_reward_claimed_at->isToday()) {
            return false;
        }

        // Grant rewards
        if ($membership->daily_gold_bonus > 0) {
            $this->currencyService->addGold($user, $membership->daily_gold_bonus);
        }

        if ($membership->daily_premium_coins > 0) {
            $this->currencyService->addPremiumCoins($user, $membership->daily_premium_coins);
        }

        $user->last_vip_reward_claimed_at = now();
        $user->save();

        Log::info("Daily VIP rewards granted to user {$user->id}: {$membership->daily_gold_bonus} gold, {$membership->daily_premium_coins} premium coins");

        return true;
    }

    /**
     * Apply VIP multipliers to XP and gold.
     */
    public function applyVipMultipliers(User $user, int $xp, int $gold): array
    {
        if (!$this->isVip($user)) {
            return [
                'xp' => $xp,
                'gold' => $gold,
                'xp_bonus' => 0,
                'gold_bonus' => 0,
            ];
        }

        $membership = $this->getActiveMembership($user);
        if (!$membership) {
            return [
                'xp' => $xp,
                'gold' => $gold,
                'xp_bonus' => 0,
                'gold_bonus' => 0,
            ];
        }

        $xpBonus = (int) round($xp * ($membership->xp_bonus_percent / 100));
        $goldBonus = (int) round($gold * ($membership->gold_bonus_percent / 100));

        return [
            'xp' => $xp + $xpBonus,
            'gold' => $gold + $goldBonus,
            'xp_bonus' => $xpBonus,
            'gold_bonus' => $goldBonus,
        ];
    }

    /**
     * Get active membership for a user.
     */
    public function getActiveMembership(User $user): ?Membership
    {
        if (!$this->isVip($user)) {
            return null;
        }

        return $user->membership;
    }
}

