<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Mob;
use Illuminate\Support\Str;

class MobSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Seed RNG for deterministic generation
        mt_srand(42);

        // Egyptian-themed name parts
        $prefixes = [
            'Sand', 'Temple', 'Desert', 'Ancient', 'Sacred', 'Golden', 'Royal', 'Divine',
            'Cursed', 'Mystic', 'Pharaoh', 'Nile', 'Pyramid', 'Sphinx', 'Scarab', 'Anubis',
            'Horus', 'Ra', 'Osiris', 'Isis', 'Seth', 'Thoth', 'Bastet', 'Sobek',
        ];

        $suffixes = [
            'Scorpion', 'Warden', 'Sentinel', 'Guardian', 'Warrior', 'Priest', 'Scout', 'Hunter',
            'Beast', 'Spirit', 'Phantom', 'Shadow', 'Servant', 'Slave', 'Soldier', 'Champion',
            'Titan', 'Colossus', 'Giant', 'Monster', 'Demon', 'Serpent', 'Cobra', 'Vulture',
        ];

        $mobs = [];
        $mobCounter = 0;

        // Distribution: Levels 1-10: 300 mobs
        for ($i = 0; $i < 300; $i++) {
            $level = mt_rand(1, 10);
            $mobs[] = $this->generateMob($level, 'common', $prefixes, $suffixes, $mobCounter++);
        }

        // Levels 11-20: 250 mobs (common/rare)
        for ($i = 0; $i < 250; $i++) {
            $level = mt_rand(11, 20);
            $rarity = mt_rand(1, 100) <= 70 ? 'common' : 'rare';
            $mobs[] = $this->generateMob($level, $rarity, $prefixes, $suffixes, $mobCounter++);
        }

        // Levels 21-40: 250 mobs (rare/epic)
        for ($i = 0; $i < 250; $i++) {
            $level = mt_rand(21, 40);
            $rarityRoll = mt_rand(1, 100);
            $rarity = $rarityRoll <= 40 ? 'rare' : 'epic';
            $mobs[] = $this->generateMob($level, $rarity, $prefixes, $suffixes, $mobCounter++);
        }

        // Levels 41-60: 150 mobs (epic/legendary)
        for ($i = 0; $i < 150; $i++) {
            $level = mt_rand(41, 60);
            $rarityRoll = mt_rand(1, 100);
            $rarity = $rarityRoll <= 60 ? 'epic' : 'legendary';
            $mobs[] = $this->generateMob($level, $rarity, $prefixes, $suffixes, $mobCounter++);
        }

        // Levels 61-80: 50 mobs (legendary/boss)
        for ($i = 0; $i < 50; $i++) {
            $level = mt_rand(61, 80);
            $rarity = 'legendary';
            $mobs[] = $this->generateMob($level, $rarity, $prefixes, $suffixes, $mobCounter++);
        }

        // Insert in chunks of 100
        foreach (array_chunk($mobs, 100) as $chunk) {
            Mob::insert($chunk);
        }
    }

    /**
     * Generate a single mob with given parameters.
     */
    protected function generateMob(int $level, string $rarity, array $prefixes, array $suffixes, int $counter): array
    {
        // Generate unique name
        $prefix = $prefixes[mt_rand(0, count($prefixes) - 1)];
        $suffix = $suffixes[mt_rand(0, count($suffixes) - 1)];
        $name = "{$prefix} {$suffix}";
        
        // Ensure unique name by appending number if needed
        if ($counter > 0 && $counter % 10 === 0) {
            $name .= " " . ($counter / 10);
        }
        
        $slug = Str::slug($name . '-' . $counter);

        // Calculate stats
        $baseHp = 50 + ($level * 8);
        $hp = (int) round($baseHp * mt_rand(90, 130) / 100);
        
        $attack = (int) round(5 + ($level * 1.5) + mt_rand(-2, 5));
        $defense = (int) round(2 + ($level * 1.2) + mt_rand(-1, 4));
        $speed = 10 + mt_rand(-3, 6);
        $luck = max(0, min(20, floor($level / 10) + mt_rand(0, 2)));

        $xpReward = (int) round($level * 20 * mt_rand(90, 120) / 100);
        $goldReward = (int) round($level * 8 * mt_rand(90, 130) / 100);
        
        $premiumCoinChance = min(0.05, 0.005 + ($level * 0.0005));

        // Determine AI type based on level and rarity
        $aiTypes = ['passive', 'aggressive', 'strategic'];
        if ($level < 20) {
            $aiType = $aiTypes[mt_rand(0, 1)]; // passive or aggressive
        } else {
            $aiType = $aiTypes[mt_rand(1, 2)]; // aggressive or strategic
        }

        // Create loot table
        $lootTable = [
            'common_drop' => [
                'chance' => 0.15,
                'item_id' => null, // Will reference shop items
            ],
            'rare_drop' => [
                'chance' => $rarity === 'rare' || $rarity === 'epic' || $rarity === 'legendary' ? 0.10 : 0.05,
                'item_id' => null,
            ],
        ];

        return [
            'name' => $name,
            'slug' => $slug,
            'level' => $level,
            'hp' => $hp,
            'attack' => $attack,
            'defense' => $defense,
            'speed' => $speed,
            'luck' => $luck,
            'xp_reward' => $xpReward,
            'gold_reward' => $goldReward,
            'premium_coin_chance' => $premiumCoinChance,
            'rarity' => $rarity,
            'loot_table' => json_encode($lootTable),
            'ai_type' => $aiType,
            'sprite_source' => "/assets/sprites/mobs/{$slug}.png",
            'created_at' => now(),
            'updated_at' => now(),
        ];
    }
}
