<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\ShopItem;
use Illuminate\Support\Str;

class ShopItemSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $items = [];
        $itemCounter = 0;

        // Egyptian-themed name arrays
        $weaponNames = [
            'Horus Skyblade', 'Anubis Scythe', 'Ra Sunsword', 'Osiris Staff', 'Seth Warhammer',
            'Bastet Claws', 'Thoth Quill', 'Sobek Trident', 'Isis Wand', 'Scarab Dagger',
            'Pharaoh Blade', 'Nile Spear', 'Pyramid Axe', 'Sphinx Bow', 'Desert Scimitar',
        ];

        $armorNames = [
            'Pharaoh Armor', 'Ra Regalia', 'Anubis Guard', 'Horus Wings', 'Osiris Robes',
            'Temple Plate', 'Sacred Mail', 'Divine Shield', 'Golden Helm', 'Royal Cuirass',
        ];

        $relicNames = [
            'Eye of Horus', 'Ankh of Life', 'Scarab Amulet', 'Papyrus Charm', 'Nile Stone',
            'Pyramid Fragment', 'Sphinx Riddle', 'Pharaoh Crown', 'Ra Sun Disk', 'Osiris Heart',
        ];

        $potionNames = [
            'Elixir of the Nile', 'Ra Blessing', 'Anubis Potion', 'Horus Tonic', 'Osiris Brew',
            'Sacred Water', 'Divine Nectar', 'Temple Essence', 'Pharaoh Draught', 'Golden Serum',
        ];

        $skinNames = [
            'Pharaoh Skin', 'Ra Aura', 'Anubis Form', 'Horus Wings', 'Osiris Shroud',
        ];

        // Generate 30 weapons
        for ($i = 0; $i < 30; $i++) {
            $items[] = $this->generateWeapon($weaponNames, $itemCounter++);
        }

        // Generate 25 armor
        for ($i = 0; $i < 25; $i++) {
            $items[] = $this->generateArmor($armorNames, $itemCounter++);
        }

        // Generate 15 relics
        for ($i = 0; $i < 15; $i++) {
            $items[] = $this->generateRelic($relicNames, $itemCounter++);
        }

        // Generate 20 potions
        for ($i = 0; $i < 20; $i++) {
            $items[] = $this->generatePotion($potionNames, $itemCounter++);
        }

        // Generate 10 skins
        for ($i = 0; $i < 10; $i++) {
            $items[] = $this->generateSkin($skinNames, $itemCounter++);
        }

        // Insert in chunks of 50
        foreach (array_chunk($items, 50) as $chunk) {
            ShopItem::insert($chunk);
        }
    }

    /**
     * Generate a weapon item.
     */
    protected function generateWeapon(array $names, int $counter): array
    {
        $rarity = $this->determineRarity($counter, 30);
        $name = $names[$counter % count($names)] . ($counter > count($names) ? ' ' . ($counter / count($names) + 1) : '');
        
        return [
            'name' => $name,
            'slug' => Str::slug($name . '-' . $counter),
            'type' => 'weapon',
            'rarity' => $rarity,
            'attack_bonus' => $this->getAttackBonus($rarity),
            'defense_bonus' => 0,
            'hp_bonus' => 0,
            'xp_bonus_percent' => 0,
            'gold_bonus_percent' => 0,
            'cost_gold' => $this->getGoldCost($rarity, 'weapon'),
            'cost_premium' => $this->getPremiumCost($rarity, 'weapon'),
            'is_vip_only' => $rarity === 'legendary',
            'image_url' => "/assets/items/weapon-{$rarity}-" . Str::slug($name) . ".png",
            'created_at' => now(),
            'updated_at' => now(),
        ];
    }

    /**
     * Generate an armor item.
     */
    protected function generateArmor(array $names, int $counter): array
    {
        $rarity = $this->determineRarity($counter, 25);
        $name = $names[$counter % count($names)] . ($counter > count($names) ? ' ' . ($counter / count($names) + 1) : '');
        
        return [
            'name' => $name,
            'slug' => Str::slug($name . '-' . $counter),
            'type' => 'armor',
            'rarity' => $rarity,
            'attack_bonus' => 0,
            'defense_bonus' => $this->getDefenseBonus($rarity),
            'hp_bonus' => $this->getHpBonus($rarity),
            'xp_bonus_percent' => 0,
            'gold_bonus_percent' => 0,
            'cost_gold' => $this->getGoldCost($rarity, 'armor'),
            'cost_premium' => $this->getPremiumCost($rarity, 'armor'),
            'is_vip_only' => $rarity === 'legendary',
            'image_url' => "/assets/items/armor-{$rarity}-" . Str::slug($name) . ".png",
            'created_at' => now(),
            'updated_at' => now(),
        ];
    }

    /**
     * Generate a relic item.
     */
    protected function generateRelic(array $names, int $counter): array
    {
        $rarity = $this->determineRarity($counter, 15);
        $name = $names[$counter % count($names)] . ($counter > count($names) ? ' ' . ($counter / count($names) + 1) : '');
        
        return [
            'name' => $name,
            'slug' => Str::slug($name . '-' . $counter),
            'type' => 'relic',
            'rarity' => $rarity,
            'attack_bonus' => 0,
            'defense_bonus' => 0,
            'hp_bonus' => 0,
            'xp_bonus_percent' => $this->getXpBonusPercent($rarity),
            'gold_bonus_percent' => $this->getGoldBonusPercent($rarity),
            'cost_gold' => $this->getGoldCost($rarity, 'relic'),
            'cost_premium' => $this->getPremiumCost($rarity, 'relic'),
            'is_vip_only' => $rarity === 'legendary',
            'image_url' => "/assets/items/relic-{$rarity}-" . Str::slug($name) . ".png",
            'created_at' => now(),
            'updated_at' => now(),
        ];
    }

    /**
     * Generate a potion item.
     */
    protected function generatePotion(array $names, int $counter): array
    {
        $rarity = $this->determineRarity($counter, 20);
        $name = $names[$counter % count($names)] . ($counter > count($names) ? ' ' . ($counter / count($names) + 1) : '');
        
        return [
            'name' => $name,
            'slug' => Str::slug($name . '-' . $counter),
            'type' => 'potion',
            'rarity' => $rarity,
            'attack_bonus' => 0,
            'defense_bonus' => 0,
            'hp_bonus' => $this->getHpBonus($rarity), // Healing potions
            'xp_bonus_percent' => 0,
            'gold_bonus_percent' => 0,
            'cost_gold' => $this->getGoldCost($rarity, 'potion'),
            'cost_premium' => $this->getPremiumCost($rarity, 'potion'),
            'is_vip_only' => false,
            'image_url' => "/assets/items/potion-{$rarity}-" . Str::slug($name) . ".png",
            'created_at' => now(),
            'updated_at' => now(),
        ];
    }

    /**
     * Generate a skin item.
     */
    protected function generateSkin(array $names, int $counter): array
    {
        $rarity = $this->determineRarity($counter, 10);
        $name = $names[$counter % count($names)] . ($counter > count($names) ? ' ' . ($counter / count($names) + 1) : '');
        
        return [
            'name' => $name,
            'slug' => Str::slug($name . '-' . $counter),
            'type' => 'skin',
            'rarity' => $rarity,
            'attack_bonus' => 0,
            'defense_bonus' => 0,
            'hp_bonus' => 0,
            'xp_bonus_percent' => 0,
            'gold_bonus_percent' => 0,
            'cost_gold' => 0,
            'cost_premium' => $this->getPremiumCost($rarity, 'skin'),
            'is_vip_only' => $rarity === 'epic' || $rarity === 'legendary',
            'image_url' => "/assets/items/skin-{$rarity}-" . Str::slug($name) . ".png",
            'created_at' => now(),
            'updated_at' => now(),
        ];
    }

    /**
     * Determine rarity based on distribution: Common 45%, Rare 30%, Epic 20%, Legendary 5%.
     */
    protected function determineRarity(int $counter, int $totalForType): string
    {
        $rarityRoll = ($counter * 7) % 100; // Deterministic based on counter
        
        if ($rarityRoll < 45) return 'common';
        if ($rarityRoll < 75) return 'rare';
        if ($rarityRoll < 95) return 'epic';
        return 'legendary';
    }

    /**
     * Get attack bonus based on rarity.
     */
    protected function getAttackBonus(string $rarity): int
    {
        return match($rarity) {
            'common' => rand(1, 3),
            'rare' => rand(4, 8),
            'epic' => rand(9, 15),
            'legendary' => rand(16, 25),
        };
    }

    /**
     * Get defense bonus based on rarity.
     */
    protected function getDefenseBonus(string $rarity): int
    {
        return match($rarity) {
            'common' => rand(1, 2),
            'rare' => rand(3, 6),
            'epic' => rand(7, 12),
            'legendary' => rand(13, 20),
        };
    }

    /**
     * Get HP bonus based on rarity.
     */
    protected function getHpBonus(string $rarity): int
    {
        return match($rarity) {
            'common' => rand(10, 20),
            'rare' => rand(25, 50),
            'epic' => rand(60, 100),
            'legendary' => rand(120, 200),
        };
    }

    /**
     * Get XP bonus percent based on rarity.
     */
    protected function getXpBonusPercent(string $rarity): int
    {
        return match($rarity) {
            'common' => rand(5, 10),
            'rare' => rand(15, 25),
            'epic' => rand(30, 50),
            'legendary' => rand(60, 100),
        };
    }

    /**
     * Get gold bonus percent based on rarity.
     */
    protected function getGoldBonusPercent(string $rarity): int
    {
        return match($rarity) {
            'common' => rand(5, 10),
            'rare' => rand(15, 25),
            'epic' => rand(30, 50),
            'legendary' => rand(60, 100),
        };
    }

    /**
     * Get gold cost based on rarity and type.
     */
    protected function getGoldCost(string $rarity, string $type): int
    {
        if ($type === 'skin') return 0; // Skins are premium only
        
        return match($rarity) {
            'common' => rand(50, 200),
            'rare' => rand(300, 800),
            'epic' => rand(1000, 2000),
            'legendary' => 0, // Legendary is premium only
        };
    }

    /**
     * Get premium cost based on rarity and type.
     */
    protected function getPremiumCost(string $rarity, string $type): int
    {
        if ($rarity === 'common' && $type !== 'skin') return 0;
        
        return match($rarity) {
            'common' => rand(5, 10), // For skins
            'rare' => rand(5, 15),
            'epic' => rand(10, 40),
            'legendary' => rand(50, 200),
        };
    }
}
