# Battle System Documentation

## Overview

The battle system is a turn-based combat system that supports both automated and manual battle modes. Players can battle against mobs (monsters) to earn XP, gold, premium coins, and items. The system includes a comprehensive leveling system (1-200 levels), 1000 unique mobs, and 100 shop items.

## Battle Modes

### Auto Mode

In auto mode, the battle is fully simulated on the server and returns a complete turn-by-turn log. The frontend simply plays back the animations based on the returned log.

**API Endpoint:** `POST /api/battle/start`

**Request:**
```json
{
  "mob_id": 42,
  "mode": "auto"
}
```

**Response:**
```json
{
  "battle_id": "uuid",
  "mode": "auto",
  "user_id": 1,
  "mob": {
    "id": 42,
    "name": "Sand Scorpion",
    "hp": 34,
    "max_hp": 50,
    "sprite": "/assets/sprites/mobs/sand-scorpion.png"
  },
  "player": {
    "hp": 120,
    "max_hp": 120,
    "attack": 12,
    "defense": 7,
    "speed": 15,
    "luck": 4
  },
  "turns": [
    {
      "turn": 1,
      "actor": "player",
      "action": "attack",
      "damage": 12,
      "is_crit": false,
      "mob_hp": 38,
      "player_hp": 120,
      "log": "You strike Sand Scorpion for 12 damage"
    },
    {
      "turn": 1,
      "actor": "mob",
      "action": "strikes",
      "damage": 6,
      "is_crit": false,
      "mob_hp": 38,
      "player_hp": 114,
      "log": "Sand Scorpion strikes you for 6 damage"
    }
  ],
  "result": "win",
  "rewards": {
    "xp": 100,
    "gold": 50,
    "premium_coins": 0,
    "items": []
  }
}
```

### Manual Mode

In manual mode, the server returns an initial battle state and waits for the client to send turn actions. Each turn, the client sends an action and receives the updated battle state.

**API Endpoint:** `POST /api/battle/start`

**Request:**
```json
{
  "mob_id": 42,
  "mode": "manual"
}
```

**Response (Initial State):**
```json
{
  "battle_id": "uuid",
  "mode": "manual",
  "user_id": 1,
  "mob": { ... },
  "player": { ... },
  "turns": [],
  "result": null,
  "rewards": null
}
```

**Perform Turn:** `POST /api/battle/turn`

**Request:**
```json
{
  "battle_id": "uuid",
  "action": "attack"
}
```

**Response:**
```json
{
  "battle_id": "uuid",
  "mode": "manual",
  "mob": { "hp": 38, ... },
  "player": { "hp": 114, ... },
  "turns": [ ... ],
  "result": null,
  "rewards": null
}
```

## Battle Log Structure

The battle log is a JSON array of turn objects. Each turn object contains:

- `turn`: Turn number
- `actor`: "player" or "mob"
- `action`: Action type (e.g., "attack", "strikes", "hits")
- `damage`: Damage dealt
- `is_crit`: Whether it was a critical hit
- `mob_hp`: Mob's HP after this action
- `player_hp`: Player's HP after this action
- `log`: Human-readable log message

## Frontend Animation Playback Guide

### Sprite Assets Requirements

Sprite assets should be placed in:
- `/public/assets/sprites/mobs/` - Mob sprites
- `/public/assets/sprites/players/` - Player sprites
- `/public/assets/sprites/effects/` - Effect sprites (damage numbers, crit effects, etc.)

**Sprite Sheet Format:**
- Use PNG sprite sheets with frame sequences
- Recommended: 32x32 or 64x64 pixels per frame
- Frame sequences: idle, attack, hit, death

**Naming Convention:**
- Mob sprites: `{mob-slug}.png`
- Player sprites: `player-{skin-id}.png`

### Animation Timing Guidelines

- **Attack animation**: 400-600ms
- **Damage pop / hit stun**: 200-300ms
- **Death animation**: 800-1200ms
- **Idle loop**: 600-1000ms per loop
- **Turn transition delay**: 300-500ms between turns

### Turn-by-Turn Playback Logic

#### Automated Mode Playback

```javascript
async function playAutoBattle(battleData) {
  for (const turn of battleData.turns) {
    if (turn.actor === 'player') {
      await playPlayerAnimation(turn.action);
      await showDamageEffect(turn.damage, turn.is_crit, 'mob');
      await animateHPBar('mob', turn.mob_hp, battleData.mob.max_hp);
    } else {
      await playMobAnimation(turn.action);
      await showDamageEffect(turn.damage, turn.is_crit, 'player');
      await animateHPBar('player', turn.player_hp, battleData.player.max_hp);
    }
    
    // Wait for turn transition
    await delay(300);
  }
  
  // Show result
  if (battleData.result === 'win') {
    showVictoryScreen(battleData.rewards);
  } else {
    showDefeatScreen();
  }
}
```

#### Manual Mode UI Flow

1. **Display Battle State:**
   - Show player and mob HP bars
   - Display available actions (Attack, Use Item, Skill, Escape)
   - Show turn counter

2. **On Player Action:**
   - Disable action buttons
   - Send POST request to `/api/battle/turn`
   - Play player animation based on action
   - Show damage effect
   - Update HP bars

3. **On Mob Response:**
   - Play mob animation
   - Show damage effect
   - Update HP bars

4. **Check Battle Result:**
   - If `result` is not null, battle is complete
   - Show rewards if won
   - Enable action buttons for next turn if battle continues

### Animation Implementation Examples

#### Using CSS Animations

```css
@keyframes attack {
  0% { transform: translateX(0); }
  50% { transform: translateX(20px); }
  100% { transform: translateX(0); }
}

.character-attacking {
  animation: attack 500ms ease-in-out;
}
```

#### Using Canvas/WebGL (PixiJS)

```javascript
import * as PIXI from 'pixi.js';

function playAttackAnimation(sprite, target) {
  const attackTween = new PIXI.Tween(sprite)
    .to({ x: target.x - 20 }, 250)
    .to({ x: sprite.x }, 250);
  
  attackTween.start();
}
```

#### Using Sprite Sheets

```javascript
function animateSpriteSheet(sprite, frames, duration) {
  const frameTime = duration / frames.length;
  let currentFrame = 0;
  
  const interval = setInterval(() => {
    sprite.style.backgroundPosition = `-${frames[currentFrame].x}px -${frames[currentFrame].y}px`;
    currentFrame++;
    
    if (currentFrame >= frames.length) {
      clearInterval(interval);
    }
  }, frameTime);
}
```

## API Endpoint Reference

### Battle Endpoints

- `POST /api/battle/start` - Start a new battle
  - Body: `{ mob_id: int, mode: "auto"|"manual" }`
  - Returns: Battle instance with initial state

- `POST /api/battle/turn` - Perform a turn (manual mode only)
  - Body: `{ battle_id: string, action: "attack"|"use_item"|"skill"|"attempt_escape" }`
  - Returns: Updated battle state

- `GET /api/battle/result/{battle_id}` - Get final battle result
  - Returns: Complete battle data with rewards

- `GET /api/battle/{battle_id}` - Get current battle state
  - Returns: Current battle state (for manual mode)

### Mob Endpoints

- `GET /api/mobs` - List mobs with filters
  - Query params: `level`, `rarity`, `per_page`
  - Returns: Paginated list of mobs

- `GET /api/mobs/{id}` - Get specific mob details
  - Returns: Mob data

- `GET /api/mobs/for-level/{level}` - Get recommended mobs for level
  - Returns: Collection of mobs appropriate for player level

### Shop Endpoints

- `GET /api/shop/items` - List shop items
  - Query params: `type`, `rarity`, `per_page`
  - Returns: Paginated list of items

- `POST /api/shop/buy` - Purchase an item
  - Body: `{ item_id: int, currency: "gold"|"premium" }`
  - Returns: Purchase result

- `GET /api/shop/items/{id}` - Get specific item details
  - Returns: Item data

## Damage Formula Details

The damage formula is implemented exactly as specified:

```
base = attacker.attack - floor(defender.defense / 2)
if base < 1: base = 1
random_add = random_int(1, 6)
damage = max(1, base + random_add)
if random(1..100) <= attacker.crit_rate: damage *= 2
```

**Crit Rate Calculation:**
```
crit_rate = clamp(5 + floor(luck / 2), 5, 40)
```

**Examples:**
- Attack: 20, Defense: 10 → base = 20 - 5 = 15, damage = 15 + (1-6) = 16-21
- Attack: 5, Defense: 20 → base = 5 - 10 = -5 → base = 1, damage = 1 + (1-6) = 2-7
- Luck: 20 → crit_rate = 5 + 10 = 15% chance

## Turn Order

Turn order is determined by comparing speed stats:
- Higher speed acts first
- If speeds are equal, player acts first
- Speed is derived from player's `stamina` stat (for now)

## Loot System Explanation

### Reward Calculation

On battle victory, rewards are calculated as follows:

1. **Base Rewards:**
   - XP: `mob.xp_reward`
   - Gold: `mob.gold_reward`

2. **Premium Coins:**
   - Roll based on `mob.premium_coin_chance`
   - If successful, award 1-3 premium coins

3. **Item Drops:**
   - Check `mob.loot_table` for drop chances
   - Common drop: 15% chance
   - Rare drop: 10% chance (for rare/epic/legendary mobs)

4. **VIP Multipliers:**
   - If user is VIP, apply XP and gold multipliers from membership
   - XP multiplier: `xp_bonus_percent`
   - Gold multiplier: `gold_bonus_percent`

### Loot Table Structure

```json
{
  "common_drop": {
    "chance": 0.15,
    "item_id": null
  },
  "rare_drop": {
    "chance": 0.10,
    "item_id": null
  }
}
```

## Leveling System

### XP Curve

The XP required for each level follows this formula:

```
xpForNextLevel(level) = floor(100 * level * (1.12)^(level-1))
```

**Examples:**
- Level 1: 100 XP
- Level 10: ~3,110 XP
- Level 50: ~1,200,000 XP
- Level 100: ~50,000,000 XP

### Stat Gains Per Level

On level up, stats increase as follows:

- `max_hp += 5 + floor(level * 0.5)`
- `attack += 1 + floor(level * 0.2)`
- `defense += 1 + floor(level * 0.15)`
- `stamina += 1`
- `luck += 1` (every 3 levels)

Stats are cumulative (total bonuses from all levels).

## Tools & Libraries Recommendations

### Frontend Animation Libraries

1. **PixiJS** - High-performance 2D WebGL renderer
   - Best for complex sprite animations
   - Supports sprite sheets and bone animations
   - Example: `npm install pixi.js`

2. **Phaser** - Full-featured game framework
   - Includes physics, input, audio
   - Good for complete game implementation
   - Example: `npm install phaser`

3. **CSS Animations** - Simple DOM-based animations
   - Good for basic animations
   - No additional dependencies
   - Limited performance for complex scenes

4. **Lottie** - JSON-based animations
   - Good for UI effects (gold burst, sun beam)
   - Example: `npm install lottie-web`

### Bone Animation Tools

1. **Spine** - 2D skeletal animation
   - Export to JSON + atlas
   - Runtime: `npm install pixi-spine`

2. **DragonBones** - Alternative to Spine
   - Export to JSON + texture atlas
   - Runtime: `npm install dragonbonesjs`

### Sprite Sheet Tools

1. **TexturePacker** - Commercial tool
2. **Shoebox** - Free alternative
3. **Aseprite** - Pixel art editor with sprite sheet export

## Performance Considerations

1. **Battle Instance Storage:**
   - Manual mode battles stored in cache (Redis recommended)
   - Expire after 1 hour
   - Cleanup via scheduled job

2. **Turn Log Storage:**
   - Only store full logs for manual mode battles
   - Auto mode: Store summary in `battle_histories` table
   - Limit stored battle history per user (last N battles)

3. **Pagination:**
   - Use pagination for mob and shop item listings
   - Default: 20 items per page

4. **Caching:**
   - Cache level data (rarely changes)
   - Cache mob data for frequently accessed mobs
   - Use Redis for battle instances

## Testing

### Unit Tests

- `tests/Unit/Services/LevelServiceTest.php` - XP formula and level up logic
- `tests/Unit/Services/BattleServiceTest.php` - Damage calculation and turn order

### Integration Tests

- `tests/Feature/Api/BattleTest.php` - Full battle flow, rewards, history

Run tests:
```bash
php artisan test
```

## Future Enhancements

1. **Skills System:**
   - Implement skill actions in manual mode
   - Add skill cooldowns and mana costs

2. **Item Usage:**
   - Implement use_item action
   - Add consumable items (potions, etc.)

3. **Escape Mechanic:**
   - Implement attempt_escape action
   - Add escape chance calculation

4. **Battle Animations:**
   - Add particle effects
   - Implement screen shake on crits
   - Add victory/defeat animations

5. **Battle Replay:**
   - Store full turn logs for replay
   - Add replay viewer UI

