# Character Animation System

This document describes the video-based character animation system.

## Overview

The character animation system uses MP4/WebM video files instead of sprite images. Each character has a standard folder structure, and the app automatically loads the correct video based on the animation state.

## Folder Structure

Each character should have videos organized in:

```
/public/assets/characters/{character_slug}/videos/
    idle.mp4
    idle_anim.mp4
    attack.mp4
    attack_anim.mp4
    magic.mp4
    magic_anim.mp4
    win.mp4
    win_anim.mp4
    lose.mp4
    lose_anim.mp4
```

### Fallback Images

You can also place PNG fallback images in the same folder:

```
/public/assets/characters/{character_slug}/videos/
    idle.png
    attack.png
    win.png
    ...
```

These will be used if the video fails to load.

## Animation States

### Available States

- `idle` - Default idle state (looping)
- `idle_anim` - Animated idle state (looping)
- `attack` - Attack animation (non-looping, returns to idle)
- `attack_anim` - Animated attack (non-looping, returns to idle)
- `magic` - Magic cast animation (non-looping, returns to idle)
- `magic_anim` - Animated magic cast (looping)
- `win` - Win state (stays on last frame)
- `win_anim` - Animated win state (looping)
- `lose` - Lose state (non-looping, returns to idle)
- `lose_anim` - Animated lose state (non-looping, returns to idle)

### Looping Behavior

**Looping States:**
- `idle`
- `idle_anim`
- `win_anim`
- `magic_anim`

**Non-Looping States:**
- `attack`
- `attack_anim`
- `lose`
- `lose_anim`

**Auto-Return to Idle:**
- `attack`
- `attack_anim`
- `lose`
- `lose_anim`
- `magic`

**Stay on Last Frame:**
- `win`
- `win_anim`

## Usage

### Blade Component

The easiest way to use character animations is with the Blade component:

```blade
<x-character-animation 
    :character="$character" 
    state="idle" 
    class="w-64 h-64"
/>
```

#### Component Props

- `character` (required) - The Character model instance
- `state` (optional, default: `idle`) - The animation state to play
- `class` (optional) - Additional CSS classes
- `autoplay` (optional, default: `true`) - Whether to autoplay the video
- `showControls` (optional, default: `false`) - Show video controls
- `fallbackImage` (optional) - Custom fallback image path
- `loading` (optional, default: `true`) - Show loading animation
- `lowQualityMode` (optional) - Force low quality mode (null = auto-detect)

### Backend (Laravel)

#### Get Video Path

```php
$character = Character::find(1);
$videoPath = $character->animationVideo('attack');
// Returns: /assets/characters/ramses/videos/attack.mp4
```

#### Check if State Exists

```php
if ($character->hasAnimationState('attack')) {
    // Attack animation exists
}
```

#### Get Available States

```php
$availableStates = $character->getAvailableAnimationStates();
// Returns: ['idle', 'idle_anim', 'attack', ...]
```

#### Get Fallback Image

```php
$fallbackPath = $character->animationFallbackImage('attack');
// Returns: /assets/characters/ramses/videos/attack.png or null
```

### JavaScript

#### Using CharacterAnimator Class

```javascript
// Create animator instance
const animator = new CharacterAnimator('ramses', videoElement);

// Play different states
animator.playIdle();
animator.playAttack();
animator.playWin();
animator.playLose();

// Or use playState method
animator.playState('magic_anim');

// Get current state
const currentState = animator.getCurrentState();

// Check if looping
const isLooping = animator.isCurrentlyLooping();

// Cleanup
animator.destroy();
```

#### Listening to State Changes

```javascript
window.addEventListener('character-animation-state-change', (event) => {
    console.log('Character:', event.detail.characterSlug);
    console.log('State:', event.detail.state);
    console.log('Looping:', event.detail.isLooping);
});
```

#### Programmatic State Changes

```javascript
// Dispatch state change event
window.dispatchEvent(new CustomEvent('character-animation-state-change', {
    detail: {
        characterSlug: 'ramses',
        state: 'attack',
        isLooping: false
    }
}));
```

## Video Format Support

The system supports both MP4 and WebM formats:

1. **WebM** - Tried first (better compression, supports transparency)
2. **MP4** - Fallback (better browser support)

The browser will automatically choose the best format it supports.

### Recommended Video Settings

- **Format**: MP4 (H.264) or WebM (VP9)
- **Codec**: H.264 for MP4, VP9 for WebM
- **Transparency**: Use alpha channel if needed (WebM supports this better)
- **Resolution**: 1920x1080 or lower (optimize for web)
- **Frame Rate**: 30fps or 60fps
- **Bitrate**: Adjust based on file size requirements

## Fallback Behavior

The system has multiple fallback layers:

1. **State Fallback**: If requested state doesn't exist, falls back to `idle`
2. **Format Fallback**: If WebM doesn't exist, tries MP4
3. **Image Fallback**: If video fails to load, shows PNG fallback image
4. **Character Image Fallback**: If no state-specific PNG, uses character's `image_normal`

## Mobile Optimization

The component automatically detects mobile devices and slow connections:

- **Auto-detection**: Based on screen width (< 768px) or connection speed
- **Low Quality Mode**: Can be forced via `lowQualityMode` prop
- **Playsinline**: Videos play inline on mobile devices

## Example: Battle System Integration

```blade
{{-- In battle view --}}
<div class="battle-characters">
    {{-- Player Character --}}
    <x-character-animation 
        :character="$user->character" 
        :state="$playerState"
        class="w-64 h-64"
        x-ref="playerCharacter"
    />
    
    {{-- Enemy Character --}}
    <x-character-animation 
        :character="$enemy" 
        :state="$enemyState"
        class="w-64 h-64"
    />
</div>

<script>
// Trigger attack animation
document.addEventListener('DOMContentLoaded', () => {
    const playerVideo = document.querySelector('[x-ref="playerCharacter"] video');
    const animator = new CharacterAnimator('{{ $user->character->slug }}', playerVideo);
    
    // On attack button click
    document.getElementById('attack-btn').addEventListener('click', () => {
        animator.playAttack();
    });
    
    // On win
    window.addEventListener('battle-won', () => {
        animator.playWin();
    });
    
    // On lose
    window.addEventListener('battle-lost', () => {
        animator.playLose();
    });
});
</script>
```

## Configuration

Animation states and behavior are configured in `config/character_animations.php`:

```php
return [
    'states' => ['idle', 'idle_anim', 'attack', ...],
    'looping_states' => ['idle', 'idle_anim', ...],
    'auto_return_states' => ['attack', 'attack_anim', ...],
    'stay_on_last_frame_states' => ['win', 'win_anim'],
    'default_state' => 'idle',
    'video_formats' => ['mp4', 'webm'],
];
```

## Troubleshooting

### Video Not Playing

1. Check file exists in correct location
2. Check file permissions
3. Verify video format is supported
4. Check browser console for errors
5. Try fallback PNG image

### State Not Changing

1. Verify state name is correct
2. Check if state exists for character
3. Verify event listeners are set up
4. Check browser console for errors

### Performance Issues

1. Optimize video file size
2. Use WebM for better compression
3. Enable low quality mode for mobile
4. Consider using lower resolution videos

