const canvas = document.querySelector('canvas');
const context = canvas.getContext('2d');

canvas.width = 1024;
canvas.height = 576;

context.fillRect(0, 0, canvas.width, canvas.height);

const gravity = 0.7;
const background = new Sprite({
    position: {
        x: 0,
        y: 0
    },
    imageSrc: './backgrounds/Background1.png'
})

const shop = new Sprite({
    position: {
        x: 700,
        y: 170
    },
    imageSrc: './backgrounds/shop_anim.png',
    scale: 2.5,
    framesMax: 6
})

// Use original hardcoded images by default
// Only use dynamic sprites if explicitly enabled via GAME_DATA.useDynamic = true
let player, enemy;
let useDynamicSprites = false;

if (typeof window.GAME_DATA !== 'undefined' && window.GAME_DATA && window.GAME_DATA.useDynamic === true) {
    useDynamicSprites = true;
    const gameData = window.GAME_DATA;
    
    // Create player from character data
    // Position Y: canvas height (576) - scaled height (200) + offsetY (157) = 533
    const floorY = canvas.height - 200 + 157; // 533
    player = createFighterFromPngSequence(
        gameData.character.sprites,
        { x: 100, y: floorY },
        { x: 0, y: 0 },
        false
    );
    player.health = gameData.character.health || 100;
    player.init();
    
    // Create enemy from mob data
    // Position Y: same floor level as player
    enemy = createFighterFromPngSequence(
        gameData.mob.sprites,
        { x: 800, y: floorY },
        { x: 0, y: 0 },
        true
    );
    enemy.health = gameData.mob.health || 100;
    enemy.init();
    
    // Update health bars with actual values
    gsap.to('#playerHealth', { width: player.health + '%' });
    gsap.to('#enemyHealth', { width: enemy.health + '%' });
} else {
    // Use farmer-peasant PNG sequences for player (left character)
    // Get base URL for assets
    const baseUrl = window.location.origin;
    const farmerPeasantSprites = {
        idle: {
            frames: [
                baseUrl + '/assets/characters/farmer-peasant/idle/01.png',
                baseUrl + '/assets/characters/farmer-peasant/idle/02.png',
                baseUrl + '/assets/characters/farmer-peasant/idle/03.png',
                baseUrl + '/assets/characters/farmer-peasant/idle/04.png',
                baseUrl + '/assets/characters/farmer-peasant/idle/05.png'
            ],
            framesMax: 5,
            isSequence: true
        },
        attack1: {
            frames: [
                baseUrl + '/assets/characters/farmer-peasant/fight/01.png',
                baseUrl + '/assets/characters/farmer-peasant/fight/02.png',
                baseUrl + '/assets/characters/farmer-peasant/fight/03.png',
                baseUrl + '/assets/characters/farmer-peasant/fight/04.png',
                baseUrl + '/assets/characters/farmer-peasant/fight/05.png'
            ],
            framesMax: 5,
            isSequence: true
        }
    };
    
    // Create player using PNG sequences
    // Position Y: canvas height (576) - scaled height (200) + offsetY (157) = 533
    // This puts the character's feet on the floor
    const floorY = canvas.height - 200 + 157; // 533
    player = createFighterFromPngSequence(
        farmerPeasantSprites,
        { x: 100, y: floorY },
        { x: 0, y: 0 },
        false
    );
    player.health = 100;
    player.init();
    
    // Use desert rat PNG sequences for enemy (player 2)
    const desertRatSprites = {
        idle: {
            frames: [
                baseUrl + '/assets/mobs/desert-rat/idle/left/01.png',
                baseUrl + '/assets/mobs/desert-rat/idle/left/02.png',
                baseUrl + '/assets/mobs/desert-rat/idle/left/03.png',
                baseUrl + '/assets/mobs/desert-rat/idle/left/04.png',
                baseUrl + '/assets/mobs/desert-rat/idle/left/05.png',
                baseUrl + '/assets/mobs/desert-rat/idle/left/06.png',
                baseUrl + '/assets/mobs/desert-rat/idle/left/07.png',
                baseUrl + '/assets/mobs/desert-rat/idle/left/08.png',
                baseUrl + '/assets/mobs/desert-rat/idle/left/09.png'
            ],
            framesMax: 9,
            isSequence: true
        },
        attack1: {
            frames: [
                baseUrl + '/assets/mobs/desert-rat/attack/left/01.png',
                baseUrl + '/assets/mobs/desert-rat/attack/left/02.png',
                baseUrl + '/assets/mobs/desert-rat/attack/left/03.png',
                baseUrl + '/assets/mobs/desert-rat/attack/left/04.png',
                baseUrl + '/assets/mobs/desert-rat/attack/left/05.png',
                baseUrl + '/assets/mobs/desert-rat/attack/left/06.png',
                baseUrl + '/assets/mobs/desert-rat/attack/left/07.png',
                baseUrl + '/assets/mobs/desert-rat/attack/left/08.png',
                baseUrl + '/assets/mobs/desert-rat/attack/left/09.png'
            ],
            framesMax: 9,
            isSequence: true
        }
    };
    
    // Create enemy using desert rat PNG sequences
    // Position Y: same floor level as player
    enemy = createFighterFromPngSequence(
        desertRatSprites,
        { x: 800, y: floorY },
        { x: 0, y: 0 },
        true // isEnemy = true
    );
    enemy.health = 100;
    enemy.init();
}

// Automatic fighting system - no keyboard controls needed
let currentRound = 1;
let isPlayerTurn = true;
let roundInProgress = false;
let roundCooldown = 0; // Start with 0 so fighting begins immediately
let attackCooldown = 0;
let attackExecuted = false;
let returningToPosition = false;
const ROUND_DELAY = 60; // frames between rounds (~1 second at 60fps) - faster
const ATTACK_DELAY = 10; // frames before attack executes - faster
const ATTACK_DISTANCE = 90; // Distance to maintain for attacks (reduced so players get very close)
const ATTACK_LUNGE_DISTANCE = 80; // How far forward to move when attacking

// Store starting positions
// Ensure playerStartX is never less than 50 to prevent going out of bounds
const playerStartX = Math.max(50, 100); // Minimum 50 to stay on screen
const enemyStartX = 800;

// Game state
let gameStarted = false;

// Countdown function
function startCountdown() {
    const countdownElement = document.getElementById('countdown');
    const countdownNumber = document.getElementById('countdownNumber');
    let count = 3;
    
    countdownElement.style.display = 'flex';
    countdownNumber.textContent = count;
    
    const countdownInterval = setInterval(() => {
        count--;
        if (count > 0) {
            countdownNumber.textContent = count;
            // Add pulse animation
            countdownNumber.style.animation = 'none';
            setTimeout(() => {
                countdownNumber.style.animation = 'pulse 0.5s ease-in-out';
            }, 10);
        } else if (count === 0) {
            countdownNumber.textContent = 'FIGHT!';
            countdownNumber.style.fontSize = '80px';
            setTimeout(() => {
                countdownElement.style.display = 'none';
                gameStarted = true;
                decreaseTimer();
                // animate() is already running, just need to set gameStarted = true
            }, 500);
            clearInterval(countdownInterval);
        }
    }, 1000);
}

// Start countdown when page loads
startCountdown();

// Start animation loop immediately (to show countdown and idle fighters)
animate();

function handleAutomaticFighting() {
    // Don't fight if someone is dead
    if (player.dead || enemy.dead) {
        return;
    }

    // Handle round cooldown
    if (roundCooldown > 0) {
        roundCooldown--;
        return;
    }

    const attacker = isPlayerTurn ? player : enemy;
    const defender = isPlayerTurn ? enemy : player;
    const attackerStartX = isPlayerTurn ? playerStartX : enemyStartX;
    const defenderStartX = isPlayerTurn ? enemyStartX : playerStartX;
    
    // Calculate distance between fighters
    const distance = Math.abs(attacker.position.x - defender.position.x);
    
    // If returning to position after attack
    if (returningToPosition) {
        const distanceToStart = Math.abs(attacker.position.x - attackerStartX);
        
        if (distanceToStart > 5) {
            // Move back to starting position
            if (attacker.position.x < attackerStartX) {
                // Attacker is to the left of start, move right
                // For player: check if we'll hit right boundary
                if (attacker === player) {
                    const playerMaxX = canvas.width * 0.6;
                    const nextX = attacker.position.x + 10; // Calculate next position
                    if (nextX > playerMaxX) {
                        // Would hit right boundary, clamp to maxX
                        attacker.velocity.x = 0;
                        attacker.position.x = playerMaxX;
                        // If startX < maxX, we can still move left toward start
                        if (attackerStartX < playerMaxX) {
                            attacker.velocity.x = -10; // Move left toward start
                        }
                        console.log(`[PLAYER RETURN] Prevented hitting right boundary! Clamped to ${playerMaxX.toFixed(2)}, StartX: ${attackerStartX}`);
                    } else {
                        attacker.velocity.x = 10; // Faster return
                    }
                } else {
                    attacker.velocity.x = 10; // Faster return
                }
            } else if (attacker.position.x > attackerStartX) {
                // Attacker is to the right of start, move left
                // For player: don't allow going below playerStartX at all
                if (attacker === player) {
                    const nextX = attacker.position.x - 10; // Calculate next position
                    // Don't allow going below playerStartX - stop exactly at start position
                    if (nextX < playerStartX) {
                        // Would go below start, stop at start position
                        attacker.velocity.x = 0;
                        attacker.position.x = playerStartX; // Clamp to exact start position
                        console.log(`[PLAYER RETURN] Stopped at start position! Position: ${playerStartX}, StartX: ${attackerStartX}`);
                    } else {
                        attacker.velocity.x = -10; // Faster return
                    }
                } else {
                    attacker.velocity.x = -10; // Faster return
                }
            } else {
                // Already at position (shouldn't happen, but just in case)
                attacker.velocity.x = 0;
            }
            
            // Log return progress for player
            if (attacker === player && (!attacker._lastReturnLog || Date.now() - attacker._lastReturnLog > 200)) {
                console.log(`[PLAYER RETURN] Position: ${attacker.position.x.toFixed(2)}, StartX: ${attackerStartX}, Distance: ${distanceToStart.toFixed(2)}, Velocity: ${attacker.velocity.x}`);
                attacker._lastReturnLog = Date.now();
            }
        } else {
            // Reached starting position - snap to exact position
            // For player: ensure we're at exactly playerStartX, not below it
            const finalX = attacker === player ? playerStartX : attackerStartX;
            attacker.position.x = finalX;
            console.log(`[RETURN COMPLETE] ${attacker === player ? 'Player' : 'Enemy'} reached start position: ${attacker.position.x.toFixed(2)}`);
            attacker.velocity.x = 0;
            returningToPosition = false;
            isPlayerTurn = !isPlayerTurn;
            roundInProgress = false;
            roundCooldown = ROUND_DELAY;
            currentRound++;
            attackExecuted = false;
        }
        // Stop defender from moving during return
        defender.velocity.x = 0;
        return;
    }
    
    // If too far, move closer to center
    if (distance > ATTACK_DISTANCE) {
        roundInProgress = false;
        attackExecuted = false;
        attackCooldown = 0;
        
        if (isPlayerTurn) {
            // Player moves right toward center
            const playerMaxX = canvas.width * 0.6;
            const playerAtBoundary = player.position.x >= playerMaxX;
            
            if (player.position.x < enemy.position.x - ATTACK_DISTANCE && player.position.x < playerMaxX) {
                // Check if next position would exceed boundary
                const nextX = player.position.x + 10;
                if (nextX > playerMaxX) {
                    player.velocity.x = 0;
                    player.position.x = playerMaxX; // Clamp to boundary
                } else {
                    player.velocity.x = 10; // Faster movement
                }
            } else {
                player.velocity.x = 0;
            }
            
            // If player is stuck at boundary and still too far, let enemy move left
            if (playerAtBoundary && distance > ATTACK_DISTANCE) {
                const targetX = player.position.x + ATTACK_DISTANCE;
                if (enemy.position.x > targetX) {
                    enemy.velocity.x = -10; // Enemy moves left to close distance
                    console.log(`[STUCK FIX] Player at boundary, enemy moving left. Enemy: ${enemy.position.x.toFixed(2)}, Target: ${targetX.toFixed(2)}`);
                } else {
                    enemy.velocity.x = 0;
                }
            } else {
                enemy.velocity.x = 0;
            }
        } else {
            // Enemy moves left toward center
            const targetX = player.position.x + ATTACK_DISTANCE;
            // Set minimum X to prevent enemy from going too far left (at least 50px from left edge)
            const enemyMinX = Math.max(50, targetX - 50); // Don't go more than 50px past target
            
            if (enemy.position.x > targetX) {
                // Check if enemy would go too far left
                const nextX = enemy.position.x - 10;
                if (nextX < enemyMinX) {
                    // Would go too far left, stop at minimum
                    enemy.velocity.x = 0;
                    enemy.position.x = Math.max(enemyMinX, enemy.position.x); // Clamp to minimum
                    console.log(`[ENEMY MOVE] Stopped at minimum X: ${enemy.position.x.toFixed(2)}, Target: ${targetX.toFixed(2)}, MinX: ${enemyMinX.toFixed(2)}`);
                } else if (nextX < 0) {
                    // Would hit left boundary, stop
                    enemy.velocity.x = 0;
                    enemy.position.x = Math.max(0, enemy.position.x);
                    console.log(`[ENEMY MOVE] Stopped at left boundary: ${enemy.position.x.toFixed(2)}`);
                } else {
                    enemy.velocity.x = -10; // Faster movement
                    if (!enemy._lastMoveLog || Date.now() - enemy._lastMoveLog > 500) {
                        console.log(`[ENEMY MOVE] Moving toward player. Position: ${enemy.position.x.toFixed(2)}, Target: ${targetX.toFixed(2)}, Player: ${player.position.x.toFixed(2)}, Distance: ${Math.abs(enemy.position.x - player.position.x).toFixed(2)}`);
                        enemy._lastMoveLog = Date.now();
                    }
                }
            } else {
                // Reached or passed target, stop
                enemy.velocity.x = 0;
                if (!enemy._lastStopLog || Date.now() - enemy._lastStopLog > 500) {
                    console.log(`[ENEMY MOVE] Reached attack position. Position: ${enemy.position.x.toFixed(2)}, Target: ${targetX.toFixed(2)}, Player: ${player.position.x.toFixed(2)}`);
                    enemy._lastStopLog = Date.now();
                }
            }
            
            // Only set player velocity to 0 if enemy is not stuck
            if (enemy.velocity.x !== 0 || enemy.position.x > targetX) {
                player.velocity.x = 0;
            }
        }
    } else {
        // Close enough, perform attack
        if (!roundInProgress) {
            roundInProgress = true;
            attackCooldown = ATTACK_DELAY;
            attackExecuted = false;
            console.log(`[ATTACK PHASE] Starting attack phase. IsPlayerTurn: ${isPlayerTurn}, Distance: ${distance.toFixed(2)}, ATTACK_DISTANCE: ${ATTACK_DISTANCE}, Attacker: ${attacker === player ? 'Player' : 'Enemy'}, Attacker.isAttacking: ${attacker.isAttacking}, Attacker.dead: ${attacker.dead}`);
        } else {
            if (!attacker._lastRoundLog || Date.now() - attacker._lastRoundLog > 500) {
                console.log(`[ATTACK PHASE] Round in progress. IsPlayerTurn: ${isPlayerTurn}, attackCooldown: ${attackCooldown}, attackExecuted: ${attackExecuted}, Attacker.isAttacking: ${attacker.isAttacking}`);
                attacker._lastRoundLog = Date.now();
            }
        }
        
        // Move forward during attack
        if (attackCooldown > 0) {
            attackCooldown--;
            if (!attacker._lastCooldownLog || Date.now() - attacker._lastCooldownLog > 200) {
                console.log(`[ATTACK COOLDOWN] ${attacker === player ? 'Player' : 'Enemy'} cooldown: ${attackCooldown}, IsPlayerTurn: ${isPlayerTurn}`);
                attacker._lastCooldownLog = Date.now();
            }
            // Start moving forward - player 1 should move more aggressively to get closer
            if (isPlayerTurn) {
                // Check right boundary before moving
                const playerMaxX = canvas.width * 0.6;
                const nextX = attacker.position.x + 20;
                if (nextX > playerMaxX) {
                    attacker.velocity.x = 0;
                    attacker.position.x = playerMaxX; // Clamp to boundary
                } else {
                    attacker.velocity.x = 20; // Faster attack movement for player 1
                }
            } else {
                attacker.velocity.x = -15; // Faster attack movement
            }
            defender.velocity.x = 0;
        } else if (!attackExecuted) {
            // Execute attack
            console.log(`[ATTACK EXECUTE CHECK] ${attacker === player ? 'Player' : 'Enemy'}: dead=${attacker.dead}, defender.dead=${defender.dead}, isAttacking=${attacker.isAttacking}, attackExecuted=${attackExecuted}, attackCooldown=${attackCooldown}`);
            if (!attacker.dead && !defender.dead && !attacker.isAttacking) {
                console.log(`[ATTACK START] ${attacker === player ? 'Player' : 'Enemy'} starting attack. Position: ${attacker.position.x.toFixed(2)}, IsPlayerTurn: ${isPlayerTurn}`);
                attacker.attack();
                attackExecuted = true;
                console.log(`[ATTACK START] After attack() call. ${attacker === player ? 'Player' : 'Enemy'} isAttacking: ${attacker.isAttacking}, currentState: ${attacker.currentState || 'N/A'}`);
                // Reset attack hit flag for both player and enemy
                attacker._attackHitRegistered = false;
                attacker._lastCollisionLog = null;
                attacker._lastMissLog = null;
                // Continue moving forward during attack to ensure contact - player 1 moves faster
                if (isPlayerTurn) {
                    // Allow player to move closer even if near boundary during attack
                    const playerMaxX = canvas.width * 0.6;
                    const targetX = defender.position.x - 10; // Target: 10px before enemy (very close)
                    const nextX = attacker.position.x + 18;
                    // If we're not close enough yet, allow slight boundary crossing to get closer
                    if (attacker.position.x < targetX) {
                        if (nextX > playerMaxX + 30) {
                            // Would go too far past boundary, limit movement
                            attacker.velocity.x = Math.max(0, (playerMaxX + 30 - attacker.position.x) / 2);
                        } else {
                            attacker.velocity.x = 18; // Faster for player 1 to get closer
                        }
                    } else {
                        attacker.velocity.x = 0; // Close enough, stop
                    }
                } else {
                    attacker.velocity.x = -12; // Faster
                    console.log(`[ENEMY ATTACK] Enemy attack started. Position: ${attacker.position.x.toFixed(2)}, Velocity: ${attacker.velocity.x}`);
                }
            } else {
                console.log(`[ATTACK EXECUTE] SKIPPED - ${attacker === player ? 'Player' : 'Enemy'}: dead=${attacker.dead}, defender.dead=${defender.dead}, isAttacking=${attacker.isAttacking}`);
            }
        } else {
            // Keep moving forward for a bit during attack animation to ensure hit connects
            let attackFramesMax, currentFrame;
            // Player uses PNG sequences, enemy uses sprite sheets
            if (attacker.currentSprite || (useDynamicSprites && attacker.currentSprite)) {
                const attackSprite = attacker.sprites.attack1.sequenceSprite;
                attackFramesMax = attackSprite.framesMax;
                currentFrame = attackSprite.frameCurrent;
            } else {
                attackFramesMax = attacker.sprites.attack1.framesMax;
                currentFrame = attacker.frameCurrent;
            }
            
            if (currentFrame < attackFramesMax - 1) {
                // Still in attack animation, keep moving forward but limit distance
                if (isPlayerTurn) {
                    // Player: get very close to enemy - stop extremely close (10px)
                    const maxX = defender.position.x - 10; // Stop 10px before enemy for extremely close attack
                    const playerMaxX = canvas.width * 0.6; // Right boundary
                    // Allow player to move closer even if near boundary during attack
                    if (attacker.position.x < maxX) {
                        // Check if next position would exceed boundary significantly
                        const nextX = attacker.position.x + 18;
                        if (nextX > playerMaxX + 30) {
                            // Would go too far past boundary, limit movement
                            attacker.velocity.x = Math.max(0, (playerMaxX + 30 - attacker.position.x) / 2);
                        } else if (nextX > playerMaxX) {
                            // Slightly past boundary is OK during attack to get closer
                            attacker.velocity.x = 18; // Continue moving to get closer
                        } else {
                            attacker.velocity.x = 18; // Faster for player 1
                        }
                    } else {
                        attacker.velocity.x = 0; // Stop if too close
                    }
                } else {
                    // Enemy: get closer to player - stop closer (30px instead of 50px)
                    // But don't go below 50px from left edge
                    const minX = Math.max(50, defender.position.x - 30); // Stop 30px before player, but at least 50px from left
                    if (attacker.position.x > minX) {
                        attacker.velocity.x = -10; // Faster
                        if (!attacker._lastAttackMoveLog || Date.now() - attacker._lastAttackMoveLog > 200) {
                            console.log(`[ENEMY ATTACK MOVE] Moving forward. Position: ${attacker.position.x.toFixed(2)}, MinX: ${minX.toFixed(2)}, Player: ${defender.position.x.toFixed(2)}, Frame: ${currentFrame}/${attackFramesMax - 1}`);
                            attacker._lastAttackMoveLog = Date.now();
                        }
                    } else {
                        attacker.velocity.x = 0; // Stop if too close
                        if (!attacker._lastAttackStopLog || Date.now() - attacker._lastAttackStopLog > 200) {
                            console.log(`[ENEMY ATTACK MOVE] Stopped (too close). Position: ${attacker.position.x.toFixed(2)}, MinX: ${minX.toFixed(2)}, Player: ${defender.position.x.toFixed(2)}`);
                            attacker._lastAttackStopLog = Date.now();
                        }
                    }
                }
            } else {
                // Attack animation done, stop moving forward immediately
                attacker.velocity.x = 0;
                if (attacker === enemy) {
                    console.log(`[ENEMY ATTACK] Animation done. Final position: ${attacker.position.x.toFixed(2)}, Player: ${defender.position.x.toFixed(2)}`);
                }
            }
        }
        
        // Wait for attack animation to complete, then return to position
        let attackComplete, hitAnimationComplete;
        
        // Both player and enemy now use PNG sequences
        if (attacker.currentSprite || (useDynamicSprites && attacker.currentSprite)) {
            // PNG sequence sprite (both player and enemy)
            const attackSprite = attacker.sprites.attack1.sequenceSprite;
            const attackFramesMax = attackSprite.framesMax;
            attackComplete = attackExecuted && attackSprite.frameCurrent >= attackFramesMax - 1;
            
            // Check defender's hit animation
            if (defender.currentSprite && defender.currentState === 'takeHit') {
                const hitSprite = defender.sprites.takeHit.sequenceSprite;
                hitAnimationComplete = hitSprite.frameCurrent >= hitSprite.framesMax - 1;
            } else {
                hitAnimationComplete = true;
            }
        } else {
            // Sprite sheet (fallback - shouldn't happen now)
            const attackFramesMax = attacker.sprites.attack1.framesMax;
            // Check if attack animation is complete
            // For sprite sheets, check if we're past the last frame OR if we've switched back to idle
            const isAttackSprite = attacker.image === attacker.sprites.attack1.image;
            const isPastLastFrame = attacker.frameCurrent >= attackFramesMax - 1;
            const hasSwitchedFromAttack = !isAttackSprite && attackExecuted;
            
            // Attack is complete if we've finished the attack frames OR switched away from attack sprite
            attackComplete = attackExecuted && (isAttackSprite && isPastLastFrame || hasSwitchedFromAttack);
            
            // Also wait for defender's hit animation to complete if they were hit
            if (defender.currentSprite && defender.currentState === 'takeHit') {
                const hitSprite = defender.sprites.takeHit.sequenceSprite;
                hitAnimationComplete = hitSprite.frameCurrent >= hitSprite.framesMax - 1;
            } else {
                const defenderHitFramesMax = defender.sprites.takeHit.framesMax;
                hitAnimationComplete = defender.image !== defender.sprites.takeHit.image ||
                    defender.frameCurrent >= defenderHitFramesMax - 1;
            }
        }
        
        // Start returning when attack is complete and not currently attacking
        if (attackComplete && !attacker.isAttacking && hitAnimationComplete) {
            // Stop any forward movement immediately
            attacker.velocity.x = 0;
            // Start returning to starting position
            returningToPosition = true;
        }
    }
}

function animate() {
    window.requestAnimationFrame(animate);
    
    // Don't update game if countdown is still showing
    if (!gameStarted) {
        // Still draw background and characters in idle state
        context.fillStyle = 'black';
        context.fillRect(0, 0, canvas.width, canvas.height);
        background.update();
        shop.update();
        
        // Draw fighters in idle position (don't animate, just show them)
        // Player uses PNG sequences
        if (player.currentSprite) {
            if (player.currentSprite.loaded) {
                player.currentSprite.position = player.position;
                player.currentSprite.draw(context);
            }
        } else {
            player.draw();
        }
        
        // Enemy uses PNG sequences (desert rat) or sprite sheets (fallback)
        if (enemy.currentSprite) {
            if (enemy.currentSprite.loaded) {
                enemy.currentSprite.position = enemy.position;
                enemy.currentSprite.draw(context);
            }
        } else {
            // Sprite sheet enemy (fallback)
            enemy.draw();
        }
        return;
    }
    
    context.fillStyle = 'black';
    context.fillRect(0, 0, canvas.width, canvas.height);
    background.update();
    shop.update();
    
    // Handle automatic fighting (this sets velocities)
    handleAutomaticFighting();
    
    // Store enemy reference dimensions for size matching (first frame only)
    // Target height is 200px for both characters
    // Enemy now uses PNG sequences (desert rat)
    const targetHeight = 200;
    if (!window.enemyReferenceHeight && enemy.currentSprite && enemy.currentSprite.loaded && enemy.currentSprite.images.length > 0) {
        const enemyImg = enemy.currentSprite.images[enemy.currentSprite.frameCurrent];
        if (enemyImg && enemyImg.complete) {
            // Calculate scale to reach target height
            const enemyScale = targetHeight / enemyImg.height;
            window.enemyReferenceHeight = targetHeight; // Use fixed target height
            window.enemyReferenceWidth = enemyImg.width * enemyScale;
            
            // Force player sprites to recalculate scale on next draw
            if (player.currentSprite && player.sprites) {
                Object.keys(player.sprites).forEach(state => {
                    if (player.sprites[state].sequenceSprite) {
                        player.sprites[state].sequenceSprite._lastLoggedScale = null;
                    }
                });
            }
        }
    }
    
    // Prevent player from going below playerStartX BEFORE updating position
    // Check if next position would be below playerStartX and prevent it
    if (player.velocity.x < 0) {
        const nextX = player.position.x + player.velocity.x;
        // When returning, don't allow going below playerStartX at all
        const minX = returningToPosition ? playerStartX : Math.max(0, playerStartX - 10);
        if (nextX < minX) {
            console.log(`[BOUNDS] Player would go too far left! Current: ${player.position.x.toFixed(2)}, Velocity: ${player.velocity.x}, Next: ${nextX.toFixed(2)}, MinX: ${minX.toFixed(2)}, Returning: ${returningToPosition}, Preventing movement`);
            player.velocity.x = 0;
            player.position.x = minX;
            // If returning and we're at start, we're done
            if (returningToPosition && player.position.x >= playerStartX) {
                // Already at or past start, stop
                player.velocity.x = 0;
            } else if (returningToPosition && player.position.x < playerStartX) {
                // Still need to move right to reach start
                player.velocity.x = 10; // Move right toward start
                console.log(`[BOUNDS] Player at left boundary during return, moving right toward startX: ${playerStartX}`);
            }
        }
    }
    
    // Update fighters (different method for dynamic vs static)
    // Player uses PNG sequences, so check for currentSprite property
    if (player.currentSprite || useDynamicSprites) {
        player.update(context);
    } else {
        player.update();
    }
    
    // Enemy uses PNG sequences (desert rat) or sprite sheets (fallback)
    if (enemy.currentSprite || useDynamicSprites) {
        enemy.update(context);
    } else {
        enemy.update();
    }
    
    // Prevent characters from going off-screen (final safety check)
    // When returning, player should stay at playerStartX or above. Otherwise, allow slight movement below.
    const playerMinX = returningToPosition ? playerStartX : Math.max(0, playerStartX - 10);
    if (player.position.x < playerMinX) {
        console.log(`[BOUNDS] Player hit left boundary! Position: ${player.position.x.toFixed(2)}, Velocity: ${player.velocity.x}, Returning: ${returningToPosition}, Clamping to ${playerMinX.toFixed(2)}`);
        player.position.x = playerMinX;
        // Always stop leftward velocity when hitting left boundary
        if (player.velocity.x < 0) {
            player.velocity.x = 0;
            // If returning and we're at start, we're done
            if (returningToPosition && player.position.x >= playerStartX) {
                // Already at or past start, stop
                player.velocity.x = 0;
            } else if (returningToPosition && player.position.x < playerStartX) {
                // Still need to move right to reach start
                player.velocity.x = 10; // Move right toward start
                console.log(`[BOUNDS] Player at left boundary during return, moving right toward startX: ${playerStartX}`);
            }
        }
    }
    // Also prevent player from going too far right (past center)
    const playerMaxX = canvas.width * 0.6; // Allow player to move 60% across screen
    if (player.position.x > playerMaxX) {
        console.log(`[BOUNDS] Player hit right boundary! Position: ${player.position.x.toFixed(2)}, MaxX: ${playerMaxX}, Velocity: ${player.velocity.x}, Returning: ${returningToPosition}`);
        player.position.x = playerMaxX;
        // Always stop rightward velocity when hitting right boundary
        if (player.velocity.x > 0) {
            player.velocity.x = 0;
            // If returning and startX < maxX, we need to move left
            if (returningToPosition && playerStartX < playerMaxX) {
                player.velocity.x = -10; // Move left toward start
                console.log(`[BOUNDS] Player at right boundary during return, moving left toward startX: ${playerStartX}`);
            }
        }
    }
    // Enemy should stay on the right side (x <= canvas.width)
    if (enemy.position.x > canvas.width) {
        console.log(`[BOUNDS] Enemy hit right boundary, resetting from ${enemy.position.x} to ${canvas.width}`);
        enemy.position.x = canvas.width;
        enemy.velocity.x = 0;
    }
    // Prevent enemy from going too far left (off-screen on left side)
    // Set minimum to 50px to prevent going too far left
    const enemyAbsoluteMinX = 50;
    if (enemy.position.x < enemyAbsoluteMinX) {
        console.log(`[BOUNDS] Enemy hit LEFT boundary! Position: ${enemy.position.x}, returningToPosition: ${returningToPosition}, isPlayerTurn: ${isPlayerTurn}`);
        console.log(`[BOUNDS] Enemy startX: ${enemyStartX}, current position: ${enemy.position.x}, clamping to ${enemyAbsoluteMinX}`);
        
        // If enemy is at or past left boundary, force it back to minimum
        enemy.position.x = enemyAbsoluteMinX;
        
        // Always force return to starting position if enemy hits left boundary
        if (!returningToPosition || (returningToPosition && !isPlayerTurn)) {
            console.log(`[BOUNDS] Enemy stuck at left boundary, forcing return to ${enemyStartX}`);
            returningToPosition = true;
            enemy.velocity.x = 10; // Move right to return
            // Reset attack state
            enemy.isAttacking = false;
            if (enemy.image === enemy.sprites.attack1.image) {
                enemy.switchSprite('idle');
            }
        }
    }
    
    // Additional check: if enemy is stuck near left boundary and should be returning
    // Only trigger if enemy is way too far left (less than 50px) AND not currently attacking or moving toward target
    const targetX = player.position.x + ATTACK_DISTANCE;
    const distance = Math.abs(enemy.position.x - player.position.x);
    // Don't force return if enemy is close enough to attack (within ATTACK_DISTANCE)
    if (enemy.position.x < 50 && enemy.position.x < targetX - 100 && !returningToPosition && distance > ATTACK_DISTANCE) {
        // Enemy is too far left - only force return if it's not the player's turn (enemy's turn to attack)
        if (!isPlayerTurn && !enemy.isAttacking) {
            console.log(`[BOUNDS] Enemy too far left (${enemy.position.x.toFixed(2)}), forcing return to ${enemyStartX}`);
            returningToPosition = true;
            enemy.velocity.x = 10; // Move right to return
        }
    }

    // Handle sprite animations based on velocity and state
    // Player uses PNG sequences (farmer-peasant), enemy uses sprite sheets
    if (player.currentSprite || useDynamicSprites) {
        // Player uses PNG sequences - only has idle and attack1
        if (player.isAttacking) {
            // Don't change sprite during attack
        } else if (!player.dead) {
            player.switchSprite('idle');
        }
    } else {
        // Original sprite handling (shouldn't happen for player now)
        if (player.isAttacking) {
            // Don't change sprite during attack
        } else if (player.velocity.x !== 0) {
            player.switchSprite('run');
        } else if (player.velocity.y < 0) {
            player.switchSprite('jump');
        } else if (player.velocity.y > 0) {
            player.switchSprite('fall');
        } else if (!player.dead) {
            player.switchSprite('idle');
        }
    }
    
    // Enemy uses PNG sequences (desert rat) or sprite sheets (fallback)
    if (enemy.currentSprite || useDynamicSprites) {
        // Enemy uses PNG sequences - only has idle and attack1
        if (enemy.isAttacking) {
            // Don't change sprite during attack - keep attack animation
            // Make sure attack sprite is active
            if (enemy.currentState !== 'attack1' && enemy.sprites.attack1) {
                console.log(`[ENEMY SPRITE] Force switching to attack1 during attack. Current state: ${enemy.currentState}`);
                enemy.switchSprite('attack1');
            }
            // IMPORTANT: Don't switch away from attack1 while isAttacking is true
        } else if (!enemy.dead) {
            // Only switch to idle if not attacking AND not in attack state
            if (enemy.currentState !== 'idle' && enemy.currentState !== 'attack1') {
                enemy.switchSprite('idle');
            } else if (enemy.currentState === 'attack1' && !enemy.isAttacking) {
                // Attack animation finished, switch to idle
                enemy.switchSprite('idle');
            }
        }
    } else {
        // Original sprite sheet handling (fallback)
        if (enemy.isAttacking) {
            // Don't change sprite during attack
        } else if (enemy.velocity.x !== 0) {
            enemy.switchSprite('run');
        } else if (enemy.velocity.y < 0) {
            enemy.switchSprite('jump');
        } else if (enemy.velocity.y > 0) {
            enemy.switchSprite('fall');
        } else if (!enemy.dead) {
            enemy.switchSprite('idle');
        }
    }

        //detect colission & enemy hit
        // Player uses PNG sequences (farmer-peasant with 5 frames for attack)
        if (player.currentSprite || useDynamicSprites) {
            // For PNG sequence sprites (player)
            if (player.isAttacking && player.currentSprite && player.currentState === 'attack1') {
                const attackSprite = player.sprites.attack1.sequenceSprite;
                const frameIndex = attackSprite.frameCurrent;
                const framesMax = attackSprite.framesMax;
                // Check middle frames of attack animation (frames 2-4 for 5-frame animation)
                if (frameIndex >= 2 && frameIndex <= 4) {
                    if (rectangularCollision({
                        rectangle1: player,
                        rectangle2: enemy
                    })) {
                        enemy.takeHit();
                        player.isAttacking = false;
                        gsap.to('#enemyHealth', {
                            width: enemy.health + '%'
                        })
                    }
                }
            }
            
            // Check if player attack animation finished
            if (player.isAttacking && player.currentSprite && player.currentState === 'attack1') {
                const attackSprite = player.sprites.attack1.sequenceSprite;
                if (attackSprite.frameCurrent >= attackSprite.framesMax - 1) {
                    player.isAttacking = false;
                }
            }
        
        // Enemy hits (PNG sequence - desert rat)
        if (enemy.isAttacking && enemy.currentSprite && enemy.currentState === 'attack1') {
            const attackSprite = enemy.sprites.attack1.sequenceSprite;
            const frameIndex = attackSprite.frameCurrent;
            const framesMax = attackSprite.framesMax;
            // Check middle frames of attack animation (frames 2-6 for 9-frame animation)
            const hitFrameStart = Math.floor(framesMax * 0.2);
            const hitFrameEnd = Math.floor(framesMax * 0.7);
            if (frameIndex >= hitFrameStart && frameIndex <= hitFrameEnd) {
                // Make sure attackBox is updated before collision check
                // Dynamically adjust attack box offset based on enemy position relative to player
                // If enemy is to the left of player, attack box should extend right (positive offset)
                // If enemy is to the right of player, attack box should extend left (negative offset)
                let attackBoxOffsetX = enemy.attackBox.offset.x;
                if (enemy.position.x < player.position.x) {
                    // Enemy is on left, attack box should extend right from enemy's right edge
                    // Enemy width is ~50, so start attack box slightly to the right of enemy position
                    attackBoxOffsetX = 30; // Positive offset to extend right, starting near enemy's right edge
                } else {
                    // Enemy is on right, attack box should extend left (original)
                    attackBoxOffsetX = -170; // Negative offset to extend left
                }
                enemy.attackBox.position.x = enemy.position.x + attackBoxOffsetX;
                enemy.attackBox.position.y = enemy.position.y + enemy.attackBox.offset.y;
                
                // Debug logging
                if (!enemy._lastCollisionLog || Date.now() - enemy._lastCollisionLog > 100) {
                    console.log(`[ENEMY ATTACK CHECK] Frame: ${frameIndex}/${framesMax}, Position: ${enemy.position.x.toFixed(2)}, Player Position: ${player.position.x.toFixed(2)}`);
                    console.log(`[ENEMY ATTACK CHECK] AttackBox: x=${enemy.attackBox.position.x.toFixed(2)}, y=${enemy.attackBox.position.y.toFixed(2)}, w=${enemy.attackBox.width}, h=${enemy.attackBox.height}`);
                    console.log(`[ENEMY ATTACK CHECK] Player: x=${player.position.x.toFixed(2)}, y=${player.position.y.toFixed(2)}, w=${player.width}, h=${player.height}`);
                    console.log(`[ENEMY ATTACK CHECK] Distance: ${Math.abs(enemy.position.x - player.position.x).toFixed(2)}`);
                    enemy._lastCollisionLog = Date.now();
                }
                
                if (rectangularCollision({
                    rectangle1: enemy,
                    rectangle2: player
                })) {
                    console.log(`[ENEMY ATTACK] COLLISION DETECTED! Frame: ${frameIndex}`);
                    // Only hit once per attack - use a flag to prevent multiple hits
                    if (!enemy._attackHitRegistered) {
                        console.log(`[ENEMY ATTACK] REGISTERING HIT! Player health before: ${player.health}`);
                        player.takeHit();
                        enemy._attackHitRegistered = true;
                        // Don't set isAttacking to false immediately - let animation finish
                        enemy.velocity.x = 0; // Stop movement immediately
                        console.log(`[ENEMY ATTACK] Player health after: ${player.health}`);
                        gsap.to('#playerHealth', {
                            width: player.health + '%'
                        })
                    } else {
                        console.log(`[ENEMY ATTACK] Hit already registered, skipping`);
                    }
                } else {
                    if (!enemy._lastMissLog || Date.now() - enemy._lastMissLog > 500) {
                        console.log(`[ENEMY ATTACK] No collision - AttackBox and Player don't overlap`);
                        enemy._lastMissLog = Date.now();
                    }
                }
            }
        }
        
        // Check if enemy attack animation finished (if no hit was registered)
        if (enemy.isAttacking && enemy.currentSprite && enemy.currentState === 'attack1') {
            const attackSprite = enemy.sprites.attack1.sequenceSprite;
            if (attackSprite.frameCurrent >= attackSprite.framesMax - 1) {
                console.log(`[ENEMY ATTACK] PNG Animation finished at frame ${attackSprite.frameCurrent}, setting isAttacking to false`);
                enemy.isAttacking = false;
                enemy._attackHitRegistered = false; // Reset for next attack
            }
        }
        } else {
            // Player uses PNG sequences, enemy uses sprite sheets
            // Player collision detection (PNG sequences handled above)
            if (player.currentSprite) {
                // Already handled in PNG sequence section above
            } else {
                // Fallback for sprite sheet player (shouldn't happen now)
                if (player.isAttacking && (player.frameCurrent >= 2 && player.frameCurrent <= 5)) {
                    if (rectangularCollision({
                        rectangle1: player,
                        rectangle2: enemy
                    })) {
                        enemy.takeHit();
                        player.isAttacking = false;
                        gsap.to('#enemyHealth', {
                            width: enemy.health + '%'
                        })
                    }
                }
                if (player.isAttacking && player.frameCurrent >= 5) {
                    player.isAttacking = false;
                }
            }

            // Enemy hits (sprite sheet)
            // Check frames 1, 2, or 3 for enemy attack (4 frame animation)
            // Make sure enemy is actually attacking and using attack sprite
            if (enemy.isAttacking && !enemy.currentSprite && enemy.image === enemy.sprites.attack1.image && (enemy.frameCurrent >= 1 && enemy.frameCurrent <= 3)) {
                // Make sure attackBox is updated before collision check
                // Dynamically adjust attack box offset based on enemy position relative to player
                let attackBoxOffsetX = enemy.attackBox.offset.x;
                if (enemy.position.x < player.position.x) {
                    // Enemy is on left, attack box should extend right
                    attackBoxOffsetX = 50; // Positive offset to extend right
                } else {
                    // Enemy is on right, attack box should extend left (original)
                    attackBoxOffsetX = -170; // Negative offset to extend left
                }
                enemy.attackBox.position.x = enemy.position.x + attackBoxOffsetX;
                enemy.attackBox.position.y = enemy.position.y + enemy.attackBox.offset.y;
                
                // Debug logging
                if (!enemy._lastCollisionLog || Date.now() - enemy._lastCollisionLog > 100) {
                    console.log(`[ENEMY ATTACK CHECK] Frame: ${enemy.frameCurrent}, Position: ${enemy.position.x.toFixed(2)}, Player Position: ${player.position.x.toFixed(2)}`);
                    console.log(`[ENEMY ATTACK CHECK] AttackBox: x=${enemy.attackBox.position.x.toFixed(2)}, y=${enemy.attackBox.position.y.toFixed(2)}, w=${enemy.attackBox.width}, h=${enemy.attackBox.height}`);
                    console.log(`[ENEMY ATTACK CHECK] Player: x=${player.position.x.toFixed(2)}, y=${player.position.y.toFixed(2)}, w=${player.width}, h=${player.height}`);
                    console.log(`[ENEMY ATTACK CHECK] Distance: ${Math.abs(enemy.position.x - player.position.x).toFixed(2)}`);
                    enemy._lastCollisionLog = Date.now();
                }
                
                if (rectangularCollision({
                    rectangle1: enemy,
                    rectangle2: player
                })) {
                    console.log(`[ENEMY ATTACK] COLLISION DETECTED! Frame: ${enemy.frameCurrent}`);
                    // Only hit once per attack - use a flag to prevent multiple hits
                    if (!enemy._attackHitRegistered) {
                        console.log(`[ENEMY ATTACK] REGISTERING HIT! Player health before: ${player.health}`);
                        player.takeHit();
                        enemy._attackHitRegistered = true;
                        enemy.isAttacking = false; // Stop attack after hit (like original game)
                        enemy.velocity.x = 0; // Stop movement immediately
                        console.log(`[ENEMY ATTACK] Player health after: ${player.health}`);
                        gsap.to('#playerHealth', {
                            width: player.health + '%'
                        })
                    } else {
                        console.log(`[ENEMY ATTACK] Hit already registered, skipping`);
                    }
                } else {
                    if (!enemy._lastMissLog || Date.now() - enemy._lastMissLog > 500) {
                        console.log(`[ENEMY ATTACK] No collision - AttackBox and Player don't overlap`);
                        enemy._lastMissLog = Date.now();
                    }
                }
            } else if (enemy.isAttacking && !enemy.currentSprite) {
                // Log why collision check isn't running
                if (!enemy._lastSkipLog || Date.now() - enemy._lastSkipLog > 500) {
                    console.log(`[ENEMY ATTACK] Skipping collision check - Frame: ${enemy.frameCurrent}, IsAttackSprite: ${enemy.image === enemy.sprites.attack1.image}, IsAttacking: ${enemy.isAttacking}`);
                    enemy._lastSkipLog = Date.now();
                }
            }

            // if enemy attack animation finished - check at end of attack animation (sprite sheet)
            // Enemy attack has 4 frames (0-3), so check if we're at the last frame
            if (enemy.isAttacking && !enemy.currentSprite && enemy.image === enemy.sprites.attack1.image && enemy.frameCurrent >= 3) {
                console.log(`[ENEMY ATTACK] Animation finished at frame ${enemy.frameCurrent}, setting isAttacking to false`);
                enemy.isAttacking = false;
                enemy._attackHitRegistered = false; // Reset for next attack
                // Switch back to idle so the return logic can work
                enemy.switchSprite('idle');
            }
        }

    // end game based on health
    if (enemy.health <= 0 || player.health <= 0) {
        determinewinner({ player, enemy, timerId });
    }
}

// animate() is called after countdown finishes
// Keyboard controls removed - game is now fully automatic