/**
 * PNG Sequence Loader
 * Handles loading PNG sequences and converting them to sprite-like objects
 */

class PngSequenceSprite {
    constructor({ position, frames, scale = 1, offset = { x: 0, y: 0 } }) {
        this.position = position;
        this.frames = frames; // Array of image paths
        this.images = []; // Array of loaded Image objects
        this.scale = scale;
        this.offset = offset;
        this.frameCurrent = 0;
        this.framesElapsed = 0;
        this.framesHold = 5; // Animation speed
        this.framesMax = frames.length;
        this.loaded = false;
        this.loading = false;
        
        // Load all images
        this.loadImages();
    }
    
    loadImages() {
        if (this.loading || this.loaded) return;
        this.loading = true;
        
        let loadedCount = 0;
        const totalFrames = this.frames.length;
        
        this.images = this.frames.map((framePath, index) => {
            const img = new Image();
            img.onload = () => {
                loadedCount++;
                if (loadedCount === totalFrames) {
                    this.loaded = true;
                    this.loading = false;
                }
            };
            img.onerror = () => {
                console.error(`Failed to load frame ${index}: ${framePath}`);
                loadedCount++;
                if (loadedCount === totalFrames) {
                    this.loaded = true;
                    this.loading = false;
                }
            };
            img.src = framePath;
            return img;
        });
    }
    
    draw(context) {
        if (!context) {
            console.error('[PNG SEQUENCE SPRITE] draw() called without context parameter');
            return;
        }
        if (!this.loaded || this.images.length === 0) return;
        
        const currentImage = this.images[this.frameCurrent];
        if (!currentImage || !currentImage.complete) return;
        
        // Match the exact size - target height of 200px for both characters
        const targetHeight = 200; // Fixed target height for both characters
        let scaleToUse = targetHeight / currentImage.height;
        
        // For enemy: set reference height so player can match it
        const isEnemy = this._isEnemy === true;
        if (isEnemy) {
            // Enemy sets the reference height
            const renderedHeight = currentImage.height * scaleToUse;
            if (!window.enemyReferenceHeight || window.enemyReferenceHeight <= 0) {
                window.enemyReferenceHeight = renderedHeight;
                window.enemyReferenceWidth = currentImage.width * scaleToUse;
            }
        } else {
            // Player: use the same target height to match enemy
            // If enemy reference is set, use it; otherwise use target height
            if (window.enemyReferenceHeight && window.enemyReferenceHeight > 0) {
                scaleToUse = window.enemyReferenceHeight / currentImage.height;
            } else {
                // Use target height until enemy loads
                scaleToUse = targetHeight / currentImage.height;
            }
        }
        
        const scaledWidth = currentImage.width * scaleToUse;
        const scaledHeight = currentImage.height * scaleToUse;
        
        // Store scale for future use
        this.scale = scaleToUse;
        
        // Use the same offset as original player character (x: 215, y: 157)
        const offsetX = this.offset.x !== undefined ? this.offset.x : 215;
        const offsetY = this.offset.y !== undefined ? this.offset.y : 157;
        
        context.drawImage(
            currentImage,
            this.position.x - offsetX,
            this.position.y - offsetY,
            scaledWidth,
            scaledHeight
        );
    }
    
    animateFrames() {
        this.framesElapsed++;
        
        if (this.framesElapsed % this.framesHold === 0) {
            if (this.frameCurrent < this.framesMax - 1) {
                this.frameCurrent++;
            } else {
                this.frameCurrent = 0;
            }
        }
    }
    
    update(context) {
        this.draw(context);
        this.animateFrames();
    }
    
    switchToFrame(frameIndex) {
        if (frameIndex >= 0 && frameIndex < this.framesMax) {
            this.frameCurrent = frameIndex;
        }
    }
}

/**
 * Create a Fighter-compatible sprite object from PNG sequence data
 */
function createFighterFromPngSequence(spriteData, position, velocity, isEnemy = false) {
    // Map game states to available sprite data
    const stateMap = {
        'idle': spriteData.idle || spriteData.idle,
        'run': spriteData.run || spriteData.idle, // Fallback to idle
        'jump': spriteData.jump || spriteData.idle,
        'fall': spriteData.fall || spriteData.idle,
        'attack1': spriteData.attack1 || spriteData.attack || spriteData.idle,
        'takeHit': spriteData.takeHit || spriteData.idle,
        'death': spriteData.death || spriteData.lose || spriteData.idle,
    };
    
    // Create sprite objects for each state
    const sprites = {};
    // Use same offset as original game characters for consistent positioning
    const baseOffset = isEnemy ? { x: 215, y: 170 } : { x: 215, y: 157 };
    
    Object.keys(stateMap).forEach(state => {
        const stateData = stateMap[state];
        if (stateData && stateData.frames && stateData.frames.length > 0) {
                  // Use all frames - don't limit attack frames (desert rat has 9 attack frames)
            let framesToUse = stateData.frames;
            // Only limit if explicitly needed (not for enemy with 9 frames)
            // if (state === 'attack1' && framesToUse.length > 5 && !isEnemy) {
            //     framesToUse = framesToUse.slice(0, 5);
            // }
            
            sprites[state] = {
                imageSrc: framesToUse[0], // First frame for compatibility
                framesMax: stateData.framesMax || framesToUse.length,
                frames: framesToUse,
                isSequence: true,
                sequenceSprite: new PngSequenceSprite({
                    position: { x: 0, y: 0 },
                    frames: framesToUse,
                    scale: 2.5,
                    offset: baseOffset
                })
            };
        }
    });
    
        // Create a modified Fighter that uses PNG sequences
    const fighter = {
        position: position,
        velocity: velocity,
        width: 50,
        height: 150,
        attackBox: {
            position: { x: position.x, y: position.y },
            offset: isEnemy ? { x: -170, y: 50 } : { x: 100, y: 50 },
            width: isEnemy ? 170 : 160,
            height: 50
        },
        color: isEnemy ? 'blue' : 'red',
        isAttacking: false,
        health: 100,
        frameCurrent: 0,
        framesElapsed: 0,
        framesHold: 5,
        sprites: sprites,
        dead: false,
        currentState: 'idle',
        currentSprite: null,
        image: null, // For compatibility with original Fighter class
        
        // Initialize with idle sprite
        init() {
            if (this.sprites.idle) {
                this.currentSprite = this.sprites.idle.sequenceSprite;
                this.currentState = 'idle';
            }
        },
        
        switchSprite(spriteName) {
            // Override death animation
            if (this.currentState === 'death' && this.sprites.death) {
                const deathSprite = this.sprites.death.sequenceSprite;
                if (deathSprite.frameCurrent < deathSprite.framesMax - 1) {
                    return;
                } else {
                    this.dead = true;
                    return;
                }
            }
            
            // Override attack animation - but allow switching TO attack1
            if (this.currentState === 'attack1' && spriteName !== 'attack1' && this.sprites.attack1) {
                const attackSprite = this.sprites.attack1.sequenceSprite;
                if (attackSprite.frameCurrent < attackSprite.framesMax - 1) {
                    return; // Don't switch away from attack animation until it's done
                }
            }
            
            // Override takeHit animation
            if (this.currentState === 'takeHit' && this.sprites.takeHit) {
                const hitSprite = this.sprites.takeHit.sequenceSprite;
                if (hitSprite.frameCurrent < hitSprite.framesMax - 1) {
                    return;
                }
            }
            
            // Switch to new sprite
            if (this.sprites[spriteName]) {
                this.currentSprite = this.sprites[spriteName].sequenceSprite;
                this.currentState = spriteName;
                this.currentSprite.frameCurrent = 0;
                this.frameCurrent = 0; // For compatibility
                // Set image reference for compatibility
                if (this.currentSprite.images && this.currentSprite.images[0]) {
                    this.image = this.currentSprite.images[0];
                }
                console.log(`[SWITCH SPRITE] Switched to ${spriteName}. Current sprite: ${this.currentSprite ? 'exists' : 'null'}, Loaded: ${this.currentSprite ? this.currentSprite.loaded : 'N/A'}, Images: ${this.currentSprite ? this.currentSprite.images.length : 0}`);
            } else {
                console.log(`[SWITCH SPRITE] WARNING: Sprite '${spriteName}' not found in sprites object. Available: ${Object.keys(this.sprites).join(', ')}`);
            }
        },
        
        attack() {
            console.log(`[ATTACK METHOD] Switching to attack1. Current state: ${this.currentState}, Has attack1 sprite: ${!!this.sprites.attack1}`);
            this.switchSprite('attack1');
            this.isAttacking = true;
            // Reset frame to start of attack animation
            if (this.currentSprite) {
                this.currentSprite.frameCurrent = 0;
            }
            console.log(`[ATTACK METHOD] After switch. Current state: ${this.currentState}, Current sprite: ${this.currentSprite ? 'exists' : 'null'}, IsAttacking: ${this.isAttacking}`);
        },
        
        takeHit() {
            this.health -= 35;
            if (this.health <= 0) {
                this.switchSprite('death');
            } else {
                this.switchSprite('takeHit');
            }
        },
        
        draw(context) {
            if (!context) {
                console.error('[PNG SEQUENCE] draw() called without context parameter');
                return;
            }
            if (this.currentSprite && this.currentSprite.loaded) {
                this.currentSprite.position = this.position;
                this.currentSprite.draw(context);
                // Update frameCurrent for compatibility
                this.frameCurrent = this.currentSprite.frameCurrent;
            }
        },
        
        update(context) {
            this.draw(context);
            
            if (!this.dead && this.currentSprite) {
                this.currentSprite.animateFrames();
                // Update frameCurrent for compatibility
                this.frameCurrent = this.currentSprite.frameCurrent;
            }
            
            // Update attack box
            this.attackBox.position.x = this.position.x + this.attackBox.offset.x;
            this.attackBox.position.y = this.position.y + this.attackBox.offset.y;
            
            // Draw attack box (for debugging)
            if (this.isAttacking) {
                context.strokeStyle = this.color === 'red' ? 'red' : 'blue';
                context.lineWidth = 2;
                context.strokeRect(
                    this.attackBox.position.x,
                    this.attackBox.position.y,
                    this.attackBox.width,
                    this.attackBox.height
                );
            }
            
            // Update position
            this.position.x += this.velocity.x;
            this.position.y += this.velocity.y;
            
            // Gravity
            if (this.position.y + this.height + this.velocity.y >= context.canvas.height - 82) {
                this.velocity.y = 0;
                this.position.y = 345;
            } else {
                this.velocity.y += 0.7; // gravity
            }
        }
    };
    
    return fighter;
}

