/**
 * CharacterAnimator
 * 
 * Manages character animation state changes and provides methods
 * to play different animation states.
 */
class CharacterAnimator {
    constructor(characterSlug, videoElement = null) {
        this.characterSlug = characterSlug;
        this.videoElement = videoElement;
        this.currentState = 'idle';
        this.isLooping = false;
        
        // Get config from window or use defaults
        this.loopingStates = window.characterAnimationConfig?.loopingStates || [
            'idle',
            'idle_anim',
            'win_anim',
            'magic_anim'
        ];
        
        this.autoReturnStates = window.characterAnimationConfig?.autoReturnStates || [
            'attack',
            'attack_anim',
            'lose',
            'lose_anim',
            'magic'
        ];
        
        this.stayOnLastFrameStates = window.characterAnimationConfig?.stayOnLastFrameStates || [
            'win',
            'win_anim'
        ];
        
        // Bind methods
        this.handleVideoEnded = this.handleVideoEnded.bind(this);
        
        // Setup video element if provided
        if (this.videoElement) {
            this.setupVideoElement();
        }
    }
    
    /**
     * Setup video element event listeners
     */
    setupVideoElement() {
        if (this.videoElement) {
            this.videoElement.addEventListener('ended', this.handleVideoEnded);
        }
    }
    
    /**
     * Get video path for a state
     */
    getVideoPath(state, format = 'mp4') {
        return `/assets/characters/${this.characterSlug}/videos/${state}.${format}`;
    }
    
    /**
     * Play a specific animation state
     */
    playState(state) {
        if (!this.isValidState(state)) {
            console.warn(`Invalid animation state: ${state}`);
            state = 'idle';
        }
        
        this.currentState = state;
        this.isLooping = this.loopingStates.includes(state);
        
        // Dispatch event for Alpine.js components
        this.dispatchStateChange(state);
        
        // Update video element if available
        if (this.videoElement) {
            this.updateVideoElement(state);
        }
        
        return this;
    }
    
    /**
     * Update video element with new state
     */
    updateVideoElement(state) {
        if (!this.videoElement) return;
        
        const video = this.videoElement;
        const webmPath = this.getVideoPath(state, 'webm');
        const mp4Path = this.getVideoPath(state, 'mp4');
        
        // Update sources
        const sources = video.querySelectorAll('source');
        if (sources.length >= 2) {
            sources[0].src = webmPath; // WebM first
            sources[1].src = mp4Path;  // MP4 fallback
        } else {
            // Create sources if they don't exist
            const webmSource = document.createElement('source');
            webmSource.src = webmPath;
            webmSource.type = 'video/webm';
            
            const mp4Source = document.createElement('source');
            mp4Source.src = mp4Path;
            mp4Source.type = 'video/mp4';
            
            video.innerHTML = '';
            video.appendChild(webmSource);
            video.appendChild(mp4Source);
        }
        
        video.loop = this.isLooping;
        video.load();
        video.play().catch(err => {
            console.warn('Video autoplay prevented:', err);
        });
    }
    
    /**
     * Dispatch state change event
     */
    dispatchStateChange(state) {
        // Dispatch for Alpine.js
        window.dispatchEvent(new CustomEvent('animation-state-change', {
            detail: {
                characterSlug: this.characterSlug,
                state: state,
                isLooping: this.isLooping
            }
        }));
        
        // Also dispatch a more generic event
        window.dispatchEvent(new CustomEvent('character-animation-state-change', {
            detail: {
                characterSlug: this.characterSlug,
                state: state,
                isLooping: this.isLooping
            }
        }));
    }
    
    /**
     * Handle video ended event
     */
    handleVideoEnded() {
        if (this.autoReturnStates.includes(this.currentState)) {
            // Auto-return to idle
            this.playIdle();
        } else if (this.stayOnLastFrameStates.includes(this.currentState)) {
            // Stay on last frame (do nothing)
            return;
        }
    }
    
    /**
     * Check if state is valid
     */
    isValidState(state) {
        const validStates = window.characterAnimationConfig?.states || [
            'idle',
            'idle_anim',
            'attack',
            'attack_anim',
            'magic',
            'magic_anim',
            'win',
            'win_anim',
            'lose',
            'lose_anim'
        ];
        return validStates.includes(state);
    }
    
    // Convenience methods for each state
    playIdle() {
        return this.playState('idle');
    }
    
    playIdleAnim() {
        return this.playState('idle_anim');
    }
    
    playAttack() {
        return this.playState('attack');
    }
    
    playAttackAnim() {
        return this.playState('attack_anim');
    }
    
    playMagic() {
        return this.playState('magic');
    }
    
    playMagicAnim() {
        return this.playState('magic_anim');
    }
    
    playWin() {
        return this.playState('win');
    }
    
    playWinAnim() {
        return this.playState('win_anim');
    }
    
    playLose() {
        return this.playState('lose');
    }
    
    playLoseAnim() {
        return this.playState('lose_anim');
    }
    
    /**
     * Get current state
     */
    getCurrentState() {
        return this.currentState;
    }
    
    /**
     * Check if currently looping
     */
    isCurrentlyLooping() {
        return this.isLooping;
    }
    
    /**
     * Set video element
     */
    setVideoElement(element) {
        if (this.videoElement) {
            this.videoElement.removeEventListener('ended', this.handleVideoEnded);
        }
        this.videoElement = element;
        this.setupVideoElement();
    }
    
    /**
     * Destroy animator and cleanup
     */
    destroy() {
        if (this.videoElement) {
            this.videoElement.removeEventListener('ended', this.handleVideoEnded);
        }
        this.videoElement = null;
    }
}

// Export for use in modules (ES6)
export default CharacterAnimator;

// Also make available globally for backward compatibility
if (typeof window !== 'undefined') {
    window.CharacterAnimator = CharacterAnimator;
}

