@php
    $characterSlug = request('character', '');
    $mobSlug = request('mob', '');
    if (empty($characterSlug) || empty($mobSlug)) {
        header('Location: ' . route('characters.test-fight'));
        exit;
    }
@endphp
<x-app-layout>
    <x-slot name="header">
        <div class="flex justify-between items-center style="position: absolute; left: 50%; transform: translateX(-50%); ">
            <h2 class="font-egyptian text-3xl text-ra-gold text-shadow-gold flex items-center gap-3">
                <span  class="text-4xl">⚔️</span>
            </h2>
            <a href="{{ route('characters.test-fight') }}" class="cartouche-button text-sm">
                ← Back to Selection
            </a>
        </div>
    </x-slot>

    <style>
        .arena-background {
            background: 
                linear-gradient(180deg, #0a0a1a 0%, #1a1a2e 30%, #2d1b3d 60%, #1a1a2e 100%),
                radial-gradient(ellipse at center, rgba(139, 69, 19, 0.3) 0%, transparent 70%);
            position: relative;
            overflow: hidden;
            min-height: 100vh;
        }

        .arena-background::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-image: 
                radial-gradient(ellipse 200px 400px at 10% 20%, rgba(139, 69, 19, 0.2) 0%, transparent 50%),
                radial-gradient(ellipse 200px 400px at 90% 20%, rgba(139, 69, 19, 0.2) 0%, transparent 50%),
                radial-gradient(circle at 15% 10%, rgba(255, 193, 7, 0.15) 0%, transparent 40%),
                radial-gradient(circle at 85% 10%, rgba(255, 193, 7, 0.15) 0%, transparent 40%),
                radial-gradient(circle at 50% 50%, rgba(139, 69, 19, 0.1) 0%, transparent 60%),
                linear-gradient(180deg, transparent 0%, rgba(0, 0, 0, 0.4) 100%);
            pointer-events: none;
            animation: arenaPulse 4s ease-in-out infinite;
        }

        @keyframes arenaPulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.9; }
        }

        .arena-background::after {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-image: 
                repeating-linear-gradient(
                    45deg,
                    transparent,
                    transparent 100px,
                    rgba(255, 193, 7, 0.03) 100px,
                    rgba(255, 193, 7, 0.03) 102px
                ),
                repeating-linear-gradient(
                    -45deg,
                    transparent,
                    transparent 100px,
                    rgba(244, 67, 54, 0.03) 100px,
                    rgba(244, 67, 54, 0.03) 102px
                );
            pointer-events: none;
            animation: dustFloat 20s linear infinite;
        }

        @keyframes dustFloat {
            0% { transform: translateY(0) translateX(0); }
            100% { transform: translateY(-100px) translateX(50px); }
        }

        .arena-sand {
            background: 
                /* Background image will be set dynamically via JavaScript */
                url('/assets/images/arena_bg.jpg') center center / cover no-repeat,
                /* Fallback gradient if image doesn't load */
                linear-gradient(to bottom, 
                    #d4a574 0%, 
                    #c49060 20%,
                    #b8865a 40%,
                    #c49060 60%,
                    #d4a574 80%,
                    #c49060 100%
                );
            background-size: cover !important;
            background-position: center center !important;
            background-repeat: no-repeat !important;
            border-top: 6px solid #8b7355;
            border-bottom: 6px solid #6b5344;
            position: relative;
            box-shadow: 
                inset 0 4px 20px rgba(0, 0, 0, 0.4),
                inset 0 -4px 20px rgba(0, 0, 0, 0.3),
                0 0 50px rgba(139, 69, 19, 0.5);
            transition: background-image 1s ease-in-out;
        }
        
        /* Optional: Add overlay to darken/tint the background image */
        .arena-sand::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(212, 165, 116, 0.3); /* Adjust color and opacity */
            pointer-events: none;
            z-index: 1;
        }
        
        /* Make sure content appears above the overlay */
        .arena-sand > * {
            position: relative;
            z-index: 2;
        }

        .sand-texture {
            position: absolute;
            width: 100%;
            height: 100%;
            background-image: 
                repeating-linear-gradient(45deg, transparent, transparent 2px, rgba(0,0,0,.05) 2px, rgba(0,0,0,.05) 4px),
                repeating-linear-gradient(-45deg, transparent, transparent 2px, rgba(0,0,0,.05) 2px, rgba(0,0,0,.05) 4px),
                radial-gradient(ellipse 30px 20px at 25% 60%, rgba(139, 69, 19, 0.2) 0%, transparent 70%),
                radial-gradient(ellipse 30px 20px at 75% 70%, rgba(139, 69, 19, 0.2) 0%, transparent 70%),
                linear-gradient(30deg, transparent 48%, rgba(139, 69, 19, 0.15) 49%, rgba(139, 69, 19, 0.15) 51%, transparent 52%),
                linear-gradient(-30deg, transparent 48%, rgba(139, 69, 19, 0.15) 49%, rgba(139, 69, 19, 0.15) 51%, transparent 52%);
            pointer-events: none;
            opacity: 0.6;
        }

        .arena-walls {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-image: 
                linear-gradient(to right, rgba(0, 0, 0, 0.3) 0%, transparent 15%),
                linear-gradient(to left, rgba(0, 0, 0, 0.3) 0%, transparent 15%);
            pointer-events: none;
            z-index: 0;
        }

        .battle-atmosphere {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: 
                radial-gradient(ellipse at 30% 50%, rgba(255, 193, 7, 0.08) 0%, transparent 50%),
                radial-gradient(ellipse at 70% 50%, rgba(244, 67, 54, 0.08) 0%, transparent 50%);
            pointer-events: none;
            z-index: 1;
            animation: atmospherePulse 3s ease-in-out infinite;
        }

        @keyframes atmospherePulse {
            0%, 100% { opacity: 0.5; }
            50% { opacity: 0.8; }
        }

        .character-slot {
            position: relative;
            perspective: 1000px;
        }

        /* Ensure characters face each other properly */
        /* Both characters should face towards the center (each other) */
        
        /* Character on left (first slot) - ensure it faces right towards mob */
        /* Target the character-container class */
        .arena-sand .character-slot:first-of-type .character-container {
            transform: scaleX(-1) !important;
        }

        /* Mob on right (last slot) - ensure it faces left towards character */
        /* Target the mob-container class */
        .arena-sand .character-slot:last-of-type .mob-container {
            transform: scaleX(-1) !important;
        }

        .character-entrance {
            animation: slideInLeft 1.5s cubic-bezier(0.68, -0.55, 0.265, 1.55) forwards;
        }

        .mob-entrance {
            animation: slideInRight 1.5s cubic-bezier(0.68, -0.55, 0.265, 1.55) forwards;
        }

        @keyframes slideInLeft {
            from {
                transform: translateX(-150%) scaleX(-1);
                opacity: 0;
            }
            to {
                transform: translateX(0) scaleX(1);
                opacity: 1;
            }
        }

        @keyframes slideInRight {
            from {
                transform: translateX(150%);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }

        @keyframes pulse-glow {
            0%, 100% { filter: drop-shadow(0 0 10px rgba(255, 193, 7, 0.3)); }
            50% { filter: drop-shadow(0 0 20px rgba(255, 193, 7, 0.6)); }
        }

        .character-glow {
            animation: pulse-glow 2s infinite;
        }

        .impact-flash {
            animation: flashRed 0.3s ease-out;
        }

        @keyframes flashRed {
            0% { filter: brightness(2) saturate(2) hue-rotate(0deg); }
            100% { filter: brightness(1) saturate(1) hue-rotate(0deg); }
        }

        .shake {
            animation: shake 0.5s ease-in-out;
        }

        @keyframes shake {
            0%, 100% { transform: translateX(0); }
            25% { transform: translateX(-10px); }
            75% { transform: translateX(10px); }
        }

        .hp-bar-damage {
            animation: damageReduction 0.8s ease-out;
        }

        @keyframes damageReduction {
            0% { transform: scaleX(1); }
            100% { transform: scaleX(var(--damage-scale, 1)); }
        }

        .victory-animation {
            animation: victoryBounce 1s ease-in-out infinite;
        }

        @keyframes victoryBounce {
            0%, 100% { transform: translateY(0); }
            50% { transform: translateY(-20px); }
        }

        .defeat-animation {
            animation: defeatFall 2s ease-in forwards;
        }

        @keyframes defeatFall {
            0% { transform: translateY(0) rotateZ(0deg); opacity: 1; }
            100% { transform: translateY(100px) rotateZ(-10deg); opacity: 0.7; }
        }

        .arena-floor {
            position: relative;
            z-index: 1;
        }

        .name-plate {
            background: linear-gradient(135deg, rgba(0, 0, 0, 0.7), rgba(0, 0, 0, 0.5));
            border: 2px solid;
            padding: 0.5rem 1rem;
            border-radius: 0.5rem;
            font-family: 'Egyptian', serif;
            font-size: 1.125rem;
            font-weight: bold;
            text-shadow: 2px 2px 4px rgba(0, 0, 0, 0.8);
        }

        .damage-number {
            position: absolute;
            font-weight: bold;
            pointer-events: none;
            animation: damageFloat 1.5s ease-out forwards;
            font-size: 2rem;
            text-shadow: 2px 2px 4px rgba(0, 0, 0, 0.8);
        }

        @keyframes damageFloat {
            0% { 
                transform: translateY(0) translateX(0);
                opacity: 1;
            }
            100% { 
                transform: translateY(-60px) translateX(20px);
                opacity: 0;
            }
        }

        .arena-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.3);
            z-index: 40;
            pointer-events: none;
        }

        /* Advanced Fight Effects */
        .screen-shake {
            animation: screenShake 0.5s ease-in-out;
        }

        @keyframes screenShake {
            0%, 100% { transform: translateX(0) translateY(0); }
            10% { transform: translateX(-10px) translateY(-5px) rotate(-1deg); }
            20% { transform: translateX(10px) translateY(5px) rotate(1deg); }
            30% { transform: translateX(-8px) translateY(-3px) rotate(-0.5deg); }
            40% { transform: translateX(8px) translateY(3px) rotate(0.5deg); }
            50% { transform: translateX(-5px) translateY(-2px); }
            60% { transform: translateX(5px) translateY(2px); }
            70% { transform: translateX(-3px) translateY(-1px); }
            80% { transform: translateX(3px) translateY(1px); }
            90% { transform: translateX(-2px); }
        }

        .attack-flash {
            animation: attackFlash 0.3s ease-out;
        }

        @keyframes attackFlash {
            0% { 
                filter: brightness(1) drop-shadow(0 0 0px rgba(255, 193, 7, 0));
            }
            50% { 
                filter: brightness(1.5) drop-shadow(0 0 30px rgba(255, 193, 7, 0.8));
            }
            100% { 
                filter: brightness(1) drop-shadow(0 0 0px rgba(255, 193, 7, 0));
            }
        }

        .hit-impact {
            animation: hitImpact 0.4s ease-out;
        }

        @keyframes hitImpact {
            0% { 
                transform: scale(1);
                filter: brightness(1);
            }
            20% { 
                transform: scale(1.1);
                filter: brightness(1.8) saturate(1.5);
            }
            40% { 
                transform: scale(0.95);
                filter: brightness(1.2);
            }
            100% { 
                transform: scale(1);
                filter: brightness(1);
            }
        }

        .critical-hit {
            animation: criticalHit 0.6s ease-out;
        }

        @keyframes criticalHit {
            0% { 
                transform: scale(1);
                filter: brightness(1) drop-shadow(0 0 0px rgba(255, 0, 0, 0));
            }
            30% { 
                transform: scale(1.2) rotate(5deg);
                filter: brightness(2) drop-shadow(0 0 40px rgba(255, 0, 0, 1));
            }
            60% { 
                transform: scale(1.1) rotate(-3deg);
                filter: brightness(1.5) drop-shadow(0 0 30px rgba(255, 215, 0, 0.8));
            }
            100% { 
                transform: scale(1);
                filter: brightness(1) drop-shadow(0 0 0px rgba(255, 0, 0, 0));
            }
        }

        .block-effect {
            animation: blockEffect 0.5s ease-out;
        }

        @keyframes blockEffect {
            0% { 
                transform: scale(1);
                filter: brightness(1);
            }
            50% { 
                transform: scale(1.05);
                filter: brightness(1.3) drop-shadow(0 0 20px rgba(100, 150, 255, 0.8));
            }
            100% { 
                transform: scale(1);
                filter: brightness(1);
            }
        }

        .damage-popup {
            position: absolute;
            font-weight: 900;
            font-size: 3rem;
            pointer-events: none;
            z-index: 100;
            text-shadow: 
                0 0 10px rgba(255, 0, 0, 0.8),
                0 0 20px rgba(255, 0, 0, 0.6),
                2px 2px 4px rgba(0, 0, 0, 1);
            animation: damagePopup 1.5s ease-out forwards;
        }

        .damage-popup.critical {
            font-size: 4rem;
            color: #ffd700;
            text-shadow: 
                0 0 15px rgba(255, 215, 0, 1),
                0 0 30px rgba(255, 215, 0, 0.8),
                2px 2px 4px rgba(0, 0, 0, 1);
        }

        @keyframes damagePopup {
            0% { 
                transform: translateY(0) translateX(0) scale(0.5);
                opacity: 0;
            }
            20% { 
                transform: translateY(-20px) translateX(10px) scale(1.2);
                opacity: 1;
            }
            100% { 
                transform: translateY(-100px) translateX(30px) scale(0.8);
                opacity: 0;
            }
        }

        .attack-trail {
            position: absolute;
            width: 100px;
            height: 100px;
            background: radial-gradient(circle, rgba(255, 193, 7, 0.6) 0%, transparent 70%);
            border-radius: 50%;
            pointer-events: none;
            z-index: 25;
            animation: attackTrail 0.5s ease-out forwards;
        }

        @keyframes attackTrail {
            0% { 
                transform: scale(0) translateX(0);
                opacity: 1;
            }
            100% { 
                transform: scale(2) translateX(50px);
                opacity: 0;
            }
        }

        .charge-up {
            animation: chargeUp 0.6s ease-out;
        }

        @keyframes chargeUp {
            0% { 
                transform: scale(1);
                filter: brightness(1);
            }
            50% { 
                transform: scale(1.15);
                filter: brightness(1.4) drop-shadow(0 0 25px rgba(255, 193, 7, 0.9));
            }
            100% { 
                transform: scale(1);
                filter: brightness(1);
            }
        }

        .special-attack-glow {
            animation: specialGlow 1s ease-in-out infinite;
        }

        @keyframes specialGlow {
            0%, 100% { 
                filter: drop-shadow(0 0 10px rgba(255, 0, 255, 0.5));
            }
            50% { 
                filter: drop-shadow(0 0 30px rgba(255, 0, 255, 1));
            }
        }

        .health-bar-flash {
            animation: healthBarFlash 0.3s ease-out;
        }

        @keyframes healthBarFlash {
            0%, 100% { 
                background-color: inherit;
            }
            50% { 
                background-color: rgba(255, 255, 255, 0.3);
            }
        }

        .stamina-burst {
            animation: staminaBurst 0.4s ease-out;
        }

        @keyframes staminaBurst {
            0% { 
                transform: scaleY(1);
            }
            50% { 
                transform: scaleY(1.2);
            }
            100% { 
                transform: scaleY(1);
            }
        }

        .attack-bounce {
            animation: attackBounce 0.4s ease-out;
        }

        @keyframes attackBounce {
            0% { 
                transform: translateY(0) scale(1);
            }
            30% { 
                transform: translateY(-15px) scale(1.05);
            }
            100% { 
                transform: translateY(0) scale(1);
            }
        }

        .dodge-effect {
            animation: dodgeEffect 0.5s ease-out;
        }

        @keyframes dodgeEffect {
            0%, 100% { 
                transform: translateX(0);
            }
            25% { 
                transform: translateX(-20px);
            }
            75% { 
                transform: translateX(20px);
            }
        }

        /* Slash/Impact Effect */
        .slash-effect {
            position: absolute;
            width: 200px;
            height: 4px;
            background: linear-gradient(90deg, 
                transparent 0%,
                rgba(255, 255, 255, 0.9) 20%,
                rgba(255, 193, 7, 0.9) 50%,
                rgba(255, 255, 255, 0.9) 80%,
                transparent 100%
            );
            box-shadow: 
                0 0 20px rgba(255, 255, 255, 0.8),
                0 0 40px rgba(255, 193, 7, 0.6),
                0 0 60px rgba(255, 0, 0, 0.4);
            pointer-events: none;
            z-index: 35;
            transform-origin: left center;
            animation: slashAnimation 0.4s ease-out forwards;
        }

        .slash-effect.critical {
            width: 300px;
            height: 6px;
            background: linear-gradient(90deg, 
                transparent 0%,
                rgba(255, 215, 0, 1) 20%,
                rgba(255, 255, 255, 1) 50%,
                rgba(255, 215, 0, 1) 80%,
                transparent 100%
            );
            box-shadow: 
                0 0 30px rgba(255, 215, 0, 1),
                0 0 60px rgba(255, 215, 0, 0.8),
                0 0 90px rgba(255, 0, 0, 0.6);
        }

        @keyframes slashAnimation {
            0% { 
                opacity: 0;
                transform: scaleX(0) rotate(0deg);
            }
            30% { 
                opacity: 1;
                transform: scaleX(1) rotate(-15deg);
            }
            70% { 
                opacity: 1;
                transform: scaleX(1) rotate(15deg);
            }
            100% { 
                opacity: 0;
                transform: scaleX(0.5) rotate(0deg);
            }
        }

        /* Impact Spark Effect */
        .impact-sparks {
            position: absolute;
            width: 100%;
            height: 100%;
            pointer-events: none;
            z-index: 36;
        }

        .spark {
            position: absolute;
            width: 8px;
            height: 8px;
            background: radial-gradient(circle, rgba(255, 255, 255, 1) 0%, rgba(255, 193, 7, 0.8) 50%, transparent 100%);
            border-radius: 50%;
            animation: sparkFly 0.8s ease-out forwards;
        }

        @keyframes sparkFly {
            0% { 
                transform: translate(0, 0) scale(1);
                opacity: 1;
            }
            100% { 
                transform: translate(var(--spark-x, 50px), var(--spark-y, -50px)) scale(0);
                opacity: 0;
            }
        }

        /* Energy Wave Effect */
        .energy-wave {
            position: absolute;
            width: 150px;
            height: 150px;
            border-radius: 50%;
            background: radial-gradient(circle, 
                rgba(255, 193, 7, 0.6) 0%,
                rgba(255, 0, 0, 0.4) 40%,
                transparent 70%
            );
            pointer-events: none;
            z-index: 34;
            animation: energyWave 0.5s ease-out forwards;
        }

        @keyframes energyWave {
            0% { 
                transform: scale(0);
                opacity: 1;
            }
            100% { 
                transform: scale(3);
                opacity: 0;
            }
        }

        /* Finish Power Meter */
        .finish-power-bar {
            position: relative;
            overflow: hidden;
        }

        .finish-power-fill {
            background: linear-gradient(90deg, 
                rgba(255, 0, 0, 0.8) 0%,
                rgba(255, 100, 0, 0.9) 25%,
                rgba(255, 200, 0, 1) 50%,
                rgba(255, 255, 0, 1) 75%,
                rgba(255, 255, 255, 1) 100%
            );
            box-shadow: 
                0 0 10px rgba(255, 200, 0, 0.8),
                0 0 20px rgba(255, 100, 0, 0.6),
                inset 0 0 10px rgba(255, 255, 255, 0.5);
            animation: finishPowerGlow 2s ease-in-out infinite;
        }

        .finish-power-bar.ready .finish-power-fill {
            animation: finishPowerReady 0.5s ease-in-out infinite;
        }

        @keyframes finishPowerGlow {
            0%, 100% { 
                box-shadow: 
                    0 0 10px rgba(255, 200, 0, 0.8),
                    0 0 20px rgba(255, 100, 0, 0.6),
                    inset 0 0 10px rgba(255, 255, 255, 0.5);
            }
            50% { 
                box-shadow: 
                    0 0 20px rgba(255, 200, 0, 1),
                    0 0 40px rgba(255, 100, 0, 0.8),
                    inset 0 0 15px rgba(255, 255, 255, 0.8);
            }
        }

        @keyframes finishPowerReady {
            0%, 100% { 
                transform: scaleY(1);
                box-shadow: 
                    0 0 30px rgba(255, 200, 0, 1),
                    0 0 60px rgba(255, 100, 0, 1),
                    inset 0 0 20px rgba(255, 255, 255, 1);
            }
            50% { 
                transform: scaleY(1.1);
                box-shadow: 
                    0 0 50px rgba(255, 200, 0, 1),
                    0 0 100px rgba(255, 100, 0, 1),
                    inset 0 0 30px rgba(255, 255, 255, 1);
            }
        }

        .finish-button {
            background: linear-gradient(135deg, 
                rgba(255, 200, 0, 1) 0%,
                rgba(255, 100, 0, 1) 50%,
                rgba(255, 0, 0, 1) 100%
            );
            box-shadow: 
                0 0 20px rgba(255, 200, 0, 0.8),
                0 0 40px rgba(255, 100, 0, 0.6),
                inset 0 0 10px rgba(255, 255, 255, 0.3);
            animation: finishButtonPulse 1.5s ease-in-out infinite;
            transform: scale(1);
            transition: transform 0.2s;
        }

        .finish-button:hover {
            transform: scale(1.05);
            box-shadow: 
                0 0 30px rgba(255, 200, 0, 1),
                0 0 60px rgba(255, 100, 0, 0.8),
                inset 0 0 15px rgba(255, 255, 255, 0.5);
        }

        .finish-button:active {
            transform: scale(0.95);
        }

        @keyframes finishButtonPulse {
            0%, 100% { 
                box-shadow: 
                    0 0 20px rgba(255, 200, 0, 0.8),
                    0 0 40px rgba(255, 100, 0, 0.6),
                    inset 0 0 10px rgba(255, 255, 255, 0.3);
            }
            50% { 
                box-shadow: 
                    0 0 40px rgba(255, 200, 0, 1),
                    0 0 80px rgba(255, 100, 0, 0.8),
                    inset 0 0 20px rgba(255, 255, 255, 0.5);
            }
        }
    </style>

    <script>
        console.log('=== ARENA FIGHT SCRIPT LOADING ===');
        const characterSlug = @json(request('character', ''));
        const mobSlug = @json(request('mob', ''));
        console.log('Character Slug:', characterSlug);
        console.log('Mob Slug:', mobSlug);
        
        // Sound Effects Manager
        const SoundManager = {
            audioContext: null,
            soundsEnabled: true,
            
            init() {
                try {
                    this.audioContext = new (window.AudioContext || window.webkitAudioContext)();
                } catch (e) {
                    console.warn('Web Audio API not supported, sounds disabled');
                    this.soundsEnabled = false;
                }
            },
            
            playTone(frequency, duration, type = 'sine', volume = 0.3) {
                if (!this.soundsEnabled || !this.audioContext) return;
                
                const oscillator = this.audioContext.createOscillator();
                const gainNode = this.audioContext.createGain();
                
                oscillator.connect(gainNode);
                gainNode.connect(this.audioContext.destination);
                
                oscillator.frequency.value = frequency;
                oscillator.type = type;
                
                gainNode.gain.setValueAtTime(volume, this.audioContext.currentTime);
                gainNode.gain.exponentialRampToValueAtTime(0.01, this.audioContext.currentTime + duration);
                
                oscillator.start(this.audioContext.currentTime);
                oscillator.stop(this.audioContext.currentTime + duration);
            },
            
            playAttack() {
                this.playTone(200, 0.1, 'square', 0.2);
                setTimeout(() => this.playTone(150, 0.15, 'square', 0.15), 50);
            },
            
            playHit() {
                this.playTone(100, 0.2, 'sawtooth', 0.25);
                setTimeout(() => this.playTone(80, 0.15, 'sawtooth', 0.2), 100);
            },
            
            playSpecialAttack() {
                this.playTone(300, 0.1, 'sine', 0.15);
                setTimeout(() => this.playTone(400, 0.1, 'sine', 0.2), 100);
                setTimeout(() => this.playTone(200, 0.15, 'square', 0.25), 200);
            },
            
            playBlock() {
                this.playTone(150, 0.2, 'triangle', 0.2);
            },
            
            playCountdown() {
                this.playTone(400, 0.15, 'sine', 0.2);
            },
            
            playFightStart() {
                this.playTone(500, 0.1, 'sine', 0.3);
                setTimeout(() => this.playTone(300, 0.2, 'square', 0.25), 150);
            },
            
            playVictory() {
                this.playTone(523, 0.2, 'sine', 0.25);
                setTimeout(() => this.playTone(659, 0.2, 'sine', 0.25), 200);
                setTimeout(() => this.playTone(784, 0.3, 'sine', 0.3), 400);
            },
            
            playDefeat() {
                this.playTone(200, 0.3, 'sawtooth', 0.3);
                setTimeout(() => this.playTone(150, 0.4, 'sawtooth', 0.25), 300);
            },
            
            playRoundStart() {
                this.playTone(350, 0.2, 'sine', 0.2);
                setTimeout(() => this.playTone(400, 0.2, 'sine', 0.2), 250);
            }
        };
        
        SoundManager.init();
        
        window.arenaFightData = {
            characterSlug: characterSlug,
            mobSlug: mobSlug,
            characterState: 'idle',
            mobState: 'idle',
            character: null,
            mob: null,
            fightInProgress: false,
            fightStarted: false,
            countdown: null,
            roundCountdown: null,
            fightRound: 0,
            characterHP: 100,
            mobHP: 100,
            maxHP: 100,
            characterStamina: 100,
            mobStamina: 100,
            maxStamina: 100,
            characterFinishPower: 0,
            mobFinishPower: 0,
            maxFinishPower: 100,
            finishMoveReady: false,
            mobFinishMoveReady: false,
            winner: null,
            fightLog: [],
            charactersData: @json($charactersData),
            mobsData: @json($mobsData),
            characterX: 10,
            mobX: 90,
            characterBlock: false,
            mobBlock: false,
            initialized: false,
            victoryRewards: null,
            hasReloaded: false,
            arenaAtmosphere: 1, // 1, 2, or 3 for different backgrounds
            arenaBackgrounds: [
                '/assets/images/arena_bg.jpg',
                '/assets/images/arena_bg_2.png',
                '/assets/images/arena_bg_3.png'
            ],
            
            init() {
                if (this.initialized) {
                    console.log('=== INIT ALREADY CALLED, SKIPPING ===');
                    return;
                }
                this.initialized = true;
                
                // Set random arena atmosphere (1, 2, or 3)
                this.arenaAtmosphere = Math.floor(Math.random() * 3) + 1;
                this.updateArenaBackground();
                
                console.log('=== INIT CALLED ===');
                console.log('characterSlug:', this.characterSlug);
                console.log('mobSlug:', this.mobSlug);
                console.log('Arena Atmosphere:', this.arenaAtmosphere);
                console.log('charactersData keys:', Object.keys(this.charactersData));
                console.log('mobsData keys:', Object.keys(this.mobsData));
                
                if (this.characterSlug && this.charactersData[this.characterSlug]) {
                    this.character = this.charactersData[this.characterSlug];
                    console.log('Character loaded:', this.character);
                    
                    const charLevel = this.character.level || 1;
                    const charHP = Math.floor(100 + (charLevel * 10));
                    this.characterHP = charHP;
                    this.maxHP = charHP;
                    this.characterStamina = this.maxStamina;
                    this.characterFinishPower = 0;
                } else {
                    console.warn('Character not found for slug:', this.characterSlug);
                }
                
                if (this.mobSlug && this.mobsData[this.mobSlug]) {
                    this.mob = this.mobsData[this.mobSlug];
                    console.log('Mob loaded:', this.mob);
                    
                    const mobLevel = this.mob.level || 1;
                    const mobHP = Math.floor(100 + (mobLevel * 10));
                    this.mobHP = mobHP;
                } else {
                    console.warn('Mob not found for slug:', this.mobSlug);
                }
                
                console.log('=== INIT COMPLETE ===');
            },
            
            updateArenaBackground() {
                const arenaSand = document.querySelector('.arena-sand');
                if (arenaSand && this.arenaBackgrounds[this.arenaAtmosphere - 1]) {
                    const bgImage = this.arenaBackgrounds[this.arenaAtmosphere - 1];
                    arenaSand.style.backgroundImage = `url('${bgImage}'), linear-gradient(to bottom, #d4a574 0%, #c49060 20%, #b8865a 40%, #c49060 60%, #d4a574 80%, #c49060 100%)`;
                }
            },
            
            changeArenaAtmosphere(atmosphere) {
                // atmosphere: 1, 2, or 3
                if (atmosphere >= 1 && atmosphere <= 3) {
                    this.arenaAtmosphere = atmosphere;
                    this.updateArenaBackground();
                }
            },
            
            triggerCharacterAnimation(state) {
                this.characterState = state;
                window.dispatchEvent(new CustomEvent('character-animation-state-change', {
                    detail: {
                        characterSlug: this.characterSlug,
                        state: state
                    }
                }));
            },
            
            triggerMobAnimation(state) {
                this.mobState = state;
                window.dispatchEvent(new CustomEvent('mob-animation-state-change', {
                    detail: {
                        mobSlug: this.mobSlug,
                        state: state
                    }
                }));
            },
            
            // Visual Effects Functions
            screenShake(intensity = 1) {
                const arena = document.querySelector('.arena-background');
                if (arena) {
                    arena.classList.add('screen-shake');
                    setTimeout(() => arena.classList.remove('screen-shake'), 500);
                }
            },
            
            showDamagePopup(damage, x, y, isCritical = false) {
                const arena = document.querySelector('.arena-sand');
                if (!arena) return;
                
                const popup = document.createElement('div');
                popup.className = `damage-popup ${isCritical ? 'critical' : ''}`;
                popup.textContent = `-${damage}`;
                popup.style.position = 'absolute';
                popup.style.left = `${x}%`;
                popup.style.top = `${y}%`;
                popup.style.color = isCritical ? '#ffd700' : '#ff4444';
                popup.style.fontFamily = 'Arial, sans-serif';
                popup.style.fontWeight = '900';
                popup.style.zIndex = '100';
                
                arena.appendChild(popup);
                
                setTimeout(() => {
                    if (popup.parentNode) {
                        popup.parentNode.removeChild(popup);
                    }
                }, 1500);
            },
            
            triggerAttackFlash(target) {
                const targetElement = document.querySelector(`[x-show*="${target}"]`);
                if (targetElement) {
                    const animation = targetElement.closest('.character-slot') || targetElement;
                    animation.classList.add('attack-flash');
                    setTimeout(() => animation.classList.remove('attack-flash'), 300);
                }
            },
            
            triggerHitImpact(target) {
                const targetElement = document.querySelector(`[x-show*="${target}"]`);
                if (targetElement) {
                    const animation = targetElement.closest('.character-slot') || targetElement;
                    animation.classList.add('hit-impact');
                    setTimeout(() => animation.classList.remove('hit-impact'), 400);
                }
            },
            
            triggerCriticalHit(target) {
                const targetElement = document.querySelector(`[x-show*="${target}"]`);
                if (targetElement) {
                    const animation = targetElement.closest('.character-slot') || targetElement;
                    animation.classList.add('critical-hit');
                    setTimeout(() => animation.classList.remove('critical-hit'), 600);
                }
                this.screenShake(1.5);
            },
            
            triggerBlockEffect(target) {
                const targetElement = document.querySelector(`[x-show*="${target}"]`);
                if (targetElement) {
                    const animation = targetElement.closest('.character-slot') || targetElement;
                    animation.classList.add('block-effect');
                    setTimeout(() => animation.classList.remove('block-effect'), 500);
                }
            },
            
            triggerChargeUp(target) {
                const targetElement = document.querySelector(`[x-show*="${target}"]`);
                if (targetElement) {
                    const animation = targetElement.closest('.character-slot') || targetElement;
                    animation.classList.add('charge-up');
                    setTimeout(() => animation.classList.remove('charge-up'), 600);
                }
            },
            
            triggerAttackBounce(target) {
                const targetElement = document.querySelector(`[x-show*="${target}"]`);
                if (targetElement) {
                    const animation = targetElement.closest('.character-slot') || targetElement;
                    animation.classList.add('attack-bounce');
                    setTimeout(() => animation.classList.remove('attack-bounce'), 400);
                }
            },
            
            createSlashEffect(attackerX, targetX, targetY, isCritical = false) {
                const arena = document.querySelector('.arena-sand');
                if (!arena) return;
                
                // Calculate position between attacker and target
                const centerX = (attackerX + targetX) / 2;
                const centerY = targetY;
                
                // Create slash effect
                const slash = document.createElement('div');
                slash.className = `slash-effect ${isCritical ? 'critical' : ''}`;
                slash.style.left = `${centerX}%`;
                slash.style.top = `${centerY}%`;
                slash.style.transform = `translate(-50%, -50%) rotate(${attackerX < targetX ? -45 : 45}deg)`;
                
                arena.appendChild(slash);
                
                // Create energy wave at impact point
                const wave = document.createElement('div');
                wave.className = 'energy-wave';
                wave.style.left = `${targetX}%`;
                wave.style.top = `${targetY}%`;
                wave.style.transform = 'translate(-50%, -50%)';
                arena.appendChild(wave);
                
                // Create impact sparks
                const sparksContainer = document.createElement('div');
                sparksContainer.className = 'impact-sparks';
                sparksContainer.style.left = `${targetX}%`;
                sparksContainer.style.top = `${targetY}%`;
                sparksContainer.style.transform = 'translate(-50%, -50%)';
                
                // Generate multiple sparks
                for (let i = 0; i < 8; i++) {
                    const spark = document.createElement('div');
                    spark.className = 'spark';
                    const angle = (Math.PI * 2 * i) / 8;
                    const distance = 30 + Math.random() * 40;
                    const sparkX = Math.cos(angle) * distance;
                    const sparkY = Math.sin(angle) * distance;
                    spark.style.setProperty('--spark-x', `${sparkX}px`);
                    spark.style.setProperty('--spark-y', `${sparkY}px`);
                    spark.style.left = '50%';
                    spark.style.top = '50%';
                    sparksContainer.appendChild(spark);
                }
                
                arena.appendChild(sparksContainer);
                
                // Cleanup after animation
                setTimeout(() => {
                    if (slash.parentNode) slash.parentNode.removeChild(slash);
                    if (wave.parentNode) wave.parentNode.removeChild(wave);
                    if (sparksContainer.parentNode) sparksContainer.parentNode.removeChild(sparksContainer);
                }, 800);
            },
            
            flashHealthBar(isCharacter) {
                const bar = document.querySelector(isCharacter 
                    ? '.border-nile-turquoise' 
                    : '.border-blood-red');
                if (bar) {
                    bar.classList.add('health-bar-flash');
                    setTimeout(() => bar.classList.remove('health-bar-flash'), 300);
                }
            },
            
            calculateDamage(attackerPower, attackerLevel, isSpecialAttack = false) {
                let damage = Math.floor(attackerPower * (0.8 + Math.random() * 0.4));
                
                const levelBoost = 1 + (attackerLevel * 0.05);
                damage = Math.floor(damage * levelBoost);
                
                let isCritical = false;
                if (Math.random() < 0.2) {
                    damage = Math.floor(damage * 1.75);
                    isCritical = true;
                    this.fightLog.push(`⚡ CRITICAL HIT! Damage boosted!`);
                }
                
                if (isSpecialAttack) {
                    damage = Math.floor(damage * 1.5);
                }
                
                return {
                    damage: Math.max(damage, 5),
                    isCritical: isCritical
                };
            },
            
            getSpecialAttackInfo(isMob = false) {
                const specialAttacks = [
                    { name: 'Power Strike', stamina: 25 },
                    { name: 'Crushing Blow', stamina: 30 },
                    { name: 'Furious Assault', stamina: 35 },
                    { name: 'Ultimate Attack', stamina: 40 }
                ];
                return specialAttacks[Math.floor(Math.random() * specialAttacks.length)];
            },
            
            resetFight() {
                this.fightInProgress = false;
                this.countdown = null;
                this.roundCountdown = null;
                this.fightRound = 0;
                
                if (this.character) {
                    const charLevel = this.character.level || 1;
                    const charHP = Math.floor(100 + (charLevel * 10));
                    this.characterHP = charHP;
                    this.maxHP = charHP;
                }
                
                if (this.mob) {
                    const mobLevel = this.mob.level || 1;
                    const mobHP = Math.floor(100 + (mobLevel * 10));
                    this.mobHP = mobHP;
                }
                
                this.characterStamina = this.maxStamina;
                this.mobStamina = this.maxStamina;
                this.characterFinishPower = 0;
                this.mobFinishPower = 0;
                this.finishMoveReady = false;
                this.mobFinishMoveReady = false;
                this.characterBlock = false;
                this.mobBlock = false;
                this.winner = null;
                this.fightLog = [];
                this.characterX = 10;
                this.mobX = 90;
                this.triggerCharacterAnimation('idle');
                this.triggerMobAnimation('idle');
            },
            
            addFinishPower(isCharacter, amount) {
                if (isCharacter) {
                    this.characterFinishPower = Math.min(this.characterFinishPower + amount, this.maxFinishPower);
                    if (this.characterFinishPower >= this.maxFinishPower && !this.finishMoveReady) {
                        this.finishMoveReady = true;
                        this.fightLog.push(`⚡ FINISH MOVE READY! ${this.character.name} can unleash ultimate power!`);
                    }
                } else {
                    this.mobFinishPower = Math.min(this.mobFinishPower + amount, this.maxFinishPower);
                    if (this.mobFinishPower >= this.maxFinishPower && !this.mobFinishMoveReady) {
                        this.mobFinishMoveReady = true;
                        this.fightLog.push(`⚡ FINISH MOVE READY! ${this.mob.name} can unleash ultimate power!`);
                    }
                }
            },
            
            async executeFinishMove() {
                if (this.characterFinishPower < this.maxFinishPower || this.fightInProgress === false) {
                    return;
                }
                
                // Prevent multiple finish moves
                if (this.finishMoveReady === false) {
                    return;
                }
                
                this.fightLog.push(`🔥 ${this.character.name} unleashes FINISHING MOVE! 🔥`);
                this.characterFinishPower = 0;
                this.finishMoveReady = false;
                
                const arena = document.querySelector('.arena-background');
                const arenaSand = document.querySelector('.arena-sand');
                
                // Dramatic buildup - character glows
                this.triggerCharacterAnimation('magic');
                this.triggerChargeUp(this.characterSlug);
                this.fightLog.push(`💫 Power gathering...`);
                
                // Create multiple energy waves around character
                for (let i = 0; i < 3; i++) {
                    if (arenaSand) {
                        const wave = document.createElement('div');
                        wave.className = 'energy-wave';
                        wave.style.width = '200px';
                        wave.style.height = '200px';
                        wave.style.left = `${this.characterX}%`;
                        wave.style.top = '50%';
                        wave.style.transform = 'translate(-50%, -50%)';
                        wave.style.background = 'radial-gradient(circle, rgba(255, 215, 0, 0.8) 0%, rgba(255, 0, 0, 0.6) 40%, transparent 70%)';
                        arenaSand.appendChild(wave);
                        setTimeout(() => {
                            if (wave.parentNode) wave.parentNode.removeChild(wave);
                        }, 600);
                    }
                    await this.sleep(150);
                }
                
                // Screen flash
                if (arena) {
                    arena.style.filter = 'brightness(2.5) saturate(1.5)';
                    setTimeout(() => {
                        arena.style.filter = 'brightness(1) saturate(1)';
                    }, 150);
                }
                
                // Move character to center dramatically
                let charPos = this.characterX;
                while (charPos < 50) {
                    charPos += 8;
                    this.characterX = Math.min(charPos, 50);
                    await this.sleep(15);
                }
                
                // Multiple powerful attacks with enhanced effects
                for (let i = 0; i < 3; i++) {
                    this.triggerCharacterAnimation('attack');
                    this.triggerAttackBounce(this.characterSlug);
                    
                    // Create multiple slash effects
                    for (let j = 0; j < 2; j++) {
                        setTimeout(() => {
                            this.createSlashEffect(this.characterX, this.mobX, 50, true);
                        }, j * 50);
                    }
                    
                    SoundManager.playSpecialAttack();
                    
                    // Massive damage with critical multiplier
                    const baseDamage = this.character.attack_power || 50;
                    const finishDamage = Math.floor(baseDamage * (2.5 + (i * 0.3))); // Each hit does more damage
                    this.mobHP = Math.max(0, this.mobHP - finishDamage);
                    this.showDamagePopup(finishDamage, this.mobX, 40, true);
                    this.triggerCriticalHit(this.mobSlug);
                    this.screenShake(2.0);
                    this.flashHealthBar(false);
                    
                    // Create explosion effect at mob position
                    if (arenaSand) {
                        const explosion = document.createElement('div');
                        explosion.className = 'energy-wave';
                        explosion.style.width = '300px';
                        explosion.style.height = '300px';
                        explosion.style.left = `${this.mobX}%`;
                        explosion.style.top = '50%';
                        explosion.style.transform = 'translate(-50%, -50%)';
                        explosion.style.background = 'radial-gradient(circle, rgba(255, 255, 255, 1) 0%, rgba(255, 215, 0, 0.9) 20%, rgba(255, 0, 0, 0.7) 50%, transparent 80%)';
                        explosion.style.boxShadow = '0 0 100px rgba(255, 215, 0, 1)';
                        arenaSand.appendChild(explosion);
                        setTimeout(() => {
                            if (explosion.parentNode) explosion.parentNode.removeChild(explosion);
                        }, 500);
                    }
                    
                    this.fightLog.push(`💥 FINISH HIT ${i + 1}! ${finishDamage} damage!`);
                    
                    if (this.mobHP <= 0) {
                        this.fightLog.push(`🏆 ${this.mob.name} has been defeated by the FINISHING MOVE!`);
                        break;
                    }
                    
                    await this.sleep(300);
                }
                
                // Final screen shake
                this.screenShake(1.0);
                
                // Return to position
                while (this.characterX > 10) {
                    this.characterX -= 6;
                    await this.sleep(20);
                }
                this.characterX = 10;
                
                this.triggerCharacterAnimation('idle');
                this.triggerMobAnimation('idle');
            },
            
            async executeMobFinishMove() {
                if (this.mobFinishPower < this.maxFinishPower || this.fightInProgress === false) {
                    return;
                }
                
                // Prevent multiple finish moves
                if (this.mobFinishMoveReady === false) {
                    return;
                }
                
                this.fightLog.push(`🔥 ${this.mob.name} unleashes FINISHING MOVE! 🔥`);
                this.mobFinishPower = 0;
                this.mobFinishMoveReady = false;
                
                const arena = document.querySelector('.arena-background');
                const arenaSand = document.querySelector('.arena-sand');
                
                // Dramatic buildup - mob glows
                this.triggerMobAnimation('magic');
                this.triggerChargeUp(this.mobSlug);
                this.fightLog.push(`💫 Power gathering...`);
                
                // Create multiple energy waves around mob
                for (let i = 0; i < 3; i++) {
                    if (arenaSand) {
                        const wave = document.createElement('div');
                        wave.className = 'energy-wave';
                        wave.style.width = '200px';
                        wave.style.height = '200px';
                        wave.style.left = `${this.mobX}%`;
                        wave.style.top = '50%';
                        wave.style.transform = 'translate(-50%, -50%)';
                        wave.style.background = 'radial-gradient(circle, rgba(255, 0, 0, 0.8) 0%, rgba(139, 0, 0, 0.6) 40%, transparent 70%)';
                        arenaSand.appendChild(wave);
                        setTimeout(() => {
                            if (wave.parentNode) wave.parentNode.removeChild(wave);
                        }, 600);
                    }
                    await this.sleep(150);
                }
                
                // Screen flash
                if (arena) {
                    arena.style.filter = 'brightness(2.5) saturate(1.5)';
                    setTimeout(() => {
                        arena.style.filter = 'brightness(1) saturate(1)';
                    }, 150);
                }
                
                // Move mob to center dramatically
                let mobPos = this.mobX;
                while (mobPos > 50) {
                    mobPos -= 8;
                    this.mobX = Math.max(mobPos, 50);
                    await this.sleep(15);
                }
                
                // Multiple powerful attacks with enhanced effects
                for (let i = 0; i < 3; i++) {
                    this.triggerMobAnimation('attack');
                    this.triggerAttackBounce(this.mobSlug);
                    
                    // Create multiple slash effects
                    for (let j = 0; j < 2; j++) {
                        setTimeout(() => {
                            this.createSlashEffect(this.mobX, this.characterX, 50, true);
                        }, j * 50);
                    }
                    
                    SoundManager.playSpecialAttack();
                    
                    // Massive damage with critical multiplier
                    const baseDamage = this.mob.attack_power || 40;
                    const finishDamage = Math.floor(baseDamage * (2.5 + (i * 0.3))); // Each hit does more damage
                    this.characterHP = Math.max(0, this.characterHP - finishDamage);
                    this.showDamagePopup(finishDamage, this.characterX, 40, true);
                    this.triggerCriticalHit(this.characterSlug);
                    this.screenShake(2.0);
                    this.flashHealthBar(true);
                    
                    // Create explosion effect at character position
                    if (arenaSand) {
                        const explosion = document.createElement('div');
                        explosion.className = 'energy-wave';
                        explosion.style.width = '300px';
                        explosion.style.height = '300px';
                        explosion.style.left = `${this.characterX}%`;
                        explosion.style.top = '50%';
                        explosion.style.transform = 'translate(-50%, -50%)';
                        explosion.style.background = 'radial-gradient(circle, rgba(255, 255, 255, 1) 0%, rgba(255, 0, 0, 0.9) 20%, rgba(139, 0, 0, 0.7) 50%, transparent 80%)';
                        explosion.style.boxShadow = '0 0 100px rgba(255, 0, 0, 1)';
                        arenaSand.appendChild(explosion);
                        setTimeout(() => {
                            if (explosion.parentNode) explosion.parentNode.removeChild(explosion);
                        }, 500);
                    }
                    
                    this.fightLog.push(`💥 FINISH HIT ${i + 1}! ${finishDamage} damage!`);
                    
                    if (this.characterHP <= 0) {
                        this.fightLog.push(`🏆 ${this.character.name} has been defeated by the FINISHING MOVE!`);
                        break;
                    }
                    
                    await this.sleep(300);
                }
                
                // Final screen shake
                this.screenShake(1.0);
                
                // Return to position
                while (this.mobX < 90) {
                    this.mobX += 6;
                    await this.sleep(20);
                }
                this.mobX = 90;
                
                this.triggerCharacterAnimation('idle');
                this.triggerMobAnimation('idle');
            },
            
            async startFight() {
                console.log('=== START FIGHT CALLED ===');
                console.log('Character:', this.character);
                console.log('Mob:', this.mob);
                
                if (!this.character || !this.mob) {
                    console.error('ERROR: Cannot start fight - character or mob missing!');
                    return;
                }
                
                this.fightStarted = true;
                this.resetFight();
                this.fightInProgress = true;
                this.fightLog = [];
                
                console.log('Fight initialized, starting countdown...');
                
                for (let i = 3; i > 0; i--) {
                    this.countdown = i;
                    SoundManager.playCountdown();
                    console.log('Countdown:', i);
                    await this.sleep(1000);
                }
                this.countdown = 'FIGHT!';
                SoundManager.playFightStart();
                console.log('FIGHT!');
                await this.sleep(800);
                this.countdown = null;
                
                let round = 1;
                while (this.characterHP > 0 && this.mobHP > 0) {
                    this.fightRound = round;
                    console.log('Starting round', round);
                    
                    // Round number now shown in small status box, no large overlay
                    SoundManager.playRoundStart();
                    await this.sleep(500); // Reduced delay since no large display
                    
                    await this.executeFightRound(round);
                    
                    if (this.characterHP <= 0 || this.mobHP <= 0) {
                        console.log('Fight ended - someone died');
                        break;
                    }
                    
                    round++;
                    await this.sleep(500);
                }
                
                console.log('Fight complete, declaring winner...');
                await this.declareWinner();
                console.log('=== START FIGHT COMPLETE ===');
            },
            
            async executeFightRound(round) {
                const charPower = this.character.attack_power || 50;
                const mobPower = this.mob.attack_power || 40;
                
                this.fightLog.push(`⚔️ Round ${round} begins!`);
                
                const heroAttacksThisRound = round % 2 === 1;
                
                if (heroAttacksThisRound) {
                    this.triggerCharacterAnimation('idle');
                    this.triggerMobAnimation('idle');
                    this.fightLog.push(`${this.character.name} charges forward! 🚀`);
                    
                    // Dynamic character movement with acceleration
                    let charPos = this.characterX;
                    const charTarget = 65;
                    let speed = 2;
                    while (charPos < charTarget) {
                        speed = Math.min(speed + 0.5, 5); // Accelerate
                        charPos += speed;
                        this.characterX = Math.min(charPos, charTarget);
                        await this.sleep(30);
                    }
                    
                    const mobAction = Math.random();
                    let mobBlocking = false;
                    
                    if (mobAction < 0.3 && this.mobStamina >= 25) {
                        mobBlocking = true;
                        this.fightLog.push(`🛡️ ${this.mob.name} raises defense!`);
                        this.mobBlock = true;
                        SoundManager.playBlock();
                    } else {
                        this.fightLog.push(`${this.mob.name} braces for impact!`);
                    }
                    
                    await this.sleep(400);
                    
                    // Check if character should use finish move
                    if (this.finishMoveReady && this.characterFinishPower >= this.maxFinishPower) {
                        await this.executeFinishMove();
                        if (this.mobHP <= 0) {
                            return;
                        }
                        // Continue with normal attack after finish move
                        await this.sleep(500);
                    }
                    
                    const useSpecial = Math.random() < 0.35 && this.characterStamina >= 25;
                    const special = useSpecial ? this.getSpecialAttackInfo() : null;
                    
                    // Charge up effect
                    this.triggerChargeUp(this.characterSlug);
                    if (useSpecial) {
                        this.triggerCharacterAnimation('magic');
                    } else {
                        this.triggerCharacterAnimation('attack');
                    }
                    this.triggerMobAnimation('idle');
                    
                    if (useSpecial) {
                        this.fightLog.push(`⚡ ${this.character.name} uses ${special.name}!`);
                        this.characterStamina = Math.max(0, this.characterStamina - special.stamina);
                        SoundManager.playSpecialAttack();
                    } else {
                        this.fightLog.push(`🗡️ ${this.character.name} unleashes a devastating blow!`);
                        this.characterStamina = Math.max(0, this.characterStamina - 15);
                        SoundManager.playAttack();
                    }
                    
                    // Attack bounce and flash
                    this.triggerAttackBounce(this.characterSlug);
                    this.triggerAttackFlash(this.characterSlug);
                    await this.sleep(600);
                    
                    let damageResult = this.calculateDamage(charPower, this.character.level || 1, useSpecial);
                    let charDamage = damageResult.damage;
                    let isCritical = damageResult.isCritical;
                    
                    if (mobBlocking) {
                        charDamage = Math.floor(charDamage * 0.4);
                        this.fightLog.push(`🛡️ Block! Damage reduced!`);
                        SoundManager.playBlock();
                        this.triggerBlockEffect(this.mobSlug);
                    } else {
                        SoundManager.playHit();
                        // Create slash effect from character to mob
                        this.createSlashEffect(this.characterX, this.mobX, 50, isCritical);
                        if (isCritical) {
                            this.triggerCriticalHit(this.mobSlug);
                        } else {
                            this.triggerHitImpact(this.mobSlug);
                        }
                        this.screenShake(0.8);
                    }
                    
                    // Show damage popup
                    this.showDamagePopup(charDamage, this.mobX, 40, isCritical);
                    this.flashHealthBar(false);
                    
                    this.mobHP = Math.max(0, this.mobHP - charDamage);
                    this.triggerMobAnimation('hurt');
                    this.fightLog.push(`💥 ${this.mob.name} takes ${charDamage} damage! (HP: ${this.mobHP})`);
                    this.mobBlock = false;
                    
                    // Build finish power when character deals damage
                    const finishGain = isCritical ? 40 : 30;
                    this.addFinishPower(true, finishGain);
                    
                    // Build finish power when mob takes damage (rage/adrenaline)
                    const mobFinishGain = isCritical ? 35 : 25;
                    this.addFinishPower(false, mobFinishGain);
                    
                    this.characterStamina = Math.min(this.maxStamina, this.characterStamina + 10);
                    this.mobStamina = Math.min(this.maxStamina, this.mobStamina + 10);
                    
                    await this.sleep(600);
                    
                    if (this.mobHP <= 0) {
                        this.fightLog.push(`🏆 ${this.mob.name} has been defeated!`);
                        return;
                    }
                    
                } else {
                    this.triggerCharacterAnimation('idle');
                    this.triggerMobAnimation('idle');
                    this.fightLog.push(`${this.mob.name} charges forward! 🚀`);
                    
                    // Dynamic mob movement with acceleration
                    let mobPos = this.mobX;
                    const mobTarget = 35;
                    let speed = 2;
                    while (mobPos > mobTarget) {
                        speed = Math.min(speed + 0.5, 5); // Accelerate
                        mobPos -= speed;
                        this.mobX = Math.max(mobPos, mobTarget);
                        await this.sleep(30);
                    }
                    
                    const charAction = Math.random();
                    let charBlocking = false;
                    
                    if (charAction < 0.3 && this.characterStamina >= 25) {
                        charBlocking = true;
                        this.fightLog.push(`🛡️ ${this.character.name} raises defense!`);
                        this.characterBlock = true;
                        SoundManager.playBlock();
                    } else {
                        this.fightLog.push(`${this.character.name} prepares to defend!`);
                    }
                    
                    await this.sleep(400);
                    
                    // Check if mob should use finish move
                    if (this.mobFinishMoveReady && this.mobFinishPower >= this.maxFinishPower) {
                        await this.executeMobFinishMove();
                        if (this.characterHP <= 0) {
                            return;
                        }
                        // Continue with normal attack after finish move
                        await this.sleep(500);
                    }
                    
                    const useSpecial = Math.random() < 0.35 && this.mobStamina >= 25;
                    const special = useSpecial ? this.getSpecialAttackInfo(true) : null;
                    
                    // Charge up effect
                    this.triggerChargeUp(this.mobSlug);
                    if (useSpecial) {
                        this.triggerMobAnimation('magic');
                    } else {
                        this.triggerMobAnimation('attack');
                    }
                    this.triggerCharacterAnimation('idle');
                    
                    if (useSpecial) {
                        this.fightLog.push(`⚡ ${this.mob.name} uses ${special.name}!`);
                        this.mobStamina = Math.max(0, this.mobStamina - special.stamina);
                        SoundManager.playSpecialAttack();
                    } else {
                        this.fightLog.push(`⚔️ ${this.mob.name} strikes with fury!`);
                        this.mobStamina = Math.max(0, this.mobStamina - 15);
                        SoundManager.playAttack();
                    }
                    
                    // Attack bounce and flash
                    this.triggerAttackBounce(this.mobSlug);
                    this.triggerAttackFlash(this.mobSlug);
                    await this.sleep(600);
                    
                    let damageResult = this.calculateDamage(mobPower, this.mob.level || 1, useSpecial);
                    let mobDamage = damageResult.damage;
                    let isCritical = damageResult.isCritical;
                    
                    if (charBlocking) {
                        mobDamage = Math.floor(mobDamage * 0.4);
                        this.fightLog.push(`🛡️ Block! Damage reduced!`);
                        SoundManager.playBlock();
                        this.triggerBlockEffect(this.characterSlug);
                    } else {
                        SoundManager.playHit();
                        // Create slash effect from mob to character
                        this.createSlashEffect(this.mobX, this.characterX, 50, isCritical);
                        if (isCritical) {
                            this.triggerCriticalHit(this.characterSlug);
                        } else {
                            this.triggerHitImpact(this.characterSlug);
                        }
                        this.screenShake(0.8);
                    }
                    
                    // Show damage popup
                    this.showDamagePopup(mobDamage, this.characterX, 40, isCritical);
                    this.flashHealthBar(true);
                    
                    this.characterHP = Math.max(0, this.characterHP - mobDamage);
                    this.triggerCharacterAnimation('hurt');
                    this.fightLog.push(`💥 ${this.character.name} takes ${mobDamage} damage! (HP: ${this.characterHP})`);
                    this.characterBlock = false;
                    
                    // Build finish power when character takes damage (rage/adrenaline)
                    const finishGain = isCritical ? 35 : 25;
                    this.addFinishPower(true, finishGain);
                    
                    // Build finish power when mob deals damage
                    const mobFinishGain = isCritical ? 40 : 30;
                    this.addFinishPower(false, mobFinishGain);
                    
                    this.characterStamina = Math.min(this.maxStamina, this.characterStamina + 10);
                    this.mobStamina = Math.min(this.maxStamina, this.mobStamina + 10);
                    
                    await this.sleep(600);
                    
                    if (this.characterHP <= 0) {
                        this.fightLog.push(`☠️ ${this.character.name} has been defeated!`);
                    }
                }
                
                this.triggerCharacterAnimation('idle');
                this.triggerMobAnimation('idle');
                
                // Smooth return movement
                let returnSpeed = 3;
                while (this.characterX > 10) {
                    returnSpeed = Math.max(returnSpeed - 0.2, 1);
                    this.characterX = Math.max(this.characterX - returnSpeed, 10);
                    await this.sleep(30);
                }
                returnSpeed = 3;
                while (this.mobX < 90) {
                    returnSpeed = Math.max(returnSpeed - 0.2, 1);
                    this.mobX = Math.min(this.mobX + returnSpeed, 90);
                    await this.sleep(30);
                }
                
                this.characterX = 10;
                this.mobX = 90;
                await this.sleep(300);
            },
            
            async declareWinner() {
                await this.sleep(500);
                if (this.characterHP <= 0) {
                    this.winner = 'mob';
                    this.triggerMobAnimation('win');
                    this.triggerCharacterAnimation('lose');
                    this.fightLog.push(`☠️ DEFEAT! ${this.mob.name} WINS! ☠️`);
                    SoundManager.playDefeat();
                } else if (this.mobHP <= 0) {
                    this.winner = 'character';
                    this.triggerCharacterAnimation('win');
                    this.triggerMobAnimation('lose');
                    this.fightLog.push(`🏆 VICTORY! ${this.character.name} WINS! 🏆`);
                    SoundManager.playVictory();
                    
                    // Award resources for victory
                    try {
                        const response = await fetch('{{ route("characters.arena-victory") }}', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
                                'Accept': 'application/json',
                            },
                            credentials: 'same-origin',
                        });
                        
                        if (response.ok) {
                            const data = await response.json();
                            this.victoryRewards = data.rewards;
                            this.fightLog.push(`⚔️ Attack +${data.rewards.attack}!`);
                            this.fightLog.push(`🛡️ Defense +${data.rewards.defense}!`);
                            this.fightLog.push(`💪 Stamina +${data.rewards.stamina}!`);
                            this.fightLog.push(`✨ Resources increased!`);
                        } else {
                            console.error('Failed to award victory rewards');
                        }
                    } catch (error) {
                        console.error('Error awarding victory rewards:', error);
                    }
                }
                await this.sleep(3000);
                this.fightInProgress = false;
                
                // Auto reload page once after battle ends
                if (!this.hasReloaded) {
                    this.hasReloaded = true;
                    setTimeout(() => {
                        window.location.reload();
                    }, 2000);
                }
            },
            
            sleep(ms) {
                return new Promise(resolve => setTimeout(resolve, ms));
            },
            
            getHPPercentage(hp) {
                return (hp / this.maxHP) * 100;
            },
            
            getHPColor(hp) {
                const percent = this.getHPPercentage(hp);
                if (percent > 60) return 'bg-green-500';
                if (percent > 30) return 'bg-yellow-500';
                return 'bg-red-500';
            }
        };

        function registerArenaFightData() {
            console.log('=== REGISTERING ARENA FIGHT DATA ===');
            if (typeof Alpine === 'undefined') {
                console.log('Alpine not defined yet, retrying in 50ms...');
                setTimeout(registerArenaFightData, 50);
                return;
            }
            console.log('Alpine available, registering data component');
            Alpine.data('arenaFightData', () => window.arenaFightData);
            console.log('Component registered');
        }
        
        if (document.readyState === 'loading') {
            console.log('DOM still loading, waiting for DOMContentLoaded...');
            document.addEventListener('DOMContentLoaded', registerArenaFightData);
        } else {
            console.log('DOM already loaded, registering immediately...');
            registerArenaFightData();
        }
        
        document.addEventListener('DOMContentLoaded', function() {
            document.addEventListener('navigate', (e) => {
                window.location.href = e.detail;
            });
        });
    </script>

    <div class="py-12 min-h-screen arena-background relative">
        <div class="max-w-7xl mx-auto sm:px-6 lg:px-8 relative z-10">
            <div 
                x-data="arenaFightData"
                x-init="$nextTick(() => init())">
                
                <div x-show="!character || !mob" class="quest-card mb-6 text-center" x-cloak>
                    <p class="text-desert-sand text-xl">
                        <span class="text-ra-gold font-bold">No combatants selected!</span><br>
                        <a href="{{ route('characters.test-fight') }}" class="text-nile-turquoise hover:text-nile-turquoise/80 underline">
                            Go back to select characters
                        </a>
                    </p>
                </div>

                <template x-if="character && mob">
                    <div>
                        <div x-show="countdown !== null" 
                             x-transition:enter="transition ease-out duration-200"
                             x-transition:leave="transition ease-in duration-200"
                             class="fixed inset-0 z-50 flex items-center justify-center bg-shadow-ink/80">
                            <div class="text-center">
                                <div class="font-egyptian text-9xl text-ra-gold text-shadow-gold animate-pulse" 
                                     x-text="countdown"></div>
                            </div>
                        </div>
                        
                        <!-- Round countdown removed - now shown as small status box -->

                         <div style="color: white;" class="arena-background rounded-2xl text-desert-sand border-8 border-ra-gold/50 overflow-hidden shadow-2xl mb-8 relative">
                            <div class="battle-atmosphere"></div>
                            <div class="arena-walls"></div>
                            
                            <div class="bg-shadow-ink/80 px-8 py-6 border-b-4 border-ra-gold/30 relative">
                                <!-- Small Round Number Box -->
                                <div x-show="fightInProgress && fightRound > 0" 
                                     class="absolute top-4 left-1/2 transform -translate-x-1/2 z-10">
                                    <div class="bg-ra-gold/90 border-2 border-ra-gold rounded-lg px-4 py-2 shadow-lg">
                                        <div class="flex items-center gap-2">
                                            <span class="font-egyptian text-sm text-shadow-ink font-bold">ROUND</span>
                                            <span class="font-egyptian text-xl text-shadow-ink font-bold" x-text="fightRound"></span>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="grid grid-cols-2 gap-8">
                                    <div>
                                        <div class="flex justify-between items-center mb-2">
                                            <div class="name-plate border-nile-turquoise text-nile-turquoise" 
                                                 x-text="character.name"></div>
                                            <span class="text-desert-sand font-bold text-lg" x-text="characterHP"></span>
                                        </div>
                                        <div class="w-full h-10 bg-shadow-ink rounded-lg overflow-hidden border-2 border-nile-turquoise">
                                            <div class="h-full transition-all duration-500 flex items-center justify-center font-bold text-white text-sm"
                                                 :class="getHPColor(characterHP)"
                                                 :style="'width: ' + getHPPercentage(characterHP) + '%'">
                                                <span x-show="characterHP > 0" x-text="Math.round(getHPPercentage(characterHP)) + '%'"></span>
                                            </div>
                                        </div>
                                        <div class="mt-3 mb-2 flex justify-between items-center">
                                            <span class="text-yellow-400 text-xs font-bold">⚡ Stamina</span>
                                            <span class="text-yellow-300 text-xs" x-text="Math.round(characterStamina) + ' / ' + maxStamina"></span>
                                        </div>
                                        <div class="w-full h-6 bg-shadow-ink rounded-lg overflow-hidden border-2 border-yellow-500">
                                            <div class="h-full transition-all duration-300 bg-gradient-to-r from-yellow-400 to-yellow-500"
                                                 :style="'width: ' + (characterStamina / maxStamina) * 100 + '%'">
                                            </div>
                                        </div>
                                        
                                        <!-- Finish Power Meter -->
                                        <div class="mt-3 mb-2 flex justify-between items-center">
                                            <span class="text-orange-400 text-xs font-bold">🔥 Finish Power</span>
                                            <span class="text-orange-300 text-xs" x-text="Math.round(characterFinishPower) + ' / ' + maxFinishPower"></span>
                                        </div>
                                        <div class="w-full h-6 bg-shadow-ink rounded-lg overflow-hidden border-2 border-orange-500 finish-power-bar"
                                             :class="{'ready': finishMoveReady}">
                                            <div class="h-full transition-all duration-500 finish-power-fill"
                                                 :style="'width: ' + (characterFinishPower / maxFinishPower) * 100 + '%'">
                                            </div>
                                        </div>
                                        
                                        <!-- Finish Move Button -->
                                        <div class="mt-3" x-show="finishMoveReady && fightInProgress">
                                            <button @click="executeFinishMove()" 
                                                    class="finish-button w-full py-3 px-4 rounded-lg font-bold text-white text-sm uppercase tracking-wider">
                                                🔥 FINISH MOVE 🔥
                                            </button>
                                        </div>
                                    </div>
                                    
                                    <div>
                                        <div class="flex justify-between items-center mb-2">
                                            <span class="text-desert-sand font-bold text-lg" x-text="mobHP"></span>
                                            <div class="name-plate border-blood-red text-blood-red" 
                                                 x-text="mob.name"></div>
                                        </div>
                                        <div class="w-full h-10 bg-shadow-ink rounded-lg overflow-hidden border-2 border-blood-red">
                                            <div class="h-full transition-all duration-500 flex items-center justify-center font-bold text-white text-sm"
                                                 :class="getHPColor(mobHP)"
                                                 :style="'width: ' + getHPPercentage(mobHP) + '%'">
                                                <span x-show="mobHP > 0" x-text="Math.round(getHPPercentage(mobHP)) + '%'"></span>
                                            </div>
                                        </div>
                                        <div class="mt-3 mb-2 flex justify-between items-center">
                                            <span class="text-yellow-400 text-xs font-bold">⚡ Stamina</span>
                                            <span class="text-yellow-300 text-xs" x-text="Math.round(mobStamina) + ' / ' + maxStamina"></span>
                                        </div>
                                        <div class="w-full h-6 bg-shadow-ink rounded-lg overflow-hidden border-2 border-yellow-500">
                                            <div class="h-full transition-all duration-300 bg-gradient-to-r from-yellow-400 to-yellow-500"
                                                 :style="'width: ' + (mobStamina / maxStamina) * 100 + '%'">
                                            </div>
                                        </div>
                                        
                                        <!-- Mob Finish Power Meter -->
                                        <div class="mt-3 mb-2 flex justify-between items-center">
                                            <span class="text-orange-400 text-xs font-bold">🔥 Finish Power</span>
                                            <span class="text-orange-300 text-xs" x-text="Math.round(mobFinishPower) + ' / ' + maxFinishPower"></span>
                                        </div>
                                        <div class="w-full h-6 bg-shadow-ink rounded-lg overflow-hidden border-2 border-orange-500 finish-power-bar"
                                             :class="{'ready': mobFinishMoveReady}">
                                            <div class="h-full transition-all duration-500 finish-power-fill"
                                                 :style="'width: ' + (mobFinishPower / maxFinishPower) * 100 + '%'">
                                            </div>
                                        </div>
                                        
                                        <!-- Mob Finish Move Indicator -->
                                        <div class="mt-2" x-show="mobFinishMoveReady && fightInProgress">
                                            <div class="w-full py-2 px-3 rounded-lg font-bold text-white text-xs uppercase tracking-wider text-center"
                                                 style="background: linear-gradient(135deg, rgba(255, 0, 0, 0.8) 0%, rgba(139, 0, 0, 0.8) 100%); box-shadow: 0 0 15px rgba(255, 0, 0, 0.6);">
                                                🔥 FINISH MOVE READY 🔥
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="arena-sand relative h-96 flex items-center justify-between px-12 z-20">
                                <div class="sand-texture"></div>
                                
                                <div class="character-slot z-30 absolute transition-all duration-75" 
                                     :class="!fightStarted ? 'character-entrance' : ''"
                                     :style="`left: ${characterX}%; transform: translateX(-50%);`">
                                    <div class="character-container w-64 h-64 rounded-lg overflow-hidden"
                                         :class="winner === 'character' ? 'victory-animation' : ''">
                                        @foreach($characters as $char)
                                            <div x-show="characterSlug === '{{ $char->slug }}'" style="display: none;">
                                                <x-character-animation 
                                                    :character="$char"
                                                    state="idle"
                                                    direction="right"
                                                    :autoplay="true"
                                                />
                                            </div>
                                        @endforeach
                                    </div>
                                </div>

                                <!-- <div x-show="!fightStarted" class="text-center z-30 absolute left-1/2 top-1/2 transform -translate-x-1/2 -translate-y-1/2">
                                    <div class="font-egyptian text-7xl text-ra-gold text-shadow-gold">⚔️</div>
                                </div> -->

                                <div class="character-slot z-30 absolute transition-all duration-75"
                                     :class="!fightStarted ? 'mob-entrance' : ''"
                                     :style="mobSlug === 'desert-rat' 
                                         ? `right: ${100 - mobX}%; transform: translateX(50%); bottom: -50px;` 
                                         : `right: ${100 - mobX}%; transform: translateX(50%);`">
                                    <div class="mob-container w-64 h-64 rounded-lg overflow-hidden"
                                         :class="winner === 'mob' ? 'victory-animation' : ''">
                                        @foreach($mobs as $mobItem)
                                            <div x-show="mobSlug === '{{ $mobItem->slug }}'" style="display: none;">
                                                <x-mob-animation 
                                                    :mob="$mobItem"
                                                    state="idle"
                                                    direction="left"
                                                    :autoplay="true"
                                                />
                                            </div>
                                        @endforeach
                                    </div>
                                </div>
                            </div>

                            <!-- Round display removed - now shown as small status box in header -->
                        </div>

                        <div x-show="fightLog.length > 0" class="quest-card mb-8">
                            <h3 class="font-egyptian text-2xl text-ra-gold mb-4 text-center">⚔️ Battle Log</h3>
                            <div class="max-h-56 overflow-y-auto space-y-2 bg-shadow-ink/30 rounded-lg p-4">
                                <template x-for="(log, index) in fightLog" :key="index">
                                    <div class="text-desert-sand text-sm font-mono p-2 bg-shadow-ink/50 rounded border-l-4 border-ra-gold/50" 
                                         x-text="log"></div>
                                </template>
                            </div>
                        </div>

                        <div x-show="winner" class="quest-card mb-8 text-center p-8"
                             :class="winner === 'character' ? 'bg-ra-gold/20 border-4 border-ra-gold shadow-gold-glow' : 'bg-blood-red/20 border-4 border-blood-red'">
                            <template x-if="winner === 'character'">
                                <div>
                                    <div class="font-egyptian text-6xl text-ra-gold mb-4 text-shadow-gold victory-animation">🏆 VICTORY! 🏆</div>
                                    <div class="text-desert-sand text-2xl mb-6 font-semibold" x-text="character.name + ' emerges victorious from the arena!'"></div>
                                    
                                    <div x-show="victoryRewards" class="mb-6 p-4 bg-shadow-ink/50 rounded-lg border-2 border-ra-gold/50">
                                        <div class="text-ra-gold text-xl font-bold mb-3">✨ Victory Rewards ✨</div>
                                        <div class="grid grid-cols-3 gap-4 text-center">
                                            <div class="p-3 bg-shadow-ink/70 rounded-lg">
                                                <div class="text-3xl mb-2">⚔️</div>
                                                <div class="text-desert-sand font-semibold">Attack</div>
                                                <div class="text-ra-gold text-lg font-bold" x-text="'+' + (victoryRewards?.attack || 0)"></div>
                                            </div>
                                            <div class="p-3 bg-shadow-ink/70 rounded-lg">
                                                <div class="text-3xl mb-2">🛡️</div>
                                                <div class="text-desert-sand font-semibold">Defense</div>
                                                <div class="text-ra-gold text-lg font-bold" x-text="'+' + (victoryRewards?.defense || 0)"></div>
                                            </div>
                                            <div class="p-3 bg-shadow-ink/70 rounded-lg">
                                                <div class="text-3xl mb-2">💪</div>
                                                <div class="text-desert-sand font-semibold">Stamina</div>
                                                <div class="text-ra-gold text-lg font-bold" x-text="'+' + (victoryRewards?.stamina || 0)"></div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <button 
                                        @click="$dispatch('navigate', '/characters/test-fight')"
                                        class="px-8 py-3 bg-ra-gold hover:bg-ra-gold/80 border-2 border-ra-gold rounded-lg text-shadow-ink font-semibold transition-all transform hover:scale-105">
                                        Return to Selection
                                    </button>
                                </div>
                            </template>
                            <template x-if="winner === 'mob'">
                                <div>
                                    <div class="font-egyptian text-6xl text-blood-red mb-4">☠️ DEFEAT! ☠️</div>
                                    <div class="text-desert-sand text-2xl mb-6 font-semibold" x-text="mob.name + ' has defeated you in glorious combat!'"></div>
                                    <button 
                                        @click="$dispatch('navigate', '/characters/test-fight')"
                                        class="px-8 py-3 bg-blood-red hover:bg-blood-red/80 border-2 border-blood-red rounded-lg text-white font-semibold transition-all transform hover:scale-105">
                                        Return to Selection
                                    </button>
                                </div>
                            </template>
                        </div>

                        <div class="text-center mb-8" x-show="!fightStarted">
                            <button 
                                @click="startFight()"
                                class="px-12 py-6 bg-blood-red hover:bg-blood-red/80 border-4 border-ra-gold rounded-xl text-white font-egyptian text-3xl shadow-gold-glow transition-all transform hover:scale-110 active:scale-95">
                                ⚔️ BEGIN BATTLE ⚔️
                            </button>
                        </div>

                        <div class="text-center" x-show="!fightInProgress && !winner">
                            <a href="{{ route('characters.test-fight') }}" class="cartouche-button">
                                ← Back to Selection
                            </a>
                        </div>
                    </div>
                </template>
            </div>
        </div>
    </div>
</x-app-layout>