@props(['battleResult', 'user', 'mob'])

@php
use Illuminate\Support\Facades\Storage;
$isWin = $battleResult['result'] === 'win';

// Get character images from equipped character
$userImage = null;
$userImageAnimation = null;
if ($user->character) {
    if ($isWin) {
        // Use winning images (prefer animation, fallback to static)
        $userImageAnimation = $user->character->image_winning_animation ?? null;
        $userImage = $user->character->image_winning ?? $user->character->image_win ?? $user->character->image_normal ?? null;
    } else {
        // Use lost images (prefer animation, fallback to static)
        $userImageAnimation = $user->character->image_lost_animation ?? null;
        $userImage = $user->character->image_lost ?? $user->character->image_loss ?? $user->character->image_normal ?? null;
    }
    // Convert to Storage URL if it's a stored file
    if ($userImage && !filter_var($userImage, FILTER_VALIDATE_URL)) {
        $userImage = Storage::url($userImage);
    }
    if ($userImageAnimation && !filter_var($userImageAnimation, FILTER_VALIDATE_URL)) {
        $userImageAnimation = Storage::url($userImageAnimation);
    }
}

// Fallback to user's old character images if no character equipped
if (!$userImage) {
    $userImage = $isWin ? ($user->character_image_win ?? $user->character_image_normal ?? null) : ($user->character_image_loss ?? $user->character_image_normal ?? null);
    if ($userImage && !filter_var($userImage, FILTER_VALIDATE_URL)) {
        $userImage = Storage::url($userImage);
    }
}

$mobImage = $isWin ? ($mob->image_loss ?? $mob->image_normal ?? $mob->sprite_source ?? null) : ($mob->image_win ?? $mob->image_normal ?? $mob->sprite_source ?? null);
if ($mobImage && !filter_var($mobImage, FILTER_VALIDATE_URL)) {
    $mobImage = Storage::url($mobImage);
}

$rewards = $battleResult['rewards'] ?? null;
@endphp

<div x-data="{ show: true }" 
     x-show="show" 
     x-cloak
     x-transition:enter="transition ease-out duration-300"
     x-transition:enter-start="opacity-0 scale-95"
     x-transition:enter-end="opacity-100 scale-100"
     x-transition:leave="transition ease-in duration-200"
     x-transition:leave-start="opacity-100 scale-100"
     x-transition:leave-end="opacity-0 scale-95"
     class="fixed inset-0 z-50 flex items-center justify-center bg-black/80 backdrop-blur-sm"
     style="display: flex;">
    
    <div class="relative max-w-4xl w-full mx-4 bg-gradient-to-br from-papyrus-dark via-shadow-ink to-papyrus-dark border-4 {{ $isWin ? 'border-ra-gold' : 'border-blood-red' }} rounded-2xl shadow-2xl overflow-hidden">
        <!-- Close Button -->
        <button @click="show = false" 
                class="absolute top-4 right-4 z-10 text-desert-sand hover:text-ra-gold transition-colors">
            <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
            </svg>
        </button>

        <!-- Result Header -->
        <div class="text-center py-6 {{ $isWin ? 'bg-gradient-to-r from-ra-gold/20 to-yellow-600/20' : 'bg-gradient-to-r from-blood-red/20 to-red-900/20' }}">
            @if($isWin)
                <div class="text-6xl mb-2 animate-pulse">👁️</div>
                <h2 class="font-egyptian text-5xl text-ra-gold text-shadow-gold mb-2">Victory!</h2>
                <p class="text-desert-sand text-xl">You have vanquished {{ $mob->name }}!</p>
            @else
                <div class="text-6xl mb-2">💀</div>
                <h2 class="font-egyptian text-5xl text-blood-red mb-2">Defeated!</h2>
                <p class="text-desert-sand text-xl">You were overcome by {{ $mob->name }}.</p>
            @endif
        </div>

        <!-- Images Section -->
        <div class="grid grid-cols-2 gap-4 p-6 bg-papyrus-dark/30">
            <!-- User Image -->
            <div class="text-center">
                <h3 class="font-egyptian text-xl text-nile-turquoise mb-3">{{ $user->name }}</h3>
                @if($user->character)
                    <div class="w-full h-64 mx-auto">
                        <x-character-animation 
                            :character="$user->character" 
                            :state="$isWin ? 'win' : 'lose'" 
                            class="w-full h-64 rounded-lg border-4 {{ $isWin ? 'border-ra-gold shadow-gold-glow' : 'border-blood-red' }}"
                            x-ref="userCharacterAnimation"
                        />
                    </div>
                @elseif($userImageAnimation)
                    @if(str_ends_with($userImageAnimation, '.gif') || str_contains($userImageAnimation, '.gif'))
                        <img src="{{ $userImageAnimation }}" 
                             alt="{{ $user->name }}" 
                             class="w-full h-64 object-cover rounded-lg border-4 {{ $isWin ? 'border-ra-gold shadow-gold-glow' : 'border-blood-red' }} mx-auto">
                    @else
                        <video src="{{ $userImageAnimation }}" 
                               autoplay loop muted
                               class="w-full h-64 object-cover rounded-lg border-4 {{ $isWin ? 'border-ra-gold shadow-gold-glow' : 'border-blood-red' }} mx-auto"></video>
                    @endif
                @elseif($userImage)
                    <img src="{{ $userImage }}" 
                         alt="{{ $user->name }}" 
                         class="w-full h-64 object-cover rounded-lg border-4 {{ $isWin ? 'border-ra-gold shadow-gold-glow' : 'border-blood-red' }} mx-auto">
                @else
                    <div class="w-full h-64 bg-gradient-to-br {{ $isWin ? 'from-ra-gold/20 to-yellow-600/20 border-ra-gold' : 'from-blood-red/20 to-red-900/20 border-blood-red' }} border-4 rounded-lg flex items-center justify-center">
                        <span class="text-8xl">{{ $isWin ? '👁️' : '💀' }}</span>
                    </div>
                @endif
            </div>

            <!-- Mob Image -->
            <div class="text-center">
                <h3 class="font-egyptian text-xl text-blood-red mb-3">{{ $mob->name }}</h3>
                @if($mobImage)
                    <img src="{{ $mobImage }}" 
                         alt="{{ $mob->name }}" 
                         class="w-full h-64 object-cover rounded-lg border-4 {{ $isWin ? 'border-blood-red' : 'border-ra-gold shadow-gold-glow' }} mx-auto">
                @else
                    <div class="w-full h-64 bg-gradient-to-br {{ $isWin ? 'from-blood-red/20 to-red-900/20 border-blood-red' : 'from-ra-gold/20 to-yellow-600/20 border-ra-gold' }} border-4 rounded-lg flex items-center justify-center">
                        <span class="text-8xl">👹</span>
                    </div>
                @endif
            </div>
        </div>

        <!-- Rewards Section (if won) -->
        @if($isWin && $rewards)
        <div class="p-6 bg-gradient-to-br from-ra-gold/20 to-yellow-600/20 border-t-2 border-ra-gold">
            <h3 class="font-egyptian text-2xl text-ra-gold mb-4 text-center">Divine Rewards</h3>
            <div class="grid grid-cols-3 gap-4">
                <div class="text-center p-4 bg-papyrus-dark/50 rounded-lg border border-ra-gold/30">
                    <div class="text-3xl mb-2">✨</div>
                    <div class="font-stat text-nile-turquoise text-2xl">+{{ number_format($rewards['xp'] ?? 0) }}</div>
                    <div class="text-xs text-desert-sand">Divine Energy</div>
                </div>
                <div class="text-center p-4 bg-papyrus-dark/50 rounded-lg border border-ra-gold/30">
                    <div class="text-3xl mb-2">💰</div>
                    <div class="font-stat text-ra-gold text-2xl">+{{ number_format($rewards['gold'] ?? 0) }}</div>
                    <div class="text-xs text-desert-sand">Sacred Gold</div>
                </div>
                @if(isset($rewards['premium_coins']) && $rewards['premium_coins'] > 0)
                <div class="text-center p-4 bg-papyrus-dark/50 rounded-lg border border-ra-gold/30">
                    <div class="text-3xl mb-2">💎</div>
                    <div class="font-stat text-nile-turquoise text-2xl">+{{ number_format($rewards['premium_coins']) }}</div>
                    <div class="text-xs text-desert-sand">Premium Coins</div>
                </div>
                @else
                <div class="text-center p-4 bg-papyrus-dark/50 rounded-lg border border-ra-gold/30">
                    <div class="text-3xl mb-2">📊</div>
                    <div class="font-stat text-desert-sand text-2xl">-</div>
                    <div class="text-xs text-desert-sand">No Premium</div>
                </div>
                @endif
            </div>
        </div>
        @endif

        <!-- Battle Stats -->
        <div class="p-6 bg-papyrus-dark/50 border-t-2 border-ra-gold/30">
            <h4 class="font-egyptian text-xl text-ra-gold mb-3 text-center">Battle Chronicles</h4>
            <div class="grid grid-cols-2 gap-4 text-sm">
                <div class="text-center p-3 bg-papyrus-dark/30 rounded border border-ra-gold/20">
                    <div class="text-desert-sand/70 mb-1">Turns</div>
                    <div class="font-stat text-nile-turquoise text-xl">{{ count($battleResult['turns'] ?? []) }}</div>
                </div>
                <div class="text-center p-3 bg-papyrus-dark/30 rounded border border-ra-gold/20">
                    <div class="text-desert-sand/70 mb-1">Your HP Remaining</div>
                    <div class="font-stat text-nile-turquoise text-xl">{{ $battleResult['player']['hp'] ?? 0 }} / {{ $battleResult['player']['max_hp'] ?? 0 }}</div>
                </div>
            </div>
        </div>

        <!-- Action Buttons -->
        <div class="p-6 bg-papyrus-dark/30 flex gap-4 justify-center">
            <button @click="show = false; window.location.reload();" 
                    class="cartouche-button">
                Continue
            </button>
            <a href="{{ route('battle.index') }}" 
               class="bg-papyrus-dark hover:bg-shadow-ink border-2 border-ra-gold/30 text-desert-sand font-bold py-3 px-6 rounded-lg transition-all duration-300">
                Battle Again
            </a>
        </div>
    </div>
</div>

