@props([
    'mob',
    'state' => 'idle',
    'direction' => 'left', // 'left' or 'right' - left faces character (default)
    'class' => '',
    'autoplay' => true,
    'showControls' => false,
    'fallbackImage' => null,
    'loading' => true,
])

@php
    // Get looping states from config (same as characters)
    $loopingStates = config('character_animations.looping_states', ['idle', 'magic']);
    $isLooping = in_array($state, $loopingStates);
    
    // Check if PNG sequence exists (with direction support)
    $hasPngSequence = $mob->hasPngSequence($state, $direction);
    $pngSequence = $hasPngSequence ? $mob->getPngSequence($state, $direction) : [];
    
    // Get fallback image
    $fallbackImg = null;
    if ($fallbackImage) {
        $fallbackImg = $fallbackImage;
    } elseif ($mob->image_normal) {
        $fallbackImg = $mob->image_normal;
    } elseif ($hasPngSequence && count($pngSequence) > 0) {
        // Use first frame as fallback
        $fallbackImg = $pngSequence[0];
    }
    
    // Pre-calculate values for JavaScript
    $autoReturnValue = in_array($state, config('character_animations.auto_return_states', []));
    $isLoopingJson = json_encode($isLooping);
    $autoReturnValueJson = json_encode($autoReturnValue);
    $loadingJson = json_encode($loading);
    $loopingStatesJson = json_encode(config('character_animations.looping_states', []));
    $autoReturnStatesJson = json_encode(config('character_animations.auto_return_states', []));
    $stayOnLastFrameStatesJson = json_encode(config('character_animations.stay_on_last_frame_states', []));
    $pngSequenceJson = json_encode($pngSequence);
@endphp

<div 
    class="mob-animation-wrapper relative {{ $class }}"
    data-mob-slug="{{ $mob->slug }}"
    data-initial-state="{{ $state }}"
    data-direction="{{ $direction }}"
    data-png-sequence='@json($pngSequence)'
    data-has-png-sequence="{{ $hasPngSequence ? 'true' : 'false' }}"
    data-is-looping="{{ $isLooping ? 'true' : 'false' }}"
    data-auto-return="{{ $autoReturnValue ? 'true' : 'false' }}"
    x-data="mobAnimation({
        initialState: $el.dataset.initialState,
        mobSlug: $el.dataset.mobSlug,
        direction: $el.dataset.direction || 'left',
        looping: $el.dataset.isLooping === 'true',
        autoReturn: $el.dataset.autoReturn === 'true',
        hasPngSequence: $el.dataset.hasPngSequence === 'true',
        pngSequence: JSON.parse($el.dataset.pngSequence || '[]')
    })"
    @@mob-animation-state-change.window="handleStateChange($event.detail)"
    x-bind:data-state="state"
>
    <!-- Loading Animation -->
    <div 
        x-show="loading" 
        x-cloak
        class="mob-animation-loading absolute inset-0 flex items-center justify-center bg-papyrus-dark/50 rounded-lg z-10"
        style="display: none;"
    >
        <div class="flex flex-col items-center gap-2">
            <div class="animate-spin rounded-full h-8 w-8 border-b-2 border-blood-red"></div>
            <span class="text-xs text-desert-sand/70">Loading animation...</span>
        </div>
    </div>

    <!-- PNG Sequence Animation Canvas -->
    @if($hasPngSequence && count($pngSequence) > 0)
        <canvas 
            x-ref="sequenceCanvas"
            class="mob-sequence max-w-full h-auto {{ $showControls ? '' : 'pointer-events-none' }}"
            :class="{ 'opacity-0': loading && !showFallback }"
            style="background: transparent;"
        ></canvas>
    @elseif($fallbackImg)
        <!-- Fallback Image -->
        <img 
            src="{{ str_starts_with($fallbackImg, '/') ? $fallbackImg : Storage::url($fallbackImg) }}" 
            alt="{{ $mob->name }}" 
            class="mob-fallback-image max-w-full h-auto"
        >
    @else
        <!-- No animation available -->
        <div class="w-full h-64 bg-papyrus-dark/50 rounded-lg flex items-center justify-center border-2 border-blood-red/30">
            <span class="text-desert-sand/50">No animation available</span>
        </div>
    @endif

    <!-- Fallback image overlay (shown on error) -->
    @if($fallbackImg)
        <img 
            x-show="showFallback"
            x-cloak
            src="{{ str_starts_with($fallbackImg, '/') ? $fallbackImg : Storage::url($fallbackImg) }}" 
            alt="{{ $mob->name }}" 
            class="mob-fallback-image max-w-full h-auto absolute inset-0 z-20"
            style="display: none;"
        >
    @endif
</div>

@push('scripts')
<script>
document.addEventListener('alpine:init', () => {
    Alpine.data('mobAnimation', (config) => ({
        state: config.initialState,
        mobSlug: config.mobSlug,
        direction: config.direction || 'left',
        isLooping: config.looping,
        autoReturn: config.autoReturn,
        hasPngSequence: config.hasPngSequence || false,
        pngSequence: config.pngSequence || [],
        loading: {!! $loadingJson !!},
        showFallback: false,
        currentFrame: 0,
        sequenceInterval: null,
        sequenceImages: [],
        
        init() {
            // Listen for global mob animation state changes
            window.addEventListener('mob-animation-state-change', (e) => {
                this.handleStateChange(e.detail);
            });
            
            // Watch for external state prop changes
            this.$watch('state', (newState) => {
                if (newState !== this.state) {
                    this.updateSequence();
                }
            });
            
            // Initialize animation
            this.$nextTick(() => {
                const isVisible = () => {
                    const rect = this.$el.getBoundingClientRect();
                    return rect.width > 0 && rect.height > 0 && 
                           window.getComputedStyle(this.$el).display !== 'none' &&
                           window.getComputedStyle(this.$el).visibility !== 'hidden';
                };
                
                const initAnimation = () => {
                    if (this.hasPngSequence && this.pngSequence.length > 0) {
                        this.initSequence();
                    } else {
                        this.loading = false;
                    }
                };
                
                if (isVisible()) {
                    setTimeout(initAnimation, 50);
                } else {
                    if ('IntersectionObserver' in window) {
                        const observer = new IntersectionObserver((entries) => {
                            entries.forEach(entry => {
                                if (entry.isIntersecting) {
                                    initAnimation();
                                    observer.disconnect();
                                }
                            });
                        }, { threshold: 0.1 });
                        
                        observer.observe(this.$el);
                        
                        setTimeout(() => {
                            if (isVisible()) {
                                initAnimation();
                                observer.disconnect();
                            }
                        }, 1000);
                    } else {
                        setTimeout(initAnimation, 200);
                    }
                }
            });
        },
        
        destroy() {
            this.stopSequence();
        },
        
        handleStateChange(event) {
            if (event.mobSlug === this.mobSlug || !event.mobSlug) {
                const oldState = this.state;
                
                if (oldState === event.state) {
                    return;
                }
                
                this.state = event.state;
                
                // Always reload sequence for new state
                this.stopSequence();
                this.sequenceImages = [];
                this.currentFrame = 0;
                this.pngSequence = [];
                this.loading = true;
                
                // Discover sequence for new state
                const testPathDirection = `/assets/mobs/${this.mobSlug}/${this.state}/${this.direction}/01.png`;
                const testPathBase = `/assets/mobs/${this.mobSlug}/${this.state}/01.png`;
                
                const img = new Image();
                img.onload = () => {
                    this.hasPngSequence = true;
                    this.loadSequenceForState(this.state);
                };
                img.onerror = () => {
                    const imgBase = new Image();
                    imgBase.onload = () => {
                        this.hasPngSequence = true;
                        this.loadSequenceForState(this.state);
                    };
                    imgBase.onerror = () => {
                        this.hasPngSequence = false;
                        this.pngSequence = [];
                        this.stopSequence();
                        this.loading = false;
                    };
                    imgBase.src = testPathBase;
                };
                img.src = testPathDirection;
            }
        },
        
        loadSequenceForState(state) {
            const frames = [];
            let currentFrame = 1;
            let consecutiveErrors = 0;
            const maxConsecutiveErrors = 3;
            let useDirectionFolder = true;
            
            const checkNextFrame = () => {
                if (consecutiveErrors >= maxConsecutiveErrors) {
                    if (frames.length > 0) {
                        frames.sort();
                        this.pngSequence = frames;
                        this.updateSequence();
                    } else {
                        if (useDirectionFolder) {
                            useDirectionFolder = false;
                            consecutiveErrors = 0;
                            currentFrame = 1;
                            checkNextFrame();
                        } else {
                            this.hasPngSequence = false;
                            this.loading = false;
                        }
                    }
                    return;
                }
                
                const directionPath = useDirectionFolder ? `/${this.direction}` : '';
                const framePath = `/assets/mobs/${this.mobSlug}/${state}${directionPath}/${currentFrame.toString().padStart(2, '0')}.png`;
                const img = new Image();
                
                img.onload = () => {
                    frames.push(framePath);
                    consecutiveErrors = 0;
                    currentFrame++;
                    checkNextFrame();
                };
                
                img.onerror = () => {
                    consecutiveErrors++;
                    if (consecutiveErrors >= maxConsecutiveErrors) {
                        if (frames.length > 0) {
                            frames.sort();
                            this.pngSequence = frames;
                            this.updateSequence();
                        } else {
                            if (useDirectionFolder) {
                                useDirectionFolder = false;
                                consecutiveErrors = 0;
                                currentFrame = 1;
                                checkNextFrame();
                            } else {
                                this.hasPngSequence = false;
                                this.loading = false;
                            }
                        }
                    } else {
                        currentFrame++;
                        checkNextFrame();
                    }
                };
                
                img.src = framePath;
            };
            
            checkNextFrame();
        },
        
        initSequence() {
            const canvas = this.$refs.sequenceCanvas;
            if (!canvas || this.pngSequence.length === 0) {
                this.loading = false;
                return;
            }
            
            this.loading = true;
            this.currentFrame = 0;
            this.sequenceImages = [];
            
            let loadedCount = 0;
            const totalFrames = this.pngSequence.length;
            
            if (totalFrames === 0) {
                this.loading = false;
                return;
            }
            
            this.pngSequence.forEach((src, index) => {
                const img = new Image();
                img.onload = () => {
                    loadedCount++;
                    if (loadedCount === totalFrames) {
                        this.loading = false;
                        this.startSequence();
                    }
                };
                img.onerror = () => {
                    console.warn(`Failed to load mob sequence frame ${index}: ${src}`);
                    loadedCount++;
                    if (loadedCount === totalFrames) {
                        this.loading = false;
                        if (this.sequenceImages.length > 0) {
                            this.startSequence();
                        } else {
                            this.showFallback = true;
                        }
                    }
                };
                img.src = src;
                this.sequenceImages.push(img);
            });
        },
        
        startSequence() {
            const canvas = this.$refs.sequenceCanvas;
            if (!canvas || this.sequenceImages.length === 0) return;
            
            const firstImg = this.sequenceImages[0];
            canvas.width = firstImg.width;
            canvas.height = firstImg.height;
            
            this.drawFrame(0);
            
            const frameDelay = 33;
            this.currentFrame = 0;
            
            const animate = () => {
                if (this.currentFrame < this.sequenceImages.length) {
                    this.drawFrame(this.currentFrame);
                    this.currentFrame++;
                    
                    if (this.currentFrame >= this.sequenceImages.length) {
                        const loopingStates = {!! $loopingStatesJson !!};
                        if (this.isLooping || loopingStates.includes(this.state)) {
                            this.currentFrame = 0;
                        } else {
                            const autoReturnStates = {!! $autoReturnStatesJson !!};
                            const stayOnLastFrameStates = {!! $stayOnLastFrameStatesJson !!};
                            
                            if (autoReturnStates.includes(this.state)) {
                                setTimeout(() => {
                                    this.state = 'idle';
                                    this.handleStateChange({ state: 'idle', mobSlug: this.mobSlug });
                                }, 500);
                            } else if (stayOnLastFrameStates.includes(this.state)) {
                                this.currentFrame = this.sequenceImages.length - 1;
                            }
                            return;
                        }
                    }
                    
                    this.sequenceInterval = setTimeout(animate, frameDelay);
                }
            };
            
            this.sequenceInterval = setTimeout(animate, frameDelay);
        },
        
        drawFrame(frameIndex) {
            const canvas = this.$refs.sequenceCanvas;
            if (!canvas || !this.sequenceImages[frameIndex]) return;
            
            const ctx = canvas.getContext('2d');
            ctx.clearRect(0, 0, canvas.width, canvas.height);
            ctx.drawImage(this.sequenceImages[frameIndex], 0, 0);
        },
        
        stopSequence() {
            if (this.sequenceInterval) {
                clearTimeout(this.sequenceInterval);
                this.sequenceInterval = null;
            }
        },
        
        updateSequence() {
            this.stopSequence();
            if (this.hasPngSequence && this.pngSequence.length > 0) {
                this.initSequence();
            } else {
                this.loadSequenceForState(this.state);
            }
        }
    }));
});
</script>
@endpush


