<?php

namespace Tests\Feature\Api;

use Tests\TestCase;
use App\Models\User;
use App\Models\Mob;
use App\Models\Level;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Laravel\Sanctum\Sanctum;

class BattleTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        $this->seed(\Database\Seeders\LevelsTableSeeder::class);
    }

    /**
     * Test full battle simulation (auto mode).
     */
    public function test_full_battle_simulation_auto_mode(): void
    {
        $user = User::factory()->create([
            'level' => 5,
            'xp' => 0,
            'max_hp' => 150,
            'current_hp' => 150,
            'attack' => 15,
            'defense' => 10,
            'stamina' => 150,
            'luck' => 5,
        ]);

        $mob = Mob::factory()->create([
            'level' => 5,
            'hp' => 100,
            'attack' => 12,
            'defense' => 8,
            'speed' => 10,
            'luck' => 3,
            'xp_reward' => 50,
            'gold_reward' => 25,
        ]);

        Sanctum::actingAs($user);

        $response = $this->postJson('/api/battle/start', [
            'mob_id' => $mob->id,
            'mode' => 'auto',
        ]);

        $response->assertStatus(200);
        $response->assertJsonStructure([
            'battle_id',
            'mode',
            'user_id',
            'mob',
            'player',
            'turns',
            'result',
            'rewards',
        ]);

        $data = $response->json();
        $this->assertNotNull($data['result']);
        $this->assertContains($data['result'], ['win', 'lose']);
        $this->assertNotEmpty($data['turns']);
    }

    /**
     * Test manual battle turn-by-turn.
     */
    public function test_manual_battle_turn_by_turn(): void
    {
        $user = User::factory()->create([
            'level' => 5,
            'max_hp' => 150,
            'current_hp' => 150,
            'attack' => 15,
            'defense' => 10,
            'stamina' => 150,
            'luck' => 5,
        ]);

        $mob = Mob::factory()->create([
            'level' => 5,
            'hp' => 100,
            'attack' => 12,
            'defense' => 8,
            'speed' => 5, // Lower speed so player goes first
        ]);

        Sanctum::actingAs($user);

        // Start battle
        $startResponse = $this->postJson('/api/battle/start', [
            'mob_id' => $mob->id,
            'mode' => 'manual',
        ]);

        $startResponse->assertStatus(200);
        $battleId = $startResponse->json()['battle_id'];

        // Perform a turn
        $turnResponse = $this->postJson('/api/battle/turn', [
            'battle_id' => $battleId,
            'action' => 'attack',
        ]);

        $turnResponse->assertStatus(200);
        $turnData = $turnResponse->json();
        $this->assertArrayHasKey('turns', $turnData);
        $this->assertNotEmpty($turnData['turns']);
    }

    /**
     * Test battle rewards calculation.
     */
    public function test_battle_rewards_calculation(): void
    {
        $user = User::factory()->create([
            'level' => 5,
            'max_hp' => 200,
            'current_hp' => 200,
            'attack' => 30,
            'defense' => 15,
            'stamina' => 200,
        ]);

        $mob = Mob::factory()->create([
            'level' => 5,
            'hp' => 50, // Low HP so user wins quickly
            'attack' => 5,
            'defense' => 5,
            'speed' => 5,
            'xp_reward' => 100,
            'gold_reward' => 50,
        ]);

        Sanctum::actingAs($user);

        $response = $this->postJson('/api/battle/start', [
            'mob_id' => $mob->id,
            'mode' => 'auto',
        ]);

        $response->assertStatus(200);
        $data = $response->json();

        if ($data['result'] === 'win') {
            $this->assertNotNull($data['rewards']);
            $this->assertGreaterThan(0, $data['rewards']['xp']);
            $this->assertGreaterThan(0, $data['rewards']['gold']);
        }
    }

    /**
     * Test VIP multiplier application.
     */
    public function test_vip_multiplier_application(): void
    {
        // This would require setting up VIP membership
        // For now, we test that the battle completes
        $user = User::factory()->create([
            'level' => 5,
            'max_hp' => 200,
            'current_hp' => 200,
            'attack' => 30,
            'defense' => 15,
        ]);

        $mob = Mob::factory()->create([
            'level' => 5,
            'hp' => 50,
            'attack' => 5,
            'defense' => 5,
            'xp_reward' => 100,
            'gold_reward' => 50,
        ]);

        Sanctum::actingAs($user);

        $response = $this->postJson('/api/battle/start', [
            'mob_id' => $mob->id,
            'mode' => 'auto',
        ]);

        $response->assertStatus(200);
    }

    /**
     * Test battle history saving.
     */
    public function test_battle_history_saving(): void
    {
        $user = User::factory()->create([
            'level' => 5,
            'max_hp' => 200,
            'current_hp' => 200,
            'attack' => 30,
            'defense' => 15,
        ]);

        $mob = Mob::factory()->create([
            'level' => 5,
            'hp' => 50,
            'attack' => 5,
            'defense' => 5,
            'xp_reward' => 100,
            'gold_reward' => 50,
        ]);

        Sanctum::actingAs($user);

        $response = $this->postJson('/api/battle/start', [
            'mob_id' => $mob->id,
            'mode' => 'auto',
        ]);

        $response->assertStatus(200);

        // Check that battle history was saved
        $this->assertDatabaseHas('battle_histories', [
            'user_id' => $user->id,
            'mob_id' => $mob->id,
        ]);
    }
}

