<?php

namespace Tests\Unit\Services;

use Tests\TestCase;
use App\Services\LevelService;
use App\Models\User;
use App\Models\Level;
use Illuminate\Foundation\Testing\RefreshDatabase;

class LevelServiceTest extends TestCase
{
    use RefreshDatabase;

    protected LevelService $levelService;

    protected function setUp(): void
    {
        parent::setUp();
        $this->levelService = new LevelService();
    }

    /**
     * Test XP formula for various levels.
     */
    public function test_xp_for_next_level(): void
    {
        // Level 1: floor(100 * 1 * (1.12)^0) = 100
        $this->assertEquals(100, $this->levelService->xpForNextLevel(1));

        // Level 10: floor(100 * 10 * (1.12)^9) ≈ 3110
        $xp10 = $this->levelService->xpForNextLevel(10);
        $this->assertGreaterThan(3000, $xp10);
        $this->assertLessThan(3200, $xp10);

        // Level 50: floor(100 * 50 * (1.12)^49)
        $xp50 = $this->levelService->xpForNextLevel(50);
        $this->assertGreaterThan(0, $xp50);

        // Level 100: floor(100 * 100 * (1.12)^99)
        $xp100 = $this->levelService->xpForNextLevel(100);
        $this->assertGreaterThan(0, $xp100);
    }

    /**
     * Test adding XP to user triggers level up correctly.
     */
    public function test_add_xp_to_user_triggers_level_up(): void
    {
        // Seed levels first
        $this->seed(\Database\Seeders\LevelsTableSeeder::class);

        $user = User::factory()->create([
            'level' => 1,
            'xp' => 0,
            'max_hp' => 100,
            'current_hp' => 100,
            'attack' => 5,
            'defense' => 3,
            'stamina' => 100,
            'luck' => 1,
        ]);

        $oldLevel = $user->level;
        $result = $this->levelService->addXpToUser($user, 150); // Enough for level 2

        $user->refresh();
        
        $this->assertGreaterThan($oldLevel, $user->level);
        $this->assertNotNull($result['level_up']);
        $this->assertEquals($oldLevel, $result['level_up']['old_level']);
    }

    /**
     * Test XP accumulation without level up.
     */
    public function test_xp_accumulation_without_level_up(): void
    {
        $this->seed(\Database\Seeders\LevelsTableSeeder::class);

        $user = User::factory()->create([
            'level' => 1,
            'xp' => 0,
        ]);

        $oldLevel = $user->level;
        $result = $this->levelService->addXpToUser($user, 50); // Not enough for level 2

        $user->refresh();
        
        $this->assertEquals($oldLevel, $user->level);
        $this->assertNull($result['level_up']);
        $this->assertEquals(50, $user->xp);
    }

    /**
     * Test stat bonuses applied on level up.
     */
    public function test_stat_bonuses_applied_on_level_up(): void
    {
        $this->seed(\Database\Seeders\LevelsTableSeeder::class);

        $user = User::factory()->create([
            'level' => 1,
            'xp' => 0,
            'max_hp' => 100,
            'attack' => 5,
            'defense' => 3,
            'stamina' => 100,
        ]);

        $oldMaxHp = $user->max_hp;
        $oldAttack = $user->attack;
        $oldDefense = $user->defense;

        $this->levelService->addXpToUser($user, 200); // Enough for level 2

        $user->refresh();

        $this->assertGreaterThan($oldMaxHp, $user->max_hp);
        $this->assertGreaterThan($oldAttack, $user->attack);
        $this->assertGreaterThan($oldDefense, $user->defense);
    }
}
